{{-- Camera Barcode Scanner - JavaScript Auto-Injection --}}
<script src="{{ asset('js/cdn/html5-qrcode.min.js') }}"></script>
<script>
(function() {
    'use strict';
    
    // Wait for DOM to be ready
    $(document).ready(function() {
        // Initialize camera scanner after a short delay to ensure all elements are loaded
        setTimeout(initCameraScanner, 500);
        
        // Re-check periodically for dynamically added search inputs (for AJAX-loaded content)
        setInterval(checkAndInjectButtons, 2000);
    });
    
    // Configuration for different pages/routes
    const pageConfigs = {
        // POS Create page
        'pos': {
            selectors: ['#search_product', '#search_product_barcode_type_check'],
            buttonPosition: 'before' // Insert before quick add button
        },
        // Purchase Create/Edit page
        'purchase': {
            selectors: ['#search_product'],
            buttonPosition: 'end' // Insert at end of input-group-btn
        },
        // Sell Create page
        'sell': {
            selectors: ['#search_product'],
            buttonPosition: 'end'
        },
        // Stock Transfer Create page
        'stock_transfer': {
            selectors: ['#search_product_for_srock_adjustment'],
            buttonPosition: 'end'
        },
        // Stock Adjustment Create page
        'stock_adjustment': {
            selectors: ['#search_product_for_srock_adjustment'],
            buttonPosition: 'end'
        }
    };
    
    function detectCurrentPage() {
        const path = window.location.pathname.toLowerCase();
        
        if (path.includes('/sale-pos/create') || path.includes('/sale_pos/create') || path.includes('/pos/create')) {
            return 'pos';
        } else if (path.includes('/purchases/create') || path.includes('/purchases/edit')) {
            return 'purchase';
        } else if (path.includes('/sells/create') || path.includes('/sells/edit')) {
            return 'sell';
        } else if (path.includes('/stock-transfers/create')) {
            return 'stock_transfer';
        } else if (path.includes('/stock-adjustments/create')) {
            return 'stock_adjustment';
        }
        
        // Default: try to detect by common selectors
        if ($('#search_product').length && $('#search_product_barcode_type_check').length) {
            return 'pos';
        } else if ($('#search_product').length) {
            return 'purchase'; // Default to purchase pattern
        }
        
        return null;
    }
    
    function initCameraScanner() {
        // Inject modal first (only once)
        if ($('#camera_scanner_modal').length === 0) {
            injectCameraModal();
        }
        
        // Detect current page
        const currentPage = detectCurrentPage();
        
        if (currentPage && pageConfigs[currentPage]) {
            const config = pageConfigs[currentPage];
            
            // Inject buttons for all configured selectors
            config.selectors.forEach(function(selector) {
                if ($(selector).length) {
                    injectCameraButton(selector, config.buttonPosition);
                }
            });
        } else {
            // Fallback: try to find any search_product input
            const commonSelectors = [
                '#search_product',
                '#search_product_barcode_type_check',
                '#search_product_for_srock_adjustment',
                'input[id*="search_product"]',
                'input[placeholder*="Product name"]',
                'input[placeholder*="SKU"]',
                'input[placeholder*="bar code"]'
            ];
            
            commonSelectors.forEach(function(selector) {
                $(selector).each(function() {
                    if (!$(this).closest('.input-group').find('.camera_scanner_btn').length) {
                        injectCameraButton('#' + $(this).attr('id'), 'end');
                    }
                });
            });
        }
        
        // Initialize scanner functionality
        initializeScannerLogic();
        
        console.log('Camera Barcode Scanner module initialized for page:', currentPage || 'auto-detected');
    }
    
    function checkAndInjectButtons() {
        // Check if new search inputs appeared and inject buttons if needed
        const currentPage = detectCurrentPage();
        if (currentPage && pageConfigs[currentPage]) {
            const config = pageConfigs[currentPage];
            config.selectors.forEach(function(selector) {
                if ($(selector).length && !$(selector).closest('.input-group').find('.camera_scanner_btn').length) {
                    injectCameraButton(selector, config.buttonPosition);
                }
            });
        }
    }
    
    function injectCameraButton(targetInputSelector, buttonPosition) {
        // Find the input group containing the search field
        const $inputGroup = $(targetInputSelector).closest('.input-group');
        
        // Check if button already exists in this input group
        if ($inputGroup.find('.camera_scanner_btn').length > 0) {
            return;
        }
        
        let $inputGroupBtn = $inputGroup.find('.input-group-btn');
        
        // Create input-group-btn if it doesn't exist
        if ($inputGroupBtn.length === 0) {
            $inputGroupBtn = $('<span>').addClass('input-group-btn');
            $(targetInputSelector).after($inputGroupBtn);
        }
        
        // Create camera button
        const $cameraBtn = $('<button>')
            .attr({
                type: 'button',
                class: 'btn btn-default bg-white btn-flat camera_scanner_btn',
                'data-toggle': 'modal',
                'data-target': '#camera_scanner_modal',
                title: 'Camera Scanner'
            })
            .html('<i class="fa fa-camera text-primary fa-lg"></i>');
        
        // Insert button based on position preference
        if (buttonPosition === 'before') {
            // Try to insert before quick add button (POS page)
            const $quickAddBtn = $inputGroupBtn.find('.pos_add_quick_product');
            if ($quickAddBtn.length) {
                $cameraBtn.insertBefore($quickAddBtn);
            } else {
                $inputGroupBtn.prepend($cameraBtn);
            }
        } else {
            // Insert at the end (default for most pages)
            $inputGroupBtn.append($cameraBtn);
        }
    }
    
    
    function injectCameraModal() {
        // Create modal HTML
        const modalHTML = `
<div class="modal fade" id="camera_scanner_modal" tabindex="-1" role="dialog" aria-labelledby="camera_scanner_modal_label">
    <div class="modal-dialog modal-md" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title" id="camera_scanner_modal_label">
                    <i class="fa fa-camera"></i> Camera Scanner
                </h4>
            </div>
            <div class="modal-body">
                <div id="camera_scanner_container" style="width: 100%; text-align: center;">
                    <div id="camera_scanner_camera" style="width: 100%; display: inline-block;"></div>
                    <div id="camera_scanner_result" style="margin-top: 15px; display: none;">
                        <div class="alert alert-success">
                            <strong>Barcode Scanned</strong>: 
                            <span id="scanned_barcode"></span>
                        </div>
                    </div>
                </div>
                <div id="camera_scanner_status" style="margin-top: 10px; text-align: center;"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" id="stop_camera_btn" style="display: none;">
                    <i class="fa fa-stop"></i> Stop Camera
                </button>
                <button type="button" class="btn btn-default" id="start_camera_btn">
                    <i class="fa fa-play"></i> Start Camera
                </button>
                <button type="button" class="btn btn-default" data-dismiss="modal">
                    Close
                </button>
            </div>
        </div>
    </div>
</div>
        `;
        
        // Append modal to body
        $('body').append(modalHTML);
    }
    
    function initializeScannerLogic() {
        let html5QrcodeScanner = null;
        let isScanning = false;
        let targetInputSelector = '#search_product'; // Default target input
        
        // Initialize when modal is shown
        $(document).on('shown.bs.modal', '#camera_scanner_modal', function(e) {
            // Determine which input field to use based on which button was clicked
            let $clickedBtn = null;
            
            // Find which button triggered the modal
            if (e.relatedTarget) {
                $clickedBtn = $(e.relatedTarget);
            } else {
                // Fallback: find the last clicked camera button
                $clickedBtn = $('.camera_scanner_btn').last();
            }
            
            if ($clickedBtn.length) {
                // Find the nearest search input in the same input-group
                const $searchInput = $clickedBtn.closest('.input-group').find('input[type="text"], input[type="search"]').first();
                if ($searchInput.length) {
                    targetInputSelector = '#' + $searchInput.attr('id');
                } else {
                    // Fallback: detect by page
                    const currentPage = detectCurrentPage();
                    if (currentPage === 'pos' && $('#search_product_barcode_type_check').is(':visible')) {
                        targetInputSelector = '#search_product_barcode_type_check';
                    } else {
                        targetInputSelector = '#search_product';
                    }
                }
            }
        });
        
        // Clean up when modal is hidden
        $(document).on('hidden.bs.modal', '#camera_scanner_modal', function() {
            stopScanner();
            $('#camera_scanner_result').hide();
            $('#scanned_barcode').text('');
        });
        
        // Start camera button
        $(document).on('click', '#start_camera_btn', function() {
            startScanner();
        });
        
        // Stop camera button
        $(document).on('click', '#stop_camera_btn', function() {
            stopScanner();
        });
        
        function startScanner() {
            if (isScanning) {
                return;
            }
            
            // Check if browser supports camera
            if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
                showError('Camera is not supported in this browser. Please use a modern browser with camera access.');
                return;
            }
            
            try {
                // Create scanner instance
                html5QrcodeScanner = new Html5Qrcode("camera_scanner_camera");
                
                const config = {
                    fps: 10,
                    qrbox: { width: 250, height: 250 },
                    aspectRatio: 1.0,
                    supportedScanTypes: [
                        Html5QrcodeScanType.SCAN_TYPE_CAMERA
                    ],
                    formatsToSupport: [
                        Html5QrcodeSupportedFormats.EAN_13,
                        Html5QrcodeSupportedFormats.EAN_8,
                        Html5QrcodeSupportedFormats.UPC_A,
                        Html5QrcodeSupportedFormats.UPC_E,
                        Html5QrcodeSupportedFormats.CODE_128,
                        Html5QrcodeSupportedFormats.CODE_39,
                        Html5QrcodeSupportedFormats.CODE_93,
                        Html5QrcodeSupportedFormats.CODABAR,
                        Html5QrcodeSupportedFormats.ITF,
                        Html5QrcodeSupportedFormats.QR_CODE,
                        Html5QrcodeSupportedFormats.AZTEC,
                        Html5QrcodeSupportedFormats.DATA_MATRIX
                    ]
                };
                
                // Start scanning
                html5QrcodeScanner.start(
                    { facingMode: "environment" }, // Use back camera by default
                    config,
                    onScanSuccess,
                    onScanError
                ).then(() => {
                    isScanning = true;
                    $('#start_camera_btn').hide();
                    $('#stop_camera_btn').show();
                    $('#camera_scanner_status').html('<div class="alert alert-info"><i class="fa fa-info-circle"></i> Scanning... Please point the camera at a barcode</div>');
                }).catch((err) => {
                    console.error('Failed to start scanner:', err);
                    let errorMsg = 'Failed to start camera. ';
                    if (err.message) {
                        errorMsg += err.message;
                    } else {
                        errorMsg += 'Please check camera permissions.';
                    }
                    showError(errorMsg);
                });
                
            } catch (error) {
                console.error('Scanner initialization error:', error);
                showError('Scanner initialization error: ' + error.message);
            }
        }
        
        function stopScanner() {
            if (html5QrcodeScanner && isScanning) {
                html5QrcodeScanner.stop().then(() => {
                    isScanning = false;
                    $('#start_camera_btn').show();
                    $('#stop_camera_btn').hide();
                    $('#camera_scanner_status').html('');
                    html5QrcodeScanner.clear();
                    html5QrcodeScanner = null;
                }).catch((err) => {
                    console.error('Failed to stop scanner:', err);
                    isScanning = false;
                    html5QrcodeScanner = null;
                });
            }
        }
        
        function onScanSuccess(decodedText, decodedResult) {
            // Hide scanning status
            $('#camera_scanner_status').html('');
            
            // Show success message
            $('#scanned_barcode').text(decodedText);
            $('#camera_scanner_result').show();
            
            console.log('Barcode scanned:', decodedText);
            
            // Auto-fill the search input field
            const $targetInput = $(targetInputSelector);
            if ($targetInput.length && !$targetInput.is(':disabled')) {
                // Set the value
                $targetInput.val(decodedText).focus();
                
                // Trigger events to activate search
                $targetInput.trigger('input');
                $targetInput.trigger('change');
                
                // Try to trigger product search if functions exist (POS page)
                if (typeof window.pos_search_product === 'function') {
                    try {
                        window.pos_search_product(decodedText);
                    } catch(e) {
                        console.log('pos_search_product error:', e);
                    }
                }
                
                // Try purchase search function
                if (typeof window.search_product === 'function') {
                    try {
                        window.search_product(decodedText);
                    } catch(e) {
                        console.log('search_product error:', e);
                    }
                }
                
                // Trigger keydown Enter to simulate user input (common pattern)
                setTimeout(function() {
                    // Simulate Enter key press
                    const e = $.Event('keydown', { keyCode: 13, which: 13, key: 'Enter' });
                    $targetInput.trigger(e);
                    
                    // Also trigger keyup Enter
                    const e2 = $.Event('keyup', { keyCode: 13, which: 13, key: 'Enter' });
                    $targetInput.trigger(e2);
                }, 150);
            } else if ($targetInput.length && $targetInput.is(':disabled')) {
                // Input is disabled, try to find alternative
                const altSelectors = ['#search_product', 'input[id*="search_product"]'];
                for (let i = 0; i < altSelectors.length; i++) {
                    const $alt = $(altSelectors[i]);
                    if ($alt.length && !$alt.is(':disabled')) {
                        $alt.val(decodedText).focus().trigger('input').trigger('change');
                        break;
                    }
                }
            }
            
            // Stop scanner after successful scan
            setTimeout(() => {
                stopScanner();
                $('#camera_scanner_modal').modal('hide');
            }, 1500);
        }
        
        function onScanError(errorMessage) {
            // Suppress too frequent error messages
            // Only log significant errors
            if (!errorMessage.includes('No MultiFormat Readers') && 
                !errorMessage.includes('NotFoundException') &&
                !errorMessage.includes('No QR code') &&
                !errorMessage.includes('No MultiFormat')) {
                // Log significant errors only
                console.log('Scan error:', errorMessage);
            }
        }
        
        function showError(message) {
            $('#camera_scanner_status').html(
                '<div class="alert alert-danger"><i class="fa fa-exclamation-triangle"></i> ' + message + '</div>'
            );
        }
        
        // Expose function to change target input
        window.setCameraScannerTarget = function(selector) {
            targetInputSelector = selector;
        };
    }
})();
</script>

<style>
/* Camera Scanner Button Styles */
.camera_scanner_btn {
    margin-left: 2px;
}

.camera_scanner_btn:hover {
    background-color: #f0f0f0 !important;
}

#camera_scanner_camera {
    min-height: 300px;
    background-color: #000;
    border-radius: 4px;
}

#camera_scanner_camera video {
    width: 100%;
    max-width: 100%;
    border-radius: 4px;
}

#camera_scanner_camera canvas {
    display: none;
}
</style>
