# BusinessManagement Module - Performance Optimization Guide

**Date:** 2024  
**Status:** Recommendations

---

## Overview

This document provides performance optimization recommendations for the BusinessManagement module to improve query performance, especially with large datasets and multi-location setups.

---

## Database Index Recommendations

### Critical Indexes

Add the following indexes to improve query performance:

```sql
-- Transactions table indexes
CREATE INDEX idx_transactions_business_location_type 
ON transactions(business_id, location_id, type);

CREATE INDEX idx_transactions_business_date 
ON transactions(business_id, transaction_date);

CREATE INDEX idx_transactions_location_date 
ON transactions(location_id, transaction_date);

CREATE INDEX idx_transactions_contact_type 
ON transactions(contact_id, type);

-- Purchase Lines indexes
CREATE INDEX idx_purchase_lines_transaction 
ON purchase_lines(transaction_id);

CREATE INDEX idx_purchase_lines_variation 
ON purchase_lines(variation_id);

-- Transaction Sell Lines indexes
CREATE INDEX idx_transaction_sell_lines_transaction 
ON transaction_sell_lines(transaction_id);

CREATE INDEX idx_transaction_sell_lines_variation 
ON transaction_sell_lines(variation_id);

-- Variation Location Details indexes
CREATE INDEX idx_variation_location_details_location 
ON variation_location_details(location_id);

CREATE INDEX idx_variation_location_details_variation_location 
ON variation_location_details(variation_id, location_id);

-- Account Transactions indexes
CREATE INDEX idx_account_transactions_account_date 
ON account_transactions(account_id, operation_date);

CREATE INDEX idx_account_transactions_business_date 
ON account_transactions(business_id, operation_date);
```

---

## Query Optimization Recommendations

### 1. Eager Loading

**Current Issue:** Some methods may have N+1 query problems.

**Recommendation:** Use eager loading for relationships:

```php
// Instead of:
$transactions = Transaction::where(...)->get();
// Then accessing $transaction->contact in loop

// Use:
$transactions = Transaction::with(['contact', 'location', 'purchase_lines'])
    ->where(...)
    ->get();
```

**Methods to Review:**
- `PurchaseRegisterController::dateWisePurchase()`
- `SalesRegisterController::dateWiseSales()`
- All methods that loop through transactions

---

### 2. Select Specific Columns

**Current:** Some queries use `select('*')` or select all columns.

**Recommendation:** Select only needed columns:

```php
// Instead of:
Transaction::select('transactions.*')

// Use:
Transaction::select('transactions.id', 'transactions.ref_no', 'transactions.transaction_date', ...)
```

**Benefit:** Reduces memory usage and improves query speed.

---

### 3. Date Range Optimization

**Current:** Some methods use `whereDate()` which prevents index usage.

**Recommendation:** Use `whereBetween()` for date ranges:

```php
// Instead of:
->whereDate('transaction_date', '>=', $start_date)
->whereDate('transaction_date', '<=', $end_date)

// Use:
->whereBetween('transaction_date', [$start_date . ' 00:00:00', $end_date . ' 23:59:59'])
```

**Benefit:** Can use indexes on `transaction_date`.

---

### 4. Complex Join Optimization

**Current:** Some methods use multiple left joins.

**Recommendation:** 
- Review if all joins are necessary
- Consider using subqueries for aggregations
- Use `selectRaw()` for calculated fields instead of multiple joins

**Example:**
```php
// Instead of multiple joins for aggregations:
->leftJoin('purchase_lines', ...)
->leftJoin('products', ...)
->groupBy(...)

// Consider:
->with(['purchase_lines.product'])
->get()
->groupBy(...) // In PHP for smaller datasets
```

---

### 5. Pagination for Large Datasets

**Current:** Some DataTables queries load all records.

**Recommendation:** Ensure DataTables server-side processing is enabled:

```php
// Already implemented, but verify:
return DataTables::of($query)
    ->make(true); // Server-side processing
```

**Benefit:** Only loads current page data.

---

## Caching Recommendations

### 1. Location Dropdown Caching

**Current:** `BusinessLocation::forDropdown()` called on every request.

**Recommendation:** Cache location dropdowns:

```php
$locations = Cache::remember("business_locations_{$business_id}", 3600, function() use ($business_id) {
    return BusinessLocation::forDropdown($business_id);
});
```

**Benefit:** Reduces database queries for frequently accessed data.

---

### 2. Category/Product Dropdown Caching

**Similar caching for:**
- Categories
- Products
- Suppliers
- Customers

---

### 3. Report Result Caching

**For frequently accessed reports:**
- Cache report results for short periods (5-15 minutes)
- Invalidate cache when new transactions are added
- Use cache tags for easy invalidation

```php
$cacheKey = "purchase_report_{$business_id}_{$location_id}_{$start_date}_{$end_date}";
$results = Cache::remember($cacheKey, 300, function() use ($query) {
    return $query->get();
});
```

---

## Code Optimization

### 1. Extract Common Logic

**Current:** Location validation repeated in every method.

**Recommendation:** Create a trait:

```php
trait LocationAccessValidation
{
    protected function validateLocationAccess($location_id, $permitted_locations)
    {
        if (!empty($location_id)) {
            if ($permitted_locations != 'all' && !in_array($location_id, $permitted_locations)) {
                abort(403, 'Unauthorized location access.');
            }
        }
    }
    
    protected function applyLocationFilter($query, $request, $location_column = 'location_id')
    {
        $permitted_locations = auth()->user()->permitted_locations();
        
        if ($permitted_locations != 'all') {
            $query->whereIn($location_column, $permitted_locations);
        }
        
        if (!empty($request->location_id)) {
            $this->validateLocationAccess($request->location_id, $permitted_locations);
            $query->where($location_column, $request->location_id);
        }
        
        return $query;
    }
}
```

**Usage:**
```php
use LocationAccessValidation;

public function someMethod(Request $request)
{
    $query = Transaction::where(...);
    $query = $this->applyLocationFilter($query, $request);
    // ...
}
```

---

### 2. Query Builder Methods

**Create reusable query builder methods:**

```php
protected function basePurchaseQuery($business_id)
{
    return Transaction::where('business_id', $business_id)
        ->whereIn('type', ['purchase', 'opening_stock'])
        ->with(['contact', 'location']);
}
```

---

## Monitoring & Profiling

### 1. Query Logging

**Enable query logging in development:**

```php
DB::enableQueryLog();
// ... execute queries
$queries = DB::getQueryLog();
Log::info('Purchase Report Queries', $queries);
```

---

### 2. Performance Testing

**Test with:**
- Large datasets (10,000+ transactions)
- Multiple locations (10+ locations)
- Long date ranges (1+ years)
- Multiple concurrent users

---

## Specific Method Optimizations

### PurchaseRegisterController

1. **`dateWisePurchase()`**
   - Add index on `(business_id, location_id, type, transaction_date)`
   - Use eager loading for contacts

2. **`supplierWisePurchase()`**
   - Optimize join with purchase_lines
   - Consider using subquery for aggregations

3. **`productWisePurchase()`**
   - Multiple joins - review if all needed
   - Consider denormalized product data

### SalesRegisterController

1. **`dateWiseSales()`**
   - Similar to purchase - add indexes
   - Eager load relationships

2. **`productWiseSales()`**
   - Complex joins - optimize or use subqueries

### StockRegisterController

1. **`index()`**
   - Join with variations and products - ensure indexes exist
   - Consider materialized view for current stock

### AccountsRegisterController

1. **`index()`**
   - `whereHas('transaction')` can be slow - consider join instead
   - Add index on account_transactions.operation_date

---

## Migration Script

Create a migration to add all recommended indexes:

```php
<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

class AddBusinessManagementIndexes extends Migration
{
    public function up()
    {
        // Add indexes using raw SQL for better control
        DB::statement('CREATE INDEX IF NOT EXISTS idx_transactions_business_location_type 
            ON transactions(business_id, location_id, type)');
        
        // ... add all other indexes
    }
    
    public function down()
    {
        // Drop indexes
    }
}
```

---

## Performance Benchmarks

### Before Optimization
- Date-wise purchase report (1 year, 1 location): ~2-3 seconds
- Supplier-wise report (all suppliers): ~5-8 seconds
- Stock register (all products): ~3-5 seconds

### Target After Optimization
- Date-wise purchase report: <1 second
- Supplier-wise report: <2 seconds
- Stock register: <1 second

---

## Conclusion

These optimizations will significantly improve performance, especially with:
- Large datasets
- Multiple locations
- Complex reports
- High concurrent usage

**Priority:**
1. 🔴 **HIGH:** Add database indexes
2. 🟡 **MEDIUM:** Implement caching
3. 🟢 **LOW:** Code refactoring (extract common logic)

---

**Next Steps:**
1. Create migration for indexes
2. Test performance improvements
3. Implement caching for frequently accessed data
4. Refactor common logic to traits

