<?php

namespace Modules\BusinessManagement\Http\Controllers;

use App\System;
use Composer\Semver\Comparator;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;

class InstallController extends Controller
{
    public function __construct()
    {
        $this->module_name = 'businessmanagement';
        $this->appVersion = '1.0.1';
        $this->module_display_name = 'Business Management';
    }

    /**
     * Install
     *
     * @return Response
     */
    public function index()
    {
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        ini_set('max_execution_time', 0);
        ini_set('memory_limit', '512M');

        $this->installSettings();

        //Check if businessmanagement installed or not.
        $is_installed = System::getProperty($this->module_name.'_version');
        if (! empty($is_installed)) {
            abort(404);
        }

        $action_url = action([\Modules\BusinessManagement\Http\Controllers\InstallController::class, 'install']);
        $intruction_type = 'uf';
        $action_type = 'install';
        $module_display_name = $this->module_display_name;

        return view('install.install-module')
            ->with(compact('action_url', 'intruction_type', 'action_type', 'module_display_name'));
    }

    /**
     * Initialize all install functions
     */
    private function installSettings()
    {
        config(['app.debug' => true]);
        Artisan::call('config:clear');
    }

    /**
     * Installing Business Management Module
     */
    public function install()
    {
        try {
            DB::beginTransaction();

            // Validate license inputs - MANDATORY
            request()->validate(
                [
                    'license_code' => 'required|string|min:5',
                    'login_username' => 'required|string|min:3'
                ],
                [
                    'license_code.required' => 'License code is required',
                    'license_code.min' => 'License code must be at least 5 characters',
                    'login_username.required' => 'Username is required',
                    'login_username.min' => 'Username must be at least 3 characters'
                ]
            );

            $license_code = trim(request()->license_code);
            $email = request()->ENVATO_EMAIL ?? '';
            $login_username = trim(request()->login_username);
            $pid = config('businessmanagement.pid');

            // Check license status in database first - only allow 'active' or 'inactive' licenses
            if (\Illuminate\Support\Facades\Schema::hasTable('module_licenses')) {
                $license_record = DB::table('module_licenses')
                    ->where('module_name', $this->module_name)
                    ->where('license_code', $license_code)
                    ->where('username', $login_username)
                    ->first();

                if ($license_record) {
                    // Check if license is expired or revoked
                    if ($license_record->status === 'expired') {
                        DB::rollBack();
                        return redirect()->back()
                            ->with('error', 'License has expired. Please renew your license.');
                    }
                    
                    if ($license_record->status === 'revoked') {
                        DB::rollBack();
                        return redirect()->back()
                            ->with('error', 'License has been revoked. Please contact support.');
                    }

                    // Check expiry date
                    if ($license_record->expiry_date && strtotime($license_record->expiry_date) < time()) {
                        // Update status to expired
                        DB::table('module_licenses')
                            ->where('id', $license_record->id)
                            ->update(['status' => 'expired', 'updated_at' => now()]);
                        
                        DB::rollBack();
                        return redirect()->back()
                            ->with('error', 'License has expired. Please renew your license.');
                    }
                }
            }

            // MANDATORY: Validate license with license server BEFORE installation
            // Installation will NOT proceed without valid license
            $response = businessmanagement_license(
                url('/'), 
                __DIR__, 
                $license_code, 
                $email, 
                $login_username, 
                $type = 1, 
                $pid
            );

            // If license validation returns a redirect response, it means license is invalid
            // STOP installation and return error
            if (!empty($response)) {
                DB::rollBack();
                \Log::warning('BusinessManagement module installation blocked - Invalid license', [
                    'license_code' => substr($license_code, 0, 5) . '***',
                    'username' => $login_username
                ]);
                return $response;
            }

            // Check if already installed (double check)
            $is_installed = System::getProperty($this->module_name.'_version');
            if (! empty($is_installed)) {
                DB::rollBack();
                abort(404, 'Module is already installed.');
            }

            // License is valid - Store license information
            System::addProperty($this->module_name.'_license_code', $license_code);
            System::addProperty($this->module_name.'_license_username', $login_username);
            if (!empty($email)) {
                System::addProperty($this->module_name.'_license_email', $email);
            }
            System::addProperty($this->module_name.'_license_validated_at', now()->toDateTimeString());

            // Proceed with installation only after license validation
            DB::statement('SET default_storage_engine=INNODB;');
            Artisan::call('module:migrate', ['module' => 'BusinessManagement', '--force' => true]);
            Artisan::call('module:publish', ['module' => 'BusinessManagement']);
            System::addProperty($this->module_name.'_version', $this->appVersion);

            // Update license status to 'active' when module is installed
            $this->updateLicenseStatus($license_code, $login_username, 'active');

            DB::commit();

            \Log::info('BusinessManagement module installed successfully', [
                'version' => $this->appVersion,
                'username' => $login_username
            ]);

            $output = ['success' => 1,
                'msg' => 'Business Management module installed successfully',
            ];
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('BusinessManagement installation error - File:'.$e->getFile().' Line:'.$e->getLine().' Message:'.$e->getMessage());

            $output = [
                'success' => false,
                'msg' => 'Installation failed: ' . $e->getMessage(),
            ];
        }

        return redirect()
            ->action([\App\Http\Controllers\Install\ModulesController::class, 'index'])
            ->with('status', $output);
    }

    /**
     * Uninstall
     *
     * @return Response
     */
    public function uninstall()
    {
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            // Get license information before removing
            $license_code = System::getProperty($this->module_name.'_license_code');
            $license_username = System::getProperty($this->module_name.'_license_username');

            System::removeProperty($this->module_name.'_version');
            // Remove license information on uninstall
            System::removeProperty($this->module_name.'_license_code');
            System::removeProperty($this->module_name.'_license_username');
            System::removeProperty($this->module_name.'_license_email');

            // Update license status to 'inactive' when module is uninstalled
            if (!empty($license_code) && !empty($license_username)) {
                $this->updateLicenseStatus($license_code, $license_username, 'inactive');
            }

            $output = ['success' => true,
                'msg' => __('lang_v1.success'),
            ];
        } catch (\Exception $e) {
            $output = ['success' => false,
                'msg' => $e->getMessage(),
            ];
        }

        return redirect()->back()->with(['status' => $output]);
    }

    /**
     * Update license status in database
     *
     * @param string $license_code
     * @param string $username
     * @param string $status ('active' or 'inactive')
     * @return void
     */
    private function updateLicenseStatus($license_code, $username, $status)
    {
        try {
            if (!\Illuminate\Support\Facades\Schema::hasTable('module_licenses')) {
                return; // Table doesn't exist, skip
            }

            // Update license status based on module installation state
            DB::table('module_licenses')
                ->where('module_name', $this->module_name)
                ->where('license_code', $license_code)
                ->where('username', $username)
                ->update([
                    'status' => $status,
                    'updated_at' => now()
                ]);

            \Log::info("BusinessManagement: License status updated to '{$status}'", [
                'license_code' => substr($license_code, 0, 5) . '***',
                'username' => $username
            ]);
        } catch (\Exception $e) {
            \Log::warning('BusinessManagement: Failed to update license status', [
                'error' => $e->getMessage(),
                'license_code' => substr($license_code, 0, 5) . '***'
            ]);
        }
    }

    /**
     * Update module
     *
     * @return Response
     */
    public function update()
    {
        //Check if businessmanagement_version is same as appVersion then 404
        //If appVersion > businessmanagement_version - run update script.
        //Else there is some problem.
        if (! auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();
            ini_set('max_execution_time', 0);
            ini_set('memory_limit', '512M');

            $businessmanagement_version = System::getProperty($this->module_name.'_version');

            if (Comparator::greaterThan($this->appVersion, $businessmanagement_version)) {
                ini_set('max_execution_time', 0);
                ini_set('memory_limit', '512M');
                $this->installSettings();

                DB::statement('SET default_storage_engine=INNODB;');
                Artisan::call('module:migrate', ['module' => 'BusinessManagement', '--force' => true]);
                Artisan::call('module:publish', ['module' => 'BusinessManagement']);
                System::addProperty($this->module_name.'_version', $this->appVersion);
            } else {
                abort(404);
            }

            DB::commit();

            $output = ['success' => 1,
                'msg' => 'Business Management module updated successfully to version '.$this->appVersion.' !!',
            ];

            return redirect()
                ->action([\App\Http\Controllers\Install\ModulesController::class, 'index'])
                ->with('status', $output);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = [
                'success' => false,
                'msg' => $e->getMessage(),
            ];

            return redirect()->back()->with('status', $output);
        }
    }
}

