<?php

namespace Modules\BusinessManagement\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Schema\Blueprint;
use PDOException;

class LicenseValidationController extends Controller
{
    /**
     * API endpoint info (GET request for testing/documentation)
     * 
     * @return \Illuminate\Http\JsonResponse
     */
    public function info()
    {
        return response()->json([
            'endpoint' => '/api/license/validate',
            'method' => 'POST',
            'description' => 'Public license validation API (no authentication required)',
            'required_parameters' => [
                'license_code' => 'string (required)',
                'username' => 'string (required)',
            ],
            'optional_parameters' => [
                'product_id' => 'integer (default: 25)',
                'email' => 'string',
                'url' => 'string',
                'path' => 'string',
            ],
            'response_format' => [
                'valid' => [
                    'flag' => 'valid',
                    'msg' => 'License is valid',
                    'data' => 'object'
                ],
                'invalid' => [
                    'flag' => 'invalid',
                    'msg' => 'Error message'
                ]
            ]
        ]);
    }

    /**
     * Validate license without requiring authentication
     * Public API endpoint for license validation during module installation
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function validate(Request $request)
    {
        // Check if PDO extension is available
        if (!extension_loaded('pdo')) {
            \Log::error('LicenseValidation: PDO extension not loaded');
            return response()->json([
                'flag' => 'invalid',
                'msg' => 'Server configuration error: PDO extension is not installed. Please contact your server administrator to enable PDO extension.'
            ], 200);
        }

        if (!extension_loaded('pdo_mysql')) {
            \Log::error('LicenseValidation: PDO MySQL extension not loaded');
            return response()->json([
                'flag' => 'invalid',
                'msg' => 'Server configuration error: PDO MySQL extension is not installed. Please contact your server administrator to enable PDO MySQL extension.'
            ], 200);
        }

        try {
            // Validate input
            $validator = Validator::make($request->all(), [
                'license_code' => 'required|string',
                'username' => 'required|string',
                'product_id' => 'nullable|integer',
                'email' => 'nullable|email',
                'url' => 'nullable|string',
                'path' => 'nullable|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'flag' => 'invalid',
                    'msg' => 'Invalid request parameters: ' . $validator->errors()->first()
                ], 422);
            }

            $license_code = trim($request->license_code);
            $username = trim($request->username);
            $product_id = $request->product_id ?? 25; // Default to BusinessManagement product ID

            // Map product_id to module_name (you can extend this)
            $module_name = $this->getModuleNameFromProductId($product_id);

            // Check database connection first
            try {
                DB::connection()->getPdo();
            } catch (\Exception $e) {
                \Log::error('LicenseValidation: Database connection failed', [
                    'error' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]);
                return response()->json([
                    'flag' => 'invalid',
                    'msg' => 'Database connection error. Please check your database configuration.'
                ], 200); // Return 200 to avoid triggering HTTP 500 in helper
            }

            // Check if module_licenses table exists
            try {
                if (!Schema::hasTable('module_licenses')) {
                    \Log::warning('LicenseValidation: module_licenses table does not exist');
                    
                    // Try to create the table automatically (if possible)
                    try {
                        $this->createModuleLicensesTable();
                        \Log::info('LicenseValidation: module_licenses table created automatically');
                    } catch (\Exception $createError) {
                        \Log::error('LicenseValidation: Failed to auto-create table', [
                            'error' => $createError->getMessage()
                        ]);
                        
                        return response()->json([
                            'flag' => 'invalid',
                            'msg' => 'License validation system not initialized. Please run: php artisan migrate'
                        ], 200); // Return 200 with invalid flag instead of 500
                    }
                }
            } catch (\Exception $e) {
                \Log::error('LicenseValidation: Error checking table existence', [
                    'error' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]);
                return response()->json([
                    'flag' => 'invalid',
                    'msg' => 'Unable to access license database. Please run: php artisan migrate'
                ], 200); // Return 200 to avoid triggering HTTP 500 in helper
            }

            // Query license from database
            try {
                $license = DB::table('module_licenses')
                    ->where('module_name', $module_name)
                    ->where('license_code', $license_code)
                    ->where('username', $username)
                    ->first();
            } catch (\Exception $e) {
                \Log::error('LicenseValidation: Database query error', [
                    'error' => $e->getMessage(),
                    'module' => $module_name,
                    'license_code' => substr($license_code, 0, 5) . '***',
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]);
                return response()->json([
                    'flag' => 'invalid',
                    'msg' => 'Database query error. Please check your database connection.'
                ], 200); // Return 200 to avoid triggering HTTP 500 in helper
            }

            if (!$license) {
                \Log::info('LicenseValidation: License not found', [
                    'license_code' => substr($license_code, 0, 5) . '***',
                    'username' => $username,
                    'module' => $module_name
                ]);

                // License not found - this is OK for installation (will be created)
                // Return invalid but with a helpful message
                return response()->json([
                    'flag' => 'invalid',
                    'msg' => 'License not found. Please generate a license via License Management or contact support.'
                ]);
            }

            // Check license status
            if ($license->status === 'expired') {
                return response()->json([
                    'flag' => 'invalid',
                    'msg' => 'License has expired. Please renew your license.'
                ]);
            }

            if ($license->status === 'revoked') {
                return response()->json([
                    'flag' => 'invalid',
                    'msg' => 'License has been revoked. Please contact support.'
                ]);
            }

            if ($license->status === 'inactive') {
                // Inactive licenses are still valid for installation (will become active after install)
                // But we should allow them
            }

            // Check expiry date
            if (!empty($license->expiry_date) && 
                $license->expiry_date !== '0000-00-00' && 
                $license->expiry_date !== null &&
                strtolower($license->expiry_date) !== 'lifetime') {
                
                try {
                    $expiry_timestamp = strtotime($license->expiry_date);
                    if ($expiry_timestamp !== false && $expiry_timestamp < time()) {
                        // Update status to expired
                        try {
                            DB::table('module_licenses')
                                ->where('id', $license->id)
                                ->update(['status' => 'expired', 'updated_at' => now()]);
                        } catch (\Exception $e) {
                            \Log::warning('LicenseValidation: Failed to update expired status', [
                                'error' => $e->getMessage()
                            ]);
                        }

                        return response()->json([
                            'flag' => 'invalid',
                            'msg' => 'License has expired. Please renew your license.'
                        ]);
                    }
                } catch (\Exception $e) {
                    \Log::warning('LicenseValidation: Error parsing expiry date', [
                        'expiry_date' => $license->expiry_date,
                        'error' => $e->getMessage()
                    ]);
                    // Continue validation if date parsing fails
                }
            }

            // License is valid
            \Log::info('LicenseValidation: License validated successfully', [
                'license_code' => substr($license_code, 0, 5) . '***',
                'username' => $username,
                'module' => $module_name,
                'status' => $license->status
            ]);

            return response()->json([
                'flag' => 'valid',
                'msg' => 'License is valid',
                'data' => [
                    'module_name' => $license->module_name,
                    'license_type' => $license->license_type,
                    'expiry_date' => $license->expiry_date,
                    'status' => $license->status
                ]
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            // Re-throw validation exceptions to let Laravel handle them
            throw $e;
        } catch (\PDOException $e) {
            \Log::error('LicenseValidation: Database PDO error', [
                'error' => $e->getMessage(),
                'code' => $e->getCode(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => substr($e->getTraceAsString(), 0, 300)
            ]);

            return response()->json([
                'flag' => 'invalid',
                'msg' => 'Database connection error. Please check your database configuration.'
            ], 200); // Return 200 to avoid triggering HTTP 500 in helper
        } catch (\Exception $e) {
            \Log::error('LicenseValidation: Unexpected exception', [
                'error' => $e->getMessage(),
                'type' => get_class($e),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => substr($e->getTraceAsString(), 0, 500) // Limit trace length
            ]);

            // Return a user-friendly error message with 200 status
            // This prevents the helper from showing "HTTP 500" error
            return response()->json([
                'flag' => 'invalid',
                'msg' => 'An error occurred during license validation: ' . $e->getMessage() . '. Please check server logs for details.'
            ], 200); // Return 200 to avoid triggering HTTP 500 in helper
        }
    }

    /**
     * Map product_id to module_name
     * 
     * @param int $product_id
     * @return string
     */
    private function getModuleNameFromProductId($product_id)
    {
        // Map product IDs to module names
        $mapping = [
            25 => 'businessmanagement',
            // Add more mappings as needed
            // 26 => 'othermodule',
        ];

        return $mapping[$product_id] ?? 'businessmanagement';
    }

    /**
     * Create module_licenses table if it doesn't exist
     * This is a fallback if migration hasn't been run
     * 
     * @return void
     * @throws \Exception
     */
    private function createModuleLicensesTable()
    {
        if (Schema::hasTable('module_licenses')) {
            return; // Table already exists
        }

        Schema::create('module_licenses', function (Blueprint $table) {
            $table->increments('id');
            $table->string('module_name', 100)->index();
            $table->string('license_code', 100)->unique();
            $table->string('username', 100);
            $table->string('email', 100)->nullable();
            $table->enum('license_type', ['single', 'multi', 'unlimited'])->default('single');
            $table->date('expiry_date')->nullable();
            $table->enum('status', ['active', 'inactive', 'expired', 'revoked'])->default('active');
            $table->text('notes')->nullable();
            $table->timestamps();

            $table->index(['module_name', 'license_code']);
            $table->index(['module_name', 'username']);
        });
    }
}

