<?php

/**
 * BusinessManagement License Validation Helper
 * 
 * Validates license code and username with license server
 * 
 * @param string $url Application URL
 * @param string $path Module path
 * @param string $license_code License code/purchase key
 * @param string $email User email
 * @param string $username Username/purchase username
 * @param int $type License validation type (1 = primary, 2 = backup)
 * @param int|null $pid Product ID
 * @return mixed Redirect response if invalid, null if valid
 */
function businessmanagement_license($url, $path, $license_code, $email, $username, $type = 1, $pid = null)
{
    try {
        // First, try local license validation (for self-hosted licenses)
        // This is the PRIMARY validation method since API endpoint may not exist
        $local_validation = businessmanagement_validate_local_license($license_code, $username);
        
        if ($local_validation === true) {
            // Local license is valid, return null to continue installation
            \Log::info('BusinessManagement: Local license validated successfully', ['username' => $username]);
            return null;
        }
        
        // If local validation failed, try remote validation as fallback
        $request_url = ($type == 1) 
            ? base64_decode(config('businessmanagement.lic1', '')) 
            : base64_decode(config('businessmanagement.lic2', ''));
        
        if (empty($request_url)) {
            \Log::warning('BusinessManagement: License validation URL not configured, using local validation only');
            
            // If local validation returned false (invalid), show error
            if ($local_validation === false) {
                return redirect()->back()
                    ->with('error', 'Invalid license. Please check your license code and username, or generate a new license locally.');
            }
            
            // If local validation returned null (not found), show error
            return redirect()->back()
                ->with('error', 'License not found. Please generate a license via License Management or contact support.');
        }
        
        $pid = is_null($pid) ? config('businessmanagement.pid', 25) : $pid;

        $ch = curl_init();
        
        $postData = http_build_query([
            'url' => $url,
            'path' => $path,
            'license_code' => $license_code,
            'email' => $email,
            'username' => $username,
            'product_id' => $pid
        ]);

        $curlConfig = [
            CURLOPT_URL => $request_url,
            CURLOPT_POST => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_POSTFIELDS => $postData,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/x-www-form-urlencoded',
                'User-Agent: BusinessManagement-Module/1.0'
            ]
        ];
        
        curl_setopt_array($ch, $curlConfig);
        $raw_response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        $curl_errno = curl_errno($ch);
        
        curl_close($ch);

        // Log the request for debugging
        \Log::info('BusinessManagement: License validation request', [
            'url' => $request_url,
            'http_code' => $http_code,
            'username' => $username,
            'pid' => $pid
        ]);

        if ($curl_errno) {
            \Log::warning('BusinessManagement: CURL error during license validation', [
                'error' => $curl_error,
                'errno' => $curl_errno,
                'url' => $request_url
            ]);
            
            // If remote validation fails, check if we have a local license
            if ($local_validation === false) {
                return redirect()->back()
                    ->with('error', 'Unable to connect to license server. Please check your internet connection or contact support. Error: ' . $curl_error);
            }
            
            return redirect()->back()
                ->with('error', 'License validation failed: ' . $curl_error);
        }

        // Handle HTTP errors (404, 500, etc.)
        if ($http_code >= 400) {
            \Log::error('BusinessManagement: License server returned HTTP error', [
                'http_code' => $http_code,
                'url' => $request_url,
                'response_preview' => substr($raw_response, 0, 500),
                'local_validation' => $local_validation
            ]);
            
            // Try to parse error message from response
            $error_message = null;
            if ($raw_response) {
                $decoded_error = json_decode($raw_response, true);
                if (is_array($decoded_error) && isset($decoded_error['msg'])) {
                    $error_message = $decoded_error['msg'];
                }
            }
            
            // If 404, the endpoint doesn't exist - try local validation or show helpful error
            if ($http_code == 404) {
                \Log::warning('BusinessManagement: License API endpoint not found (404). Falling back to local validation.');
                
                // If we have local validation that already failed, show error
                if ($local_validation === false) {
                    return redirect()->back()
                        ->with('error', 'License validation endpoint not found. Please verify the license server URL is correct, or ensure your license is registered locally.');
                }
                
                // If local validation returned null (not found), we can't validate
                // But since remote is down, we'll show a helpful message
                return redirect()->back()
                    ->with('error', 'License validation server is not available (404). Please contact support to verify your license or register it locally.');
            }
            
            // For 500 errors, try to show the actual error message from API
            if ($http_code == 500) {
                // If we have a parsed error message, use it
                if ($error_message) {
                    \Log::warning('BusinessManagement: License server returned 500 with message', [
                        'message' => $error_message
                    ]);
                    
                    // If local validation already failed, show the API error
                    if ($local_validation === false) {
                        return redirect()->back()
                            ->with('error', $error_message);
                    }
                    
                    // If local validation returned null, try to use local validation as fallback
                    return redirect()->back()
                        ->with('error', $error_message . ' Please try generating a license locally via License Management.');
                }
                
                // If no error message in response, show generic error
                \Log::error('BusinessManagement: License server returned 500 without error message', [
                    'response' => substr($raw_response, 0, 500)
                ]);
                
                // Fallback to local validation message if available
                if ($local_validation === false) {
                    return redirect()->back()
                        ->with('error', 'License server error. Please check your database connection or generate a license locally via License Management.');
                }
                
                return redirect()->back()
                    ->with('error', 'License server error (HTTP 500). Please check server logs or generate a license locally via License Management.');
            }
            
            // For other HTTP errors, show the error message if available
            $msg = $error_message ?: 'License server error (HTTP ' . $http_code . '). Please try again later or contact support.';
            return redirect()->back()
                ->with('error', $msg);
        }
        
        if ($raw_response) {
            // Log raw response for debugging (first 500 chars)
            \Log::info('BusinessManagement: Raw license server response', [
                'http_code' => $http_code,
                'response_length' => strlen($raw_response),
                'response_preview' => substr($raw_response, 0, 500),
                'request_url' => $request_url
            ]);
            
            // Try to decode JSON
            $decoded_result = json_decode($raw_response, true);
            $json_error = json_last_error();
            
            // If JSON decode failed, log the error and raw response
            if ($json_error !== JSON_ERROR_NONE) {
                \Log::error('BusinessManagement: JSON decode error', [
                    'json_error' => $json_error,
                    'json_error_msg' => json_last_error_msg(),
                    'raw_response' => substr($raw_response, 0, 500),
                    'response_length' => strlen($raw_response),
                    'http_code' => $http_code
                ]);
                
                // Check if response is HTML (error page)
                if (stripos($raw_response, '<html') !== false || stripos($raw_response, '<!DOCTYPE') !== false) {
                    \Log::error('BusinessManagement: Server returned HTML instead of JSON', [
                        'html_preview' => substr($raw_response, 0, 200)
                    ]);
                    
                    // If it's a 404 page, provide specific error
                    if (stripos($raw_response, '404') !== false || stripos($raw_response, 'Not Found') !== false) {
                        return redirect()->back()
                            ->with('error', 'License validation endpoint not found (404). Please verify the API endpoint URL is correct in the module configuration.');
                    }
                    
                    return redirect()->back()
                        ->with('error', 'License server returned an error page. Please check the server URL or contact support.');
                }
                
                // Check if response contains "valid" or "invalid" keywords (text response)
                if (stripos($raw_response, 'valid') !== false && stripos($raw_response, 'invalid') === false) {
                    \Log::info('BusinessManagement: Found "valid" in non-JSON response');
                    // If it says valid, allow installation
                    return null;
                }
                
                return redirect()->back()
                    ->with('error', 'Invalid response format from license server. The server may not be configured correctly. Please contact support.');
            }
            
            // Handle JSON response
            if (!is_array($decoded_result)) {
                \Log::error('BusinessManagement: Decoded result is not an array', [
                    'decoded_type' => gettype($decoded_result),
                    'decoded_value' => $decoded_result
                ]);
                return redirect()->back()
                    ->with('error', 'Unexpected response format from license server.');
            }
            
            // Check for valid flag in various formats
            $is_valid = false;
            $error_msg = null;
            
            // Check different possible response formats
            if (isset($decoded_result['flag'])) {
                if ($decoded_result['flag'] == 'valid' || $decoded_result['flag'] === true || $decoded_result['flag'] === 1) {
                    $is_valid = true;
                } else {
                    $error_msg = isset($decoded_result['msg']) ? $decoded_result['msg'] : 'License validation failed';
                }
            } elseif (isset($decoded_result['status'])) {
                if ($decoded_result['status'] == 'valid' || $decoded_result['status'] == 'success') {
                    $is_valid = true;
                } else {
                    $error_msg = isset($decoded_result['message']) ? $decoded_result['message'] : 'License validation failed';
                }
            } elseif (isset($decoded_result['valid'])) {
                if ($decoded_result['valid'] === true || $decoded_result['valid'] === 1) {
                    $is_valid = true;
                } else {
                    $error_msg = isset($decoded_result['message']) ? $decoded_result['message'] : 'License validation failed';
                }
            } elseif (isset($decoded_result['success'])) {
                if ($decoded_result['success'] === true || $decoded_result['success'] === 1) {
                    $is_valid = true;
                } else {
                    $error_msg = isset($decoded_result['message']) ? $decoded_result['message'] : 'License validation failed';
                }
            } else {
                // No standard format found, log the full response
                \Log::warning('BusinessManagement: Unknown response format', [
                    'response_keys' => array_keys($decoded_result),
                    'full_response' => $decoded_result
                ]);
                $error_msg = 'Unknown response format from license server';
            }
            
            if ($is_valid) {
                // License is valid, return null to continue installation
                \Log::info('BusinessManagement: Remote license validated successfully', [
                    'username' => $username,
                    'response_format' => 'detected'
                ]);
                return null;
            } else {
                $msg = $error_msg ?: (isset($decoded_result['msg']) ? $decoded_result['msg'] : "Invalid License Details. Please check your license code and username.");
                
                \Log::warning('BusinessManagement: License validation failed', [
                    'username' => $username,
                    'message' => $msg,
                    'full_response' => $decoded_result
                ]);
                
                return redirect()->back()
                    ->with('error', $msg);
            }
        }
        
        // If no response, treat as invalid
        \Log::error('BusinessManagement: No response from license server', [
            'http_code' => $http_code
        ]);
        
        return redirect()->back()
            ->with('error', 'Unable to validate license. Please check your internet connection and try again.');
            
    } catch (\Exception $e) {
        \Log::emergency('BusinessManagement: License validation exception', [
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ]);
        
        return redirect()->back()
            ->with('error', 'An error occurred during license validation: ' . $e->getMessage());
    }
}

/**
 * Validate license locally (for self-hosted license management)
 * 
 * @param string $license_code
 * @param string $username
 * @return bool|null Returns true if valid, false if invalid, null if not found locally
 */
function businessmanagement_validate_local_license($license_code, $username)
{
    try {
        // Check if module_licenses table exists - if not, try to create it
        if (!\Illuminate\Support\Facades\Schema::hasTable('module_licenses')) {
            // Try to auto-create the table
            try {
                \Illuminate\Support\Facades\Schema::create('module_licenses', function (\Illuminate\Database\Schema\Blueprint $table) {
                    $table->increments('id');
                    $table->string('module_name', 100)->index();
                    $table->string('license_code', 100)->unique();
                    $table->string('username', 100);
                    $table->string('email', 100)->nullable();
                    $table->enum('license_type', ['single', 'multi', 'unlimited'])->default('single');
                    $table->date('expiry_date')->nullable();
                    $table->enum('status', ['active', 'inactive', 'expired', 'revoked'])->default('active');
                    $table->text('notes')->nullable();
                    $table->timestamps();

                    $table->index(['module_name', 'license_code']);
                    $table->index(['module_name', 'username']);
                });
                
                \Log::info('BusinessManagement: module_licenses table auto-created during local validation');
            } catch (\Exception $createError) {
                \Log::warning('BusinessManagement: Failed to auto-create module_licenses table', [
                    'error' => $createError->getMessage()
                ]);
                // Continue - table doesn't exist, try remote validation
                return null;
            }
        }

        // Now check for license in database
        $license = \Illuminate\Support\Facades\DB::table('module_licenses')
            ->where('module_name', 'businessmanagement')
            ->where('license_code', $license_code)
            ->where('username', $username)
            ->whereIn('status', ['active', 'inactive']) // Allow both active and inactive for installation
            ->first();

        if ($license) {
            // Check if expired or revoked
            if ($license->status === 'expired' || $license->status === 'revoked') {
                return false; // License expired or revoked
            }
            
            // Check expiry date
            if ($license->expiry_date && 
                strtolower($license->expiry_date) !== 'lifetime' &&
                $license->expiry_date !== '0000-00-00' &&
                strtotime($license->expiry_date) < time()) {
                return false; // License expired
            }
            return true; // License is valid
        }

        // Also check System properties as fallback
        $property_key = 'license_businessmanagement_' . $license_code;
        $license_data = \App\System::getProperty($property_key);
        
        if ($license_data) {
            $data = json_decode($license_data, true);
            if ($data && $data['username'] == $username) {
                // Check status
                if (isset($data['status']) && ($data['status'] === 'expired' || $data['status'] === 'revoked')) {
                    return false; // License expired or revoked
                }
                
                // Check expiry
                if (isset($data['expiry_date']) && 
                    strtolower($data['expiry_date']) !== 'lifetime' &&
                    strtotime($data['expiry_date']) < time()) {
                    return false; // License expired
                }
                return true; // License is valid
            }
        }

        return null; // License not found locally, try remote validation
    } catch (\Exception $e) {
        \Log::warning('Local license validation error: ' . $e->getMessage(), [
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ]);
        return null; // Error occurred, try remote validation
    }
}

