<?php

namespace Modules\AdvancedReports\Tests\Unit;

use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\DB;
use App\Business;
use App\Category;
use App\Brands;

/**
 * Unit Tests for Category/Brand Filter Security
 *
 * Tests the database query builders to ensure they properly filter
 * by business_id and prevent cross-tenant data access.
 *
 * @group security
 * @group unit
 */
class CategoryFilterSecurityTest extends TestCase
{
    use RefreshDatabase;

    protected $businessA;
    protected $businessB;

    protected function setUp(): void
    {
        parent::setUp();

        $this->businessA = factory(Business::class)->create();
        $this->businessB = factory(Business::class)->create();
    }

    /**
     * Test: Category LEFT JOIN includes business_id filter
     *
     * @test
     */
    public function test_category_left_join_filters_by_business_id()
    {
        $categoryA = factory(Category::class)->create([
            'business_id' => $this->businessA->id,
            'name' => 'Test Category A'
        ]);

        $categoryB = factory(Category::class)->create([
            'business_id' => $this->businessB->id,
            'name' => 'Test Category B'
        ]);

        // Test query pattern used in fixed controllers
        $query = DB::table('categories as c')
            ->leftJoin('categories as c2', function($join) {
                $join->on('c.id', '=', 'c2.id')
                     ->where('c2.business_id', '=', $this->businessA->id);
            })
            ->where('c.business_id', $this->businessA->id)
            ->select('c.*', 'c2.name as joined_name')
            ->get();

        // Should only return Business A categories
        $this->assertEquals(1, $query->count());
        $this->assertEquals($categoryA->id, $query->first()->id);
        $this->assertEquals('Test Category A', $query->first()->name);
    }

    /**
     * Test: Brand LEFT JOIN includes business_id filter
     *
     * @test
     */
    public function test_brand_left_join_filters_by_business_id()
    {
        $brandA = factory(Brands::class)->create([
            'business_id' => $this->businessA->id,
            'name' => 'Brand A'
        ]);

        $brandB = factory(Brands::class)->create([
            'business_id' => $this->businessB->id,
            'name' => 'Brand B'
        ]);

        // Test query pattern
        $query = DB::table('brands as b')
            ->leftJoin('brands as b2', function($join) {
                $join->on('b.id', '=', 'b2.id')
                     ->where('b2.business_id', '=', $this->businessA->id);
            })
            ->where('b.business_id', $this->businessA->id)
            ->select('b.*')
            ->get();

        // Should only return Business A brands
        $this->assertEquals(1, $query->count());
        $this->assertEquals($brandA->id, $query->first()->id);
    }

    /**
     * Test: whereIn subquery validates business_id
     *
     * @test
     */
    public function test_wherein_subquery_validates_business_ownership()
    {
        $categoryA1 = factory(Category::class)->create(['business_id' => $this->businessA->id]);
        $categoryA2 = factory(Category::class)->create(['business_id' => $this->businessA->id]);
        $categoryB1 = factory(Category::class)->create(['business_id' => $this->businessB->id]);

        // Attempt to filter by all categories (including Business B)
        $categoryIds = [$categoryA1->id, $categoryA2->id, $categoryB1->id];

        // Secure pattern: Validate ownership in subquery
        $query = DB::table('categories')
            ->whereIn('id', function($subquery) use ($categoryIds) {
                $subquery->select('id')
                    ->from('categories')
                    ->where('business_id', $this->businessA->id)
                    ->whereIn('id', $categoryIds);
            })
            ->get();

        // Should only return Business A categories
        $this->assertEquals(2, $query->count());
        $returnedIds = $query->pluck('id')->toArray();
        $this->assertContains($categoryA1->id, $returnedIds);
        $this->assertContains($categoryA2->id, $returnedIds);
        $this->assertNotContains($categoryB1->id, $returnedIds,
            'Should not return Business B category');
    }

    /**
     * Test: Multiple business_id filters work correctly
     *
     * @test
     */
    public function test_multiple_business_id_filters_in_query()
    {
        $categoryA = factory(Category::class)->create(['business_id' => $this->businessA->id]);
        $brandA = factory(Brands::class)->create(['business_id' => $this->businessA->id]);

        $categoryB = factory(Category::class)->create(['business_id' => $this->businessB->id]);
        $brandB = factory(Brands::class)->create(['business_id' => $this->businessB->id]);

        // Query with multiple LEFT JOINs (pattern from fixed controllers)
        $query = DB::table('categories as c')
            ->leftJoin('brands as b', function($join) {
                $join->on('c.id', '=', 'b.id')
                     ->where('b.business_id', '=', $this->businessA->id);
            })
            ->where('c.business_id', $this->businessA->id)
            ->select('c.*', 'b.name as brand_name')
            ->get();

        // Should only see Business A data
        $this->assertEquals(1, $query->count());
        $this->assertEquals($categoryA->id, $query->first()->id);
    }

    /**
     * Test: Empty array handling in whereIn filters
     *
     * @test
     */
    public function test_empty_category_array_returns_all_business_categories()
    {
        factory(Category::class, 3)->create(['business_id' => $this->businessA->id]);
        factory(Category::class, 2)->create(['business_id' => $this->businessB->id]);

        // When category_ids is empty, should return all categories for the business
        $categoryIds = [];

        $query = DB::table('categories')
            ->where('business_id', $this->businessA->id);

        if (!empty($categoryIds)) {
            $query->whereIn('id', function($subquery) use ($categoryIds) {
                $subquery->select('id')
                    ->from('categories')
                    ->where('business_id', $this->businessA->id)
                    ->whereIn('id', $categoryIds);
            });
        }

        $results = $query->get();

        // Should return all Business A categories (3), not Business B (2)
        $this->assertEquals(3, $results->count());
    }

    /**
     * Test: Invalid category ID returns nothing
     *
     * @test
     */
    public function test_invalid_category_id_returns_no_data()
    {
        factory(Category::class)->create(['business_id' => $this->businessA->id]);

        // Try to access non-existent category
        $invalidCategoryId = 99999;

        $query = DB::table('categories')
            ->whereIn('id', function($subquery) use ($invalidCategoryId) {
                $subquery->select('id')
                    ->from('categories')
                    ->where('business_id', $this->businessA->id)
                    ->whereIn('id', [$invalidCategoryId]);
            })
            ->get();

        $this->assertEmpty($query, 'Should return empty result for invalid category ID');
    }

    /**
     * Test: Cross-business category ID returns nothing
     *
     * @test
     */
    public function test_cross_business_category_id_returns_nothing()
    {
        $categoryA = factory(Category::class)->create(['business_id' => $this->businessA->id]);
        $categoryB = factory(Category::class)->create(['business_id' => $this->businessB->id]);

        // Business A user tries to access Business B category
        $query = DB::table('categories')
            ->whereIn('id', function($subquery) use ($categoryB) {
                $subquery->select('id')
                    ->from('categories')
                    ->where('business_id', $this->businessA->id) // Business A filter
                    ->whereIn('id', [$categoryB->id]); // Business B category
            })
            ->get();

        $this->assertEmpty($query, 'Should not return categories from other businesses');
    }
}
