<?php

namespace Modules\AdvancedReports\Tests\Feature;

use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use App\User;
use App\Business;
use App\Category;
use App\Brands;
use App\Product;
use App\BusinessLocation;

/**
 * Multi-Tenant Security Test Suite
 *
 * Tests to ensure that category and brand filters properly validate business_id
 * to prevent cross-tenant data leakage in the AdvancedReports module.
 *
 * @group security
 * @group multi-tenant
 */
class MultiTenantSecurityTest extends TestCase
{
    use RefreshDatabase;

    protected $businessA;
    protected $businessB;
    protected $userA;
    protected $userB;
    protected $categoryA;
    protected $categoryB;
    protected $brandA;
    protected $brandB;

    /**
     * Set up test environment with two separate businesses
     */
    protected function setUp(): void
    {
        parent::setUp();

        // Create Business A
        $this->businessA = factory(Business::class)->create([
            'name' => 'Business A - Test Corp'
        ]);

        // Create Business B
        $this->businessB = factory(Business::class)->create([
            'name' => 'Business B - Competitor Corp'
        ]);

        // Create users for each business
        $this->userA = factory(User::class)->create([
            'business_id' => $this->businessA->id,
            'username' => 'user_business_a'
        ]);

        $this->userB = factory(User::class)->create([
            'business_id' => $this->businessB->id,
            'username' => 'user_business_b'
        ]);

        // Create categories for each business
        $this->categoryA = factory(Category::class)->create([
            'business_id' => $this->businessA->id,
            'name' => 'Category A - Electronics'
        ]);

        $this->categoryB = factory(Category::class)->create([
            'business_id' => $this->businessB->id,
            'name' => 'Category B - Clothing'
        ]);

        // Create brands for each business
        $this->brandA = factory(Brands::class)->create([
            'business_id' => $this->businessA->id,
            'name' => 'Brand A - TechCo'
        ]);

        $this->brandB = factory(Brands::class)->create([
            'business_id' => $this->businessB->id,
            'name' => 'Brand B - FashionCo'
        ]);
    }

    /**
     * Test: ProductCategoryController prevents cross-business category access
     *
     * @test
     */
    public function test_product_category_analytics_blocks_cross_business_category_access()
    {
        // Act as Business A user
        $this->actingAs($this->userA);

        // Try to access Business B's category
        $response = $this->getJson("/advanced-reports/product-category/analytics?" . http_build_query([
            'start_date' => now()->subDays(30)->format('Y-m-d'),
            'end_date' => now()->format('Y-m-d'),
            'category_id' => [$this->categoryB->id] // Business B category
        ]));

        // Assert no data is returned for Business B category
        $response->assertStatus(200);
        $categoryData = $response->json('category_contribution.categories');

        // Should be empty or not contain Business B data
        $this->assertEmpty($categoryData, 'Business A should not see Business B category data');
    }

    /**
     * Test: ProductCategoryController only returns owned categories
     *
     * @test
     */
    public function test_product_category_analytics_returns_only_owned_categories()
    {
        // Create products for both businesses
        $productA = factory(Product::class)->create([
            'business_id' => $this->businessA->id,
            'category_id' => $this->categoryA->id,
            'brand_id' => $this->brandA->id
        ]);

        $productB = factory(Product::class)->create([
            'business_id' => $this->businessB->id,
            'category_id' => $this->categoryB->id,
            'brand_id' => $this->brandB->id
        ]);

        // Act as Business A user
        $this->actingAs($this->userA);

        // Request analytics for both categories (attempt to access Business B data)
        $response = $this->getJson("/advanced-reports/product-category/analytics?" . http_build_query([
            'start_date' => now()->subDays(30)->format('Y-m-d'),
            'end_date' => now()->format('Y-m-d'),
            'category_id' => [$this->categoryA->id, $this->categoryB->id]
        ]));

        $response->assertStatus(200);
        $categories = $response->json('category_contribution.categories');

        // Should only return Business A categories
        foreach ($categories as $category) {
            $this->assertEquals($this->categoryA->id, $category['category_id'],
                'Should only return categories owned by Business A');
        }
    }

    /**
     * Test: Brand filter prevents cross-business access
     *
     * @test
     */
    public function test_product_category_analytics_blocks_cross_business_brand_access()
    {
        $this->actingAs($this->userA);

        // Try to filter by Business B's brand
        $response = $this->getJson("/advanced-reports/product-category/analytics?" . http_build_query([
            'start_date' => now()->subDays(30)->format('Y-m-d'),
            'end_date' => now()->format('Y-m-d'),
            'brand_id' => [$this->brandB->id] // Business B brand
        ]));

        $response->assertStatus(200);
        $categoryData = $response->json('category_contribution.categories');

        // Should be empty or not contain Business B brand data
        $this->assertEmpty($categoryData, 'Business A should not see Business B brand data');
    }

    /**
     * Test: StockReportController prevents category data leakage
     *
     * @test
     */
    public function test_stock_report_blocks_cross_business_category_display()
    {
        // Create products with categories
        $productA = factory(Product::class)->create([
            'business_id' => $this->businessA->id,
            'category_id' => $this->categoryA->id
        ]);

        $productB = factory(Product::class)->create([
            'business_id' => $this->businessB->id,
            'category_id' => $this->categoryB->id
        ]);

        $this->actingAs($this->userA);

        $response = $this->getJson("/advanced-reports/stock/data");

        $response->assertStatus(200);

        // Verify no Business B category names appear in the response
        $responseBody = json_encode($response->json());
        $this->assertStringNotContainsString($this->categoryB->name, $responseBody,
            'Stock report should not display Business B category names');
        $this->assertStringNotContainsString('Category B', $responseBody,
            'Stock report should not display Business B data');
    }

    /**
     * Test: ItemwiseSalesReportController filters categories correctly
     *
     * @test
     */
    public function test_itemwise_sales_blocks_cross_business_categories()
    {
        $this->actingAs($this->userA);

        $response = $this->getJson("/advanced-reports/itemwise-sales/data?" . http_build_query([
            'start_date' => now()->subDays(30)->format('Y-m-d'),
            'end_date' => now()->format('Y-m-d'),
            'category_id' => $this->categoryB->id // Try to access Business B category
        ]));

        $response->assertStatus(200);

        // Should not return any data for Business B category
        $data = $response->json('data');
        $this->assertEmpty($data, 'Should not return data for Business B category');
    }

    /**
     * Test: Multiple business users cannot see each other's data
     *
     * @test
     */
    public function test_complete_data_isolation_between_businesses()
    {
        // Create comprehensive test data for both businesses
        $this->createTestData($this->businessA, $this->categoryA, $this->brandA);
        $this->createTestData($this->businessB, $this->categoryB, $this->brandB);

        // Test as Business A
        $this->actingAs($this->userA);
        $responseA = $this->getJson("/advanced-reports/product-category/analytics?" . http_build_query([
            'start_date' => now()->subDays(30)->format('Y-m-d'),
            'end_date' => now()->format('Y-m-d'),
        ]));

        // Test as Business B
        $this->actingAs($this->userB);
        $responseB = $this->getJson("/advanced-reports/product-category/analytics?" . http_build_query([
            'start_date' => now()->subDays(30)->format('Y-m-d'),
            'end_date' => now()->format('Y-m-d'),
        ]));

        // Verify complete data isolation
        $dataA = json_encode($responseA->json());
        $dataB = json_encode($responseB->json());

        $this->assertStringNotContainsString($this->categoryB->name, $dataA,
            'Business A should not see Business B categories');
        $this->assertStringNotContainsString($this->categoryA->name, $dataB,
            'Business B should not see Business A categories');
        $this->assertStringNotContainsString($this->brandB->name, $dataA,
            'Business A should not see Business B brands');
        $this->assertStringNotContainsString($this->brandA->name, $dataB,
            'Business B should not see Business A brands');
    }

    /**
     * Test: SQL injection attempt via category_id parameter
     *
     * @test
     */
    public function test_sql_injection_protection_in_category_filter()
    {
        $this->actingAs($this->userA);

        // Attempt SQL injection via category_id
        $response = $this->getJson("/advanced-reports/product-category/analytics?" . http_build_query([
            'start_date' => now()->subDays(30)->format('Y-m-d'),
            'end_date' => now()->format('Y-m-d'),
            'category_id' => ["1' OR '1'='1", "999 UNION SELECT * FROM users--"]
        ]));

        // Should handle gracefully without executing malicious SQL
        $response->assertStatus(200);
        // Should return empty or only valid category data
        $categoryData = $response->json('category_contribution.categories');
        $this->assertTrue(is_array($categoryData), 'Should return array, not error');
    }

    /**
     * Test: Direct ID manipulation in URL cannot bypass security
     *
     * @test
     */
    public function test_direct_id_manipulation_blocked()
    {
        $this->actingAs($this->userA);

        // Directly try to access Business B category by manipulating IDs
        for ($attemptId = $this->categoryB->id - 5; $attemptId <= $this->categoryB->id + 5; $attemptId++) {
            $response = $this->getJson("/advanced-reports/product-category/analytics?" . http_build_query([
                'category_id' => [$attemptId]
            ]));

            $categoryData = $response->json('category_contribution.categories');

            foreach ($categoryData as $category) {
                $this->assertNotEquals($this->categoryB->id, $category['category_id'] ?? null,
                    "Should not return Business B category (ID: {$this->categoryB->id}) when attempting ID: {$attemptId}");
            }
        }
    }

    /**
     * Helper: Create test data for a business
     */
    private function createTestData($business, $category, $brand)
    {
        factory(Product::class, 5)->create([
            'business_id' => $business->id,
            'category_id' => $category->id,
            'brand_id' => $brand->id
        ]);
    }
}
