<?php

namespace Modules\AccountingReports\Services;

use Modules\AccountingReports\Entities\ChartOfAccount;
use Modules\AccountingReports\Entities\JournalEntryLine;
use Modules\AccountingReports\Entities\JournalEntryHeader;
use App\Account;
use App\AccountTransaction;
use App\Utils\TransactionUtil;
use App\BusinessLocation;
use Illuminate\Support\Facades\Schema;
use DB;

class TrialBalanceService
{
    protected $transactionUtil;

    public function __construct(TransactionUtil $transactionUtil)
    {
        $this->transactionUtil = $transactionUtil;
    }

    /**
     * Get trial balance data (matching UltimatePOS logic)
     */
    public function getTrialBalance($businessId, $fromDate, $toDate, $locationId = null)
    {
        // Use toDate as end_date (like UltimatePOS original)
        $endDate = $toDate ?: $fromDate;
        
        // Validate dates
        if (!$endDate) {
            $endDate = \Carbon\Carbon::now()->format('Y-m-d');
        }

        // Get supplier due (Accounts Payable) - matches UltimatePOS logic
        try {
            $purchase_details = $this->transactionUtil->getPurchaseTotals(
                $businessId,
                null,
                $endDate,
                $locationId
            );
            $supplier_due = 0;
            if ($purchase_details && isset($purchase_details['purchase_due'])) {
                $supplier_due = (float)$purchase_details['purchase_due'];
            }
        } catch (\Exception $e) {
            \Log::error('TrialBalance: Error getting purchase totals - ' . $e->getMessage());
            $supplier_due = 0;
        }

        // Get customer due (Accounts Receivable) - matches UltimatePOS logic
        try {
            $sell_details = $this->transactionUtil->getSellTotals(
                $businessId,
                null,
                $endDate,
                $locationId
            );
            $customer_due = 0;
            if ($sell_details && isset($sell_details['invoice_due'])) {
                $customer_due = (float)$sell_details['invoice_due'];
            }
        } catch (\Exception $e) {
            \Log::error('TrialBalance: Error getting sell totals - ' . $e->getMessage());
            $customer_due = 0;
        }

        // Get account balances (from account_transactions table - matches UltimatePOS)
        try {
            $account_balances = $this->getAccountBalanceLegacy($businessId, $endDate, $locationId);
        } catch (\Exception $e) {
            \Log::error('TrialBalance: Error getting account balances legacy - ' . $e->getMessage());
            $account_balances = [];
        }

        // Also try to get from new journal entries if available
        try {
            $journal_account_balances = $this->getAccountBalanceFromJournals($businessId, $endDate, $locationId);
        } catch (\Exception $e) {
            \Log::error('TrialBalance: Error getting journal balances - ' . $e->getMessage());
            $journal_account_balances = [];
        }

        // Merge both sources (prioritize journal entries if available)
        $all_accounts = [];
        foreach ($journal_account_balances as $name => $balance) {
            $all_accounts[$name] = $balance;
        }
        foreach ($account_balances as $name => $balance) {
            // Only add if not already in journal entries
            if (!isset($all_accounts[$name])) {
                $all_accounts[$name] = $balance;
            }
        }

        // Calculate totals (matching UltimatePOS logic)
        // In UltimatePOS: positive balance = credit, negative = debit
        $total_debit = 0;
        $total_credit = 0;
        
        // Customer due (Accounts Receivable) - debit
        if ($customer_due > 0) {
            $total_debit += abs($customer_due);
        }
        
        // Supplier due (Accounts Payable) - credit
        if ($supplier_due > 0) {
            $total_credit += abs($supplier_due);
        }
        
        foreach ($all_accounts as $name => $balance) {
            if ($balance >= 0) {
                $total_credit += abs($balance);
            } else {
                $total_debit += abs($balance);
            }
        }

        return [
            'supplier_due' => $supplier_due,
            'customer_due' => $customer_due,
            'account_balances' => $all_accounts,
            'total_debit' => $total_debit,
            'total_credit' => $total_credit,
            'is_balanced' => abs($total_debit - $total_credit) < 0.01,
            // Also include detailed format for new view
            'accounts' => $this->formatAccountsForDetailedView($all_accounts, $customer_due, $supplier_due),
            'totals' => [
                'opening_debit' => 0,
                'opening_credit' => 0,
                'period_debit' => 0,
                'period_credit' => 0,
                'closing_debit' => $total_debit,
                'closing_credit' => $total_credit,
            ],
        ];
    }

    /**
     * Get account balances from legacy account_transactions table (matches UltimatePOS logic)
     */
    protected function getAccountBalanceLegacy($businessId, $endDate, $locationId = null)
    {
        // Match UltimatePOS logic exactly
        try {
            $query = Account::leftjoin(
                'account_transactions as AT',
                'AT.account_id',
                '=',
                'accounts.id'
            )
            ->whereNull('AT.deleted_at')
            ->where('accounts.business_id', $businessId)
            ->whereDate('AT.operation_date', '<=', $endDate);

        $permitted_locations = auth()->user()->permitted_locations();
        $account_ids = [];
        
        if ($permitted_locations != 'all') {
            $locations = BusinessLocation::where('business_id', $businessId)
                            ->whereIn('id', $permitted_locations)
                            ->get();

            foreach ($locations as $location) {
                if (!empty($location->default_payment_accounts)) {
                    $default_payment_accounts = json_decode($location->default_payment_accounts, true);
                    foreach ($default_payment_accounts as $key => $account) {
                        if (!empty($account['is_enabled']) && !empty($account['account'])) {
                            $account_ids[] = $account['account'];
                        }
                    }
                }
            }
            $account_ids = array_unique($account_ids);
        }

        if ($permitted_locations != 'all') {
            $query->whereIn('accounts.id', $account_ids);
        }

        if (!empty($locationId)) {
            $location = BusinessLocation::find($locationId);
            if (!empty($location->default_payment_accounts)) {
                $default_payment_accounts = json_decode($location->default_payment_accounts, true);
                $account_ids = [];
                foreach ($default_payment_accounts as $key => $account) {
                    if (!empty($account['is_enabled']) && !empty($account['account'])) {
                        $account_ids[] = $account['account'];
                    }
                }
                $query->whereIn('accounts.id', $account_ids);
            }
        }

            $account_details = $query->select([
                'accounts.name',
                DB::raw("COALESCE(SUM(IF(AT.type='credit', AT.amount, -1*AT.amount)), 0) as balance")
            ])
            ->groupBy('accounts.id')
            ->get()
            ->pluck('balance', 'name')
            ->toArray();

            // Filter out zero balances
            $account_details = array_filter($account_details, function($balance) {
                return abs((float)$balance) > 0.01;
            });

            return $account_details;
        } catch (\Exception $e) {
            \Log::error('TrialBalance getAccountBalanceLegacy Error: ' . $e->getMessage());
            \Log::error('Stack: ' . $e->getTraceAsString());
            return [];
        }
    }

    /**
     * Get account balances from new journal entries
     */
    protected function getAccountBalanceFromJournals($businessId, $endDate, $locationId = null)
    {
        // Check if journal entry tables exist
        if (!\Schema::hasTable('ar_journal_entry_headers') || !\Schema::hasTable('ar_journal_entry_lines')) {
            return [];
        }

        try {
            $query = JournalEntryLine::whereHas('journalEntry', function($q) use ($businessId, $endDate, $locationId) {
                $q->where('business_id', $businessId)
                  ->where('status', 'posted')
                  ->where('voucher_date', '<=', $endDate);
                
                if ($locationId) {
                    $q->where('location_id', $locationId);
                }
            })
            ->join('ar_chart_of_accounts as coa', 'ar_journal_entry_lines.account_id', '=', 'coa.id')
            ->where('coa.business_id', $businessId)
            ->select([
                'coa.name',
                DB::raw("COALESCE(SUM(debit - credit), 0) as balance")
            ])
            ->groupBy('coa.id', 'coa.name')
            ->get();

            $balances = [];
            foreach ($query as $row) {
                $balance = (float) ($row->balance ?? 0);
                if (abs($balance) > 0.01) {
                    $balances[$row->name] = $balance;
                }
            }

            return $balances;
        } catch (\Exception $e) {
            \Log::error('TrialBalance: Error in getAccountBalanceFromJournals - ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Format accounts for detailed view (opening/period/closing format)
     */
    protected function formatAccountsForDetailedView($accountBalances, $customerDue, $supplierDue)
    {
        $accounts = [];
        
        // Add customer due as Accounts Receivable
        if (abs($customerDue) > 0.01) {
            $accounts[] = [
                'account_code' => '1300',
                'account_name' => 'Accounts Receivable',
                'opening_debit' => 0,
                'opening_credit' => 0,
                'period_debit' => abs($customerDue),
                'period_credit' => 0,
                'closing_debit' => abs($customerDue),
                'closing_credit' => 0,
            ];
        }
        
        // Add supplier due as Accounts Payable
        if (abs($supplierDue) > 0.01) {
            $accounts[] = [
                'account_code' => '3100',
                'account_name' => 'Accounts Payable',
                'opening_debit' => 0,
                'opening_credit' => 0,
                'period_debit' => 0,
                'period_credit' => abs($supplierDue),
                'closing_debit' => 0,
                'closing_credit' => abs($supplierDue),
            ];
        }
        
        // Add other accounts
        foreach ($accountBalances as $name => $balance) {
            if (abs($balance) > 0.01) {
                $accounts[] = [
                    'account_code' => '',
                    'account_name' => $name,
                    'opening_debit' => 0,
                    'opening_credit' => 0,
                    'period_debit' => $balance >= 0 ? 0 : abs($balance),
                    'period_credit' => $balance >= 0 ? abs($balance) : 0,
                    'closing_debit' => $balance >= 0 ? 0 : abs($balance),
                    'closing_credit' => $balance >= 0 ? abs($balance) : 0,
                ];
            }
        }
        
        return $accounts;
    }


}


