<?php

namespace Modules\AccountingReports\Services;

use App\Utils\TransactionUtil;
use App\Utils\BusinessUtil;
use App\Account;
use App\AccountTransaction;
use App\BusinessLocation;
use Modules\AccountingReports\Services\TrialBalanceService;
use DB;

class RatioAnalysisService
{
    protected $transactionUtil;
    protected $businessUtil;
    protected $trialBalanceService;

    public function __construct(TransactionUtil $transactionUtil, BusinessUtil $businessUtil, TrialBalanceService $trialBalanceService)
    {
        $this->transactionUtil = $transactionUtil;
        $this->businessUtil = $businessUtil;
        $this->trialBalanceService = $trialBalanceService;
    }

    /**
     * Calculate all financial ratios for a given period
     */
    public function calculateRatios($businessId, $startDate, $endDate, $locationId = null)
    {
        try {
            $permitted_locations = auth()->user()->permitted_locations();
            
            // Validate dates
            if (empty($startDate) || empty($endDate)) {
                throw new \Exception('Start date and end date are required');
            }

            // Get P&L data
            $plData = $this->transactionUtil->getProfitLossDetails(
                $businessId, 
                $locationId, 
                $startDate, 
                $endDate, 
                null, 
                $permitted_locations
            );

            if (!is_array($plData)) {
                $plData = [];
            }

            // Get trial balance data for balance sheet items
            $trialBalance = $this->trialBalanceService->getTrialBalance(
                $businessId, 
                $endDate, 
                $endDate, 
                $locationId
            );

            if (!is_array($trialBalance)) {
                $trialBalance = [];
            }

            // Get account balances
            $accountBalances = $this->getAccountBalances($businessId, $endDate, $locationId);

            // Calculate sales
            $sales = (float) ($plData['total_sell'] ?? 0);
            $salesDiscount = (float) ($plData['total_sell_discount'] ?? 0);
            $netSales = $sales - abs($salesDiscount);

            // Calculate gross profit
            $totalPurchase = (float) ($plData['total_purchase'] ?? 0);
            $openingStock = (float) ($plData['opening_stock'] ?? 0);
            $closingStock = (float) ($plData['closing_stock'] ?? 0);
            $cogs = $openingStock + $totalPurchase - $closingStock;
            $grossProfit = $netSales - $cogs;

            // Calculate net profit
            // Get total expense from P&L data
            // total_expense from getProfitLossDetails includes expense transactions
            $totalExpense = (float) ($plData['total_expense'] ?? 0);
            
            // Also add direct expenses if they're tracked separately
            // Direct expenses are typically part of COGS, but some might be operating expenses
            // For ratio analysis, we'll use total_expense from P&L which should include all expenses
            
            $netProfit = $grossProfit - $totalExpense;

            // Get balance sheet items
            $currentAssets = $this->getCurrentAssets($accountBalances, $trialBalance, $closingStock);
            $currentLiabilities = $this->getCurrentLiabilities($accountBalances, $trialBalance);
            $stockInHand = $closingStock;
            $loans = $this->getLoans($accountBalances);
            $capitalAccount = $this->getCapitalAccount($accountBalances);
            
            // Calculate working capital
            $workingCapital = $currentAssets - $currentLiabilities;

            // Calculate days in period
            // Ensure dates are in proper format before parsing
            try {
                if (is_string($startDate)) {
                    // Try to parse as Y-m-d format first
                    $start = \Carbon\Carbon::createFromFormat('Y-m-d', $startDate);
                } else {
                    $start = \Carbon\Carbon::parse($startDate);
                }
            } catch (\Exception $e) {
                // Fallback: try standard parse
                $start = \Carbon\Carbon::parse($startDate);
            }
            
            try {
                if (is_string($endDate)) {
                    // Try to parse as Y-m-d format first
                    $end = \Carbon\Carbon::createFromFormat('Y-m-d', $endDate);
                } else {
                    $end = \Carbon\Carbon::parse($endDate);
                }
            } catch (\Exception $e) {
                // Fallback: try standard parse
                $end = \Carbon\Carbon::parse($endDate);
            }
            
            $daysInPeriod = $start->diffInDays($end) + 1;

            // Calculate receivables (customer due)
            $customerDue = (float) ($trialBalance['customer_due'] ?? 0);

            // Calculate ratios
            $ratios = [];

            // 1. Current Ratio
            $currentRatio = $currentLiabilities > 0 ? $currentAssets / $currentLiabilities : 0;
            $ratios[] = [
                'name' => 'Current Ratio',
                'value' => $currentRatio > 0 ? number_format($currentRatio, 2) . ' : 1' : '0.00 : 1',
                'formula' => '(Current Assets : Current Liabilities)',
                'raw_value' => $currentRatio
            ];

            // 2. Quick Ratio
            $quickAssets = $currentAssets - $stockInHand;
            $quickRatio = $currentLiabilities > 0 ? $quickAssets / $currentLiabilities : 0;
            $ratios[] = [
                'name' => 'Quick Ratio',
                'value' => $quickRatio > 0 ? number_format($quickRatio, 2) . ' : 1' : '0.00 : 1',
                'formula' => '(Current Assets-Stock-in-Hand : Current Liabilities)',
                'raw_value' => $quickRatio
            ];

            // 3. Debt/Equity Ratio
            $equity = $capitalAccount + $netProfit;
            $debtEquityRatio = $equity > 0 ? $loans / $equity : 0;
            $ratios[] = [
                'name' => 'Debt/Equity Ratio',
                'value' => $debtEquityRatio > 0 ? number_format($debtEquityRatio, 2) . ' : 1' : '0.00 : 1',
                'formula' => '(Loans (Liability) : Capital Account + Nett Profit)',
                'raw_value' => $debtEquityRatio
            ];

            // 4. Gross Profit %
            $grossProfitPercent = $netSales > 0 ? ($grossProfit / $netSales) * 100 : 0;
            $ratios[] = [
                'name' => 'Gross Profit %',
                'value' => number_format($grossProfitPercent, 2) . ' %',
                'formula' => null,
                'raw_value' => $grossProfitPercent
            ];

            // 5. Net Profit %
            $netProfitPercent = $netSales > 0 ? ($netProfit / $netSales) * 100 : 0;
            $ratios[] = [
                'name' => 'Nett Profit %',
                'value' => number_format($netProfitPercent, 2) . ' %',
                'formula' => null,
                'raw_value' => $netProfitPercent
            ];

            // 6. Operating Cost %
            $operatingCost = $cogs + $totalExpense;
            $operatingCostPercent = $netSales > 0 ? ($operatingCost / $netSales) * 100 : 0;
            $ratios[] = [
                'name' => 'Operating Cost %',
                'value' => number_format($operatingCostPercent, 2) . ' %',
                'formula' => '(as percentage of Sales Accounts)',
                'raw_value' => $operatingCostPercent
            ];

            // 7. Receivables Turnover in days
            $receivablesTurnoverDays = $netSales > 0 ? ($customerDue / $netSales) * $daysInPeriod : 0;
            $ratios[] = [
                'name' => 'Recv. Turnover in days',
                'value' => number_format($receivablesTurnoverDays, 2) . ' days',
                'formula' => '(payment performance of Debtors)',
                'raw_value' => $receivablesTurnoverDays
            ];

            // 8. Return on Investment %
            $roi = $equity > 0 ? ($netProfit / $equity) * 100 : 0;
            $ratios[] = [
                'name' => 'Return on Investment %',
                'value' => number_format($roi, 2) . ' %',
                'formula' => '(Nett Profit / Capital Account + Nett Profit)',
                'raw_value' => $roi
            ];

            // 9. Return on Working Capital %
            $rowc = $workingCapital > 0 ? ($netProfit / $workingCapital) * 100 : 0;
            $ratios[] = [
                'name' => 'Return on Wkg. Capital %',
                'value' => number_format($rowc, 2) . ' %',
                'formula' => '(Nett Profit / Working Capital) %',
                'raw_value' => $rowc
            ];

            return [
                'ratios' => $ratios,
                'period' => [
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                    'days' => $daysInPeriod
                ],
                'summary' => [
                    'current_assets' => $currentAssets,
                    'current_liabilities' => $currentLiabilities,
                    'stock_in_hand' => $stockInHand,
                    'loans' => $loans,
                    'capital_account' => $capitalAccount,
                    'net_profit' => $netProfit,
                    'working_capital' => $workingCapital,
                    'net_sales' => $netSales,
                    'gross_profit' => $grossProfit,
                    'customer_due' => $customerDue
                ]
            ];
        } catch (\Exception $e) {
            \Log::error('RatioAnalysisService calculateRatios Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            throw $e;
        }
    }

    /**
     * Get account balances from account_transactions
     */
    protected function getAccountBalances($businessId, $endDate, $locationId = null)
    {
        $query = Account::leftjoin(
            'account_transactions as AT',
            'AT.account_id',
            '=',
            'accounts.id'
        )
        ->whereNull('AT.deleted_at')
        ->where('accounts.business_id', $businessId)
        ->whereDate('AT.operation_date', '<=', $endDate);

        $permitted_locations = auth()->user()->permitted_locations();
        $account_ids = [];
        
        if ($permitted_locations != 'all') {
            $locations = BusinessLocation::where('business_id', $businessId)
                            ->whereIn('id', $permitted_locations)
                            ->get();

            foreach ($locations as $location) {
                if (!empty($location->default_payment_accounts)) {
                    $default_payment_accounts = json_decode($location->default_payment_accounts, true);
                    foreach ($default_payment_accounts as $key => $account) {
                        if (!empty($account['is_enabled']) && !empty($account['account'])) {
                            $account_ids[] = $account['account'];
                        }
                    }
                }
            }
            $account_ids = array_unique($account_ids);
        }

        if ($permitted_locations != 'all') {
            $query->whereIn('accounts.id', $account_ids);
        }

        if (!empty($locationId)) {
            $location = BusinessLocation::find($locationId);
            if (!empty($location->default_payment_accounts)) {
                $default_payment_accounts = json_decode($location->default_payment_accounts, true);
                $account_ids = [];
                foreach ($default_payment_accounts as $key => $account) {
                    if (!empty($account['is_enabled']) && !empty($account['account'])) {
                        $account_ids[] = $account['account'];
                    }
                }
                $query->whereIn('accounts.id', $account_ids);
            }
        }

        // Check if account_type column exists
        $hasAccountTypeColumn = \Schema::hasColumn('accounts', 'account_type');
        
        $selectFields = [
            'accounts.id',
            'accounts.name',
            'accounts.account_type_id',
            DB::raw("COALESCE(SUM(IF(AT.type='credit', AT.amount, -1*AT.amount)), 0) as balance")
        ];
        
        $groupByFields = ['accounts.id', 'accounts.name', 'accounts.account_type_id'];
        
        if ($hasAccountTypeColumn) {
            $selectFields[] = 'accounts.account_type as account_type_enum';
            $groupByFields[] = 'accounts.account_type';
        }
        
        $account_details = $query->select($selectFields)
        ->groupBy($groupByFields)
        ->get();

        $balances = [];
        foreach ($account_details as $account) {
            // Determine account type from account_type_id relationship or account_type enum
            $accountType = 'asset'; // default
            
            // First try account_type_id relationship
            if ($account->account_type_id) {
                try {
                    $accountTypeModel = \App\AccountType::find($account->account_type_id);
                    if ($accountTypeModel) {
                        $typeName = strtolower($accountTypeModel->name ?? '');
                        // Map account type names to our types
                        if (strpos($typeName, 'asset') !== false || strpos($typeName, 'current asset') !== false || strpos($typeName, 'bank') !== false || strpos($typeName, 'cash') !== false) {
                            $accountType = 'asset';
                        } elseif (strpos($typeName, 'liability') !== false || strpos($typeName, 'payable') !== false || strpos($typeName, 'loan') !== false) {
                            $accountType = 'liability';
                        } elseif (strpos($typeName, 'equity') !== false || strpos($typeName, 'capital') !== false) {
                            $accountType = 'equity';
                        }
                    }
                } catch (\Exception $e) {
                    // If account_types table doesn't exist, continue with other checks
                }
            }
            
            // Fallback to account_type enum if available
            if ($hasAccountTypeColumn && isset($account->account_type_enum)) {
                if ($account->account_type_enum == 'capital') {
                    $accountType = 'equity';
                } elseif ($account->account_type_enum == 'saving_current') {
                    $accountType = 'asset';
                }
            }
            
            // Final fallback: infer from account name
            if ($accountType == 'asset') {
                $name = strtolower($account->name);
                if (strpos($name, 'loan') !== false || strpos($name, 'borrowing') !== false || strpos($name, 'payable') !== false || strpos($name, 'liability') !== false) {
                    $accountType = 'liability';
                } elseif (strpos($name, 'capital') !== false || strpos($name, 'equity') !== false) {
                    $accountType = 'equity';
                }
            }
            
            $balances[$account->id] = [
                'name' => $account->name,
                'type' => $accountType,
                'balance' => (float) $account->balance
            ];
        }

        return $balances;
    }

    /**
     * Get current assets from account balances
     */
    protected function getCurrentAssets($accountBalances, $trialBalance, $stockInHand)
    {
        $currentAssets = $stockInHand; // Start with inventory
        
        // Add cash and bank accounts (typically asset type)
        foreach ($accountBalances as $account) {
            if (in_array(strtolower($account['type']), ['asset', 'bank', 'cash']) && $account['balance'] > 0) {
                $currentAssets += abs($account['balance']);
            }
        }

        // Add customer due (accounts receivable)
        $currentAssets += abs($trialBalance['customer_due'] ?? 0);

        return $currentAssets;
    }

    /**
     * Get current liabilities from account balances
     */
    protected function getCurrentLiabilities($accountBalances, $trialBalance)
    {
        $currentLiabilities = 0;

        // Add supplier due (accounts payable)
        $currentLiabilities += abs($trialBalance['supplier_due'] ?? 0);

        // Add liability accounts
        foreach ($accountBalances as $account) {
            if (strtolower($account['type']) == 'liability' && $account['balance'] > 0) {
                $currentLiabilities += abs($account['balance']);
            }
        }

        return $currentLiabilities;
    }

    /**
     * Get loans (liability accounts that are loans)
     */
    protected function getLoans($accountBalances)
    {
        $loans = 0;

        foreach ($accountBalances as $account) {
            $name = strtolower($account['name']);
            if (
                strtolower($account['type']) == 'liability' && 
                (strpos($name, 'loan') !== false || 
                 strpos($name, 'borrowing') !== false ||
                 strpos($name, 'debt') !== false) &&
                $account['balance'] > 0
            ) {
                $loans += abs($account['balance']);
            }
        }

        return $loans;
    }

    /**
     * Get capital account balance
     */
    protected function getCapitalAccount($accountBalances)
    {
        $capital = 0;

        foreach ($accountBalances as $account) {
            $name = strtolower($account['name']);
            if (
                (strtolower($account['type']) == 'equity' || 
                 strpos($name, 'capital') !== false ||
                 strpos($name, 'equity') !== false) &&
                $account['balance'] > 0
            ) {
                $capital += abs($account['balance']);
            }
        }

        return $capital;
    }
}

