<?php

namespace Modules\AccountingReports\Services;

use Modules\AccountingReports\Entities\ChequeBookEntry;
use Modules\AccountingReports\Entities\ChequeBook;
use Modules\AccountingReports\Entities\BankAccount;
use App\AccountTransaction;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ChequeEntryService
{
    /**
     * Create a new cheque entry
     */
    public function createChequeEntry(array $data, int $businessId, int $userId): ChequeBookEntry
    {
        DB::beginTransaction();
        
        try {
            // Validate cheque number uniqueness for the account
            $this->validateChequeNumber($data['bank_account_id'], $data['cheque_number']);

            $chequeEntry = ChequeBookEntry::create([
                'business_id' => $businessId,
                'bank_account_id' => $data['bank_account_id'],
                'cheque_book_id' => $data['cheque_book_id'] ?? null,
                'cheque_number' => $data['cheque_number'],
                'cheque_date' => $data['cheque_date'] ?? Carbon::now(),
                'type' => $data['type'], // 'issued' or 'received'
                'status' => $data['status'] ?? 'pending',
                'amount' => $data['amount'] ?? 0,
                'payee_name' => $data['payee_name'] ?? null,
                'description' => $data['description'] ?? null,
                'transaction_payment_id' => $data['transaction_payment_id'] ?? null,
                'account_transaction_id' => $data['account_transaction_id'] ?? null,
                'presented_date' => $data['presented_date'] ?? null,
                'cleared_date' => $data['cleared_date'] ?? null,
                'bounced_date' => $data['bounced_date'] ?? null,
                'bounce_reason' => $data['bounce_reason'] ?? null,
                'created_by' => $userId,
            ]);

            // Link to account transaction if provided
            if (isset($data['account_transaction_id'])) {
                $this->linkToAccountTransaction($chequeEntry, $data['account_transaction_id']);
            }

            DB::commit();
            return $chequeEntry->fresh(['bankAccount', 'chequeBook', 'accountTransaction']);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('ChequeEntryService::createChequeEntry - Error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Update cheque entry
     */
    public function updateChequeEntry(ChequeBookEntry $chequeEntry, array $data): ChequeBookEntry
    {
        DB::beginTransaction();
        
        try {
            // Validate cheque number if changed
            if (isset($data['cheque_number']) && $data['cheque_number'] != $chequeEntry->cheque_number) {
                $this->validateChequeNumber($chequeEntry->bank_account_id, $data['cheque_number'], $chequeEntry->id);
            }

            // Handle status changes
            if (isset($data['status']) && $data['status'] != $chequeEntry->status) {
                $this->handleStatusChange($chequeEntry, $data['status'], $data);
            }

            $chequeEntry->update([
                'cheque_number' => $data['cheque_number'] ?? $chequeEntry->cheque_number,
                'cheque_date' => $data['cheque_date'] ?? $chequeEntry->cheque_date,
                'type' => $data['type'] ?? $chequeEntry->type,
                'status' => $data['status'] ?? $chequeEntry->status,
                'amount' => $data['amount'] ?? $chequeEntry->amount,
                'payee_name' => $data['payee_name'] ?? $chequeEntry->payee_name,
                'description' => $data['description'] ?? $chequeEntry->description,
                'presented_date' => $data['presented_date'] ?? $chequeEntry->presented_date,
                'cleared_date' => $data['cleared_date'] ?? $chequeEntry->cleared_date,
                'bounced_date' => $data['bounced_date'] ?? $chequeEntry->bounced_date,
                'bounce_reason' => $data['bounce_reason'] ?? $chequeEntry->bounce_reason,
            ]);

            DB::commit();
            return $chequeEntry->fresh(['bankAccount', 'chequeBook', 'accountTransaction']);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('ChequeEntryService::updateChequeEntry - Error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Delete cheque entry
     */
    public function deleteChequeEntry(ChequeBookEntry $chequeEntry): bool
    {
        // Only allow deletion of pending cheques
        if ($chequeEntry->status !== 'pending') {
            throw new \Exception('Cannot delete cheque entry that is not pending.');
        }

        return $chequeEntry->delete();
    }

    /**
     * Mark cheque as cleared
     */
    public function markAsCleared(ChequeBookEntry $chequeEntry, ?Carbon $clearedDate = null): ChequeBookEntry
    {
        $clearedDate = $clearedDate ?? Carbon::now();
        
        $chequeEntry->update([
            'status' => 'cleared',
            'cleared_date' => $clearedDate,
        ]);

        return $chequeEntry->fresh();
    }

    /**
     * Mark cheque as bounced
     */
    public function markAsBounced(ChequeBookEntry $chequeEntry, string $reason, ?Carbon $bouncedDate = null): ChequeBookEntry
    {
        $bouncedDate = $bouncedDate ?? Carbon::now();
        
        $chequeEntry->update([
            'status' => 'bounced',
            'bounced_date' => $bouncedDate,
            'bounce_reason' => $reason,
        ]);

        return $chequeEntry->fresh();
    }

    /**
     * Get available cheque numbers from a cheque book
     */
    public function getAvailableChequeNumbers(int $chequeBookId): array
    {
        $chequeBook = ChequeBook::findOrFail($chequeBookId);
        
        $usedNumbers = ChequeBookEntry::where('cheque_book_id', $chequeBookId)
            ->where('status', '!=', 'cancelled')
            ->pluck('cheque_number')
            ->toArray();

        $allNumbers = range($chequeBook->start_number, $chequeBook->end_number);
        $available = array_diff($allNumbers, $usedNumbers);

        return array_values($available);
    }

    /**
     * Get cheque entries for a bank account
     */
    public function getChequeEntries(int $bankAccountId, array $filters = []): \Illuminate\Database\Eloquent\Collection
    {
        $query = ChequeBookEntry::where('bank_account_id', $bankAccountId)
            ->with(['bankAccount', 'chequeBook', 'accountTransaction']);

        if (isset($filters['type'])) {
            $query->where('type', $filters['type']);
        }

        if (isset($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (isset($filters['start_date'])) {
            $query->whereDate('cheque_date', '>=', $filters['start_date']);
        }

        if (isset($filters['end_date'])) {
            $query->whereDate('cheque_date', '<=', $filters['end_date']);
        }

        return $query->orderBy('cheque_date', 'desc')
            ->orderBy('cheque_number', 'asc')
            ->get();
    }

    /**
     * Get cheque summary for a bank account
     */
    public function getChequeSummary(int $bankAccountId, ?Carbon $asOfDate = null): array
    {
        $asOfDate = $asOfDate ?? Carbon::now();
        
        $query = ChequeBookEntry::where('bank_account_id', $bankAccountId)
            ->whereDate('cheque_date', '<=', $asOfDate);

        $summary = [
            'total_issued' => 0,
            'total_received' => 0,
            'pending_issued' => 0,
            'pending_received' => 0,
            'cleared_issued' => 0,
            'cleared_received' => 0,
            'bounced_issued' => 0,
            'bounced_received' => 0,
        ];

        $entries = $query->get();

        foreach ($entries as $entry) {
            $type = $entry->type;
            $status = $entry->status;
            $amount = (float) $entry->amount;

            if ($type === 'issued') {
                $summary['total_issued'] += $amount;
                if ($status === 'pending') {
                    $summary['pending_issued'] += $amount;
                } elseif ($status === 'cleared') {
                    $summary['cleared_issued'] += $amount;
                } elseif ($status === 'bounced') {
                    $summary['bounced_issued'] += $amount;
                }
            } else {
                $summary['total_received'] += $amount;
                if ($status === 'pending') {
                    $summary['pending_received'] += $amount;
                } elseif ($status === 'cleared') {
                    $summary['cleared_received'] += $amount;
                } elseif ($status === 'bounced') {
                    $summary['bounced_received'] += $amount;
                }
            }
        }

        return $summary;
    }

    /**
     * Validate cheque number uniqueness
     */
    protected function validateChequeNumber(int $bankAccountId, string $chequeNumber, ?int $excludeId = null): void
    {
        $query = ChequeBookEntry::where('bank_account_id', $bankAccountId)
            ->where('cheque_number', $chequeNumber);

        if ($excludeId) {
            $query->where('id', '!=', $excludeId);
        }

        if ($query->exists()) {
            throw new \Exception('Cheque number already exists for this bank account.');
        }
    }

    /**
     * Handle status change
     */
    protected function handleStatusChange(ChequeBookEntry $chequeEntry, string $newStatus, array $data): void
    {
        switch ($newStatus) {
            case 'cleared':
                $chequeEntry->cleared_date = $data['cleared_date'] ?? Carbon::now();
                break;
            case 'bounced':
                $chequeEntry->bounced_date = $data['bounced_date'] ?? Carbon::now();
                $chequeEntry->bounce_reason = $data['bounce_reason'] ?? 'Not specified';
                break;
            case 'cancelled':
                // Can only cancel pending cheques
                if ($chequeEntry->status !== 'pending') {
                    throw new \Exception('Can only cancel pending cheques.');
                }
                break;
        }
    }

    /**
     * Link cheque entry to account transaction
     */
    protected function linkToAccountTransaction(ChequeBookEntry $chequeEntry, int $accountTransactionId): void
    {
        $accountTransaction = AccountTransaction::findOrFail($accountTransactionId);
        
        $chequeEntry->update([
            'account_transaction_id' => $accountTransactionId,
        ]);
    }
}

