<?php

namespace Modules\AccountingReports\Services;

use Modules\AccountingReports\Entities\BankAccount;
use Modules\AccountingReports\Entities\BankReconciliation;
use App\AccountTransaction;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class BankReconciliationService
{
    /**
     * Get unreconciled transactions for a bank account
     */
    public function getUnreconciledTransactions(BankAccount $bankAccount, ?Carbon $asOfDate = null): array
    {
        if (!$bankAccount->linked_account_id) {
            return [];
        }

        $asOfDate = $asOfDate ?? Carbon::now();

        // Get all transactions for the account
        $transactions = AccountTransaction::where('account_id', $bankAccount->linked_account_id)
            ->whereDate('operation_date', '<=', $asOfDate)
            ->where('id', '!=', $bankAccount->opening_balance_transaction_id) // Exclude opening balance
            ->with(['transaction', 'transaction.location'])
            ->orderBy('operation_date', 'asc')
            ->orderBy('id', 'asc')
            ->get();

        // Get already reconciled transaction IDs
        $reconciledIds = BankReconciliation::where('account_id', $bankAccount->linked_account_id)
            ->where('is_cleared', true)
            ->whereNotNull('account_transaction_id')
            ->pluck('account_transaction_id')
            ->toArray();

        $unreconciled = [];
        foreach ($transactions as $transaction) {
            // Skip if already reconciled
            if (in_array($transaction->id, $reconciledIds)) {
                continue;
            }

            $unreconciled[] = [
                'id' => $transaction->id,
                'date' => $transaction->operation_date,
                'voucher_no' => $transaction->reff_no ?? 'TXN-' . $transaction->id,
                'description' => $transaction->note ?? '',
                'amount' => (float) $transaction->amount,
                'type' => $transaction->type,
                'transaction' => $transaction->transaction,
                'location' => $transaction->transaction->location ?? null,
            ];
        }

        return $unreconciled;
    }

    /**
     * Reconcile transactions
     */
    public function reconcileTransactions(BankAccount $bankAccount, array $transactionIds, Carbon $clearedDate, ?string $bankStatementRef = null): array
    {
        if (!$bankAccount->linked_account_id) {
            throw new \Exception('Bank account must be linked to an account for reconciliation.');
        }

        DB::beginTransaction();
        
        try {
            $reconciled = [];
            
            foreach ($transactionIds as $transactionId) {
                $accountTransaction = AccountTransaction::findOrFail($transactionId);
                
                // Check if already reconciled
                $existing = BankReconciliation::where('account_transaction_id', $transactionId)
                    ->where('account_id', $bankAccount->linked_account_id)
                    ->first();

                if ($existing) {
                    // Update existing reconciliation
                    $existing->update([
                        'is_cleared' => true,
                        'cleared_date' => $clearedDate,
                        'cleared_by' => auth()->id(),
                        'bank_statement_ref' => $bankStatementRef,
                        'bank_statement_date' => $clearedDate,
                    ]);
                    $reconciled[] = $existing;
                } else {
                    // Create new reconciliation
                    $reconciliation = BankReconciliation::create([
                        'business_id' => $bankAccount->business_id,
                        'account_id' => $bankAccount->linked_account_id,
                        'account_transaction_id' => $transactionId,
                        'journal_entry_line_id' => null, // Keep for backward compatibility
                        'transaction_date' => $accountTransaction->operation_date,
                        'amount' => $accountTransaction->amount,
                        'reference' => $accountTransaction->reff_no ?? 'TXN-' . $transactionId,
                        'description' => $accountTransaction->note ?? '',
                        'is_cleared' => true,
                        'cleared_date' => $clearedDate,
                        'cleared_by' => auth()->id(),
                        'bank_statement_ref' => $bankStatementRef,
                        'bank_statement_date' => $clearedDate,
                    ]);
                    $reconciled[] = $reconciliation;
                }
            }

            DB::commit();
            return $reconciled;
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('BankReconciliationService::reconcileTransactions - Error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Unreconcile transactions
     */
    public function unreconcileTransactions(BankAccount $bankAccount, array $transactionIds): bool
    {
        DB::beginTransaction();
        
        try {
            BankReconciliation::where('account_id', $bankAccount->linked_account_id)
                ->whereIn('account_transaction_id', $transactionIds)
                ->update([
                    'is_cleared' => false,
                    'cleared_date' => null,
                    'cleared_by' => null,
                ]);

            DB::commit();
            return true;
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('BankReconciliationService::unreconcileTransactions - Error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get reconciliation statement
     */
    public function getReconciliationStatement(BankAccount $bankAccount, Carbon $startDate, Carbon $endDate): array
    {
        if (!$bankAccount->linked_account_id) {
            return [
                'opening_balance' => $bankAccount->opening_balance,
                'book_balance' => $bankAccount->opening_balance,
                'cleared_balance' => $bankAccount->opening_balance,
                'outstanding_debits' => 0,
                'outstanding_credits' => 0,
                'transactions' => [],
            ];
        }

        // Get book balance (all transactions)
        $bookBalance = AccountTransaction::where('account_id', $bankAccount->linked_account_id)
            ->whereDate('operation_date', '<=', $endDate)
            ->selectRaw('SUM(IF(type="credit", amount, -1 * amount)) as balance')
            ->value('balance') ?? 0;

        // Get cleared transactions
        $clearedTransactions = BankReconciliation::where('account_id', $bankAccount->linked_account_id)
            ->where('is_cleared', true)
            ->whereBetween('transaction_date', [$startDate, $endDate])
            ->with('journalEntryLine')
            ->get();

        $clearedBalance = $bankAccount->opening_balance;
        $outstandingDebits = 0;
        $outstandingCredits = 0;

        foreach ($clearedTransactions as $reconciliation) {
            $amount = (float) $reconciliation->amount;
            if ($reconciliation->journalEntryLine && $reconciliation->journalEntryLine->type === 'debit') {
                $clearedBalance += $amount;
                $outstandingDebits += $amount;
            } else {
                $clearedBalance -= $amount;
                $outstandingCredits += $amount;
            }
        }

        // Get outstanding (unreconciled) transactions
        $outstanding = $this->getUnreconciledTransactions($bankAccount, $endDate);

        return [
            'opening_balance' => $bankAccount->opening_balance,
            'book_balance' => (float) $bookBalance,
            'cleared_balance' => $clearedBalance,
            'outstanding_debits' => $outstandingDebits,
            'outstanding_credits' => $outstandingCredits,
            'outstanding_count' => count($outstanding),
            'transactions' => $outstanding,
        ];
    }

    /**
     * Get reconciliation summary
     */
    public function getReconciliationSummary(BankAccount $bankAccount, ?Carbon $asOfDate = null): array
    {
        $asOfDate = $asOfDate ?? Carbon::now();

        $unreconciled = $this->getUnreconciledTransactions($bankAccount, $asOfDate);
        
        $totalUnreconciled = 0;
        $unreconciledDebits = 0;
        $unreconciledCredits = 0;

        foreach ($unreconciled as $transaction) {
            $amount = $transaction['amount'];
            $totalUnreconciled += $amount;
            
            if ($transaction['type'] === 'debit') {
                $unreconciledDebits += $amount;
            } else {
                $unreconciledCredits += $amount;
            }
        }

        return [
            'total_unreconciled' => count($unreconciled),
            'unreconciled_amount' => $totalUnreconciled,
            'unreconciled_debits' => $unreconciledDebits,
            'unreconciled_credits' => $unreconciledCredits,
            'last_reconciled_date' => $this->getLastReconciledDate($bankAccount),
        ];
    }

    /**
     * Get last reconciled date
     */
    protected function getLastReconciledDate(BankAccount $bankAccount): ?Carbon
    {
        if (!$bankAccount->linked_account_id) {
            return null;
        }

        $lastReconciliation = BankReconciliation::where('account_id', $bankAccount->linked_account_id)
            ->where('is_cleared', true)
            ->orderBy('cleared_date', 'desc')
            ->first();

        return $lastReconciliation ? Carbon::parse($lastReconciliation->cleared_date) : null;
    }

    /**
     * Import bank statement
     */
    public function importBankStatement(BankAccount $bankAccount, array $statements): array
    {
        DB::beginTransaction();
        
        try {
            $imported = [];
            $errors = [];

            foreach ($statements as $statement) {
                try {
                    // Find matching transaction
                    $transaction = AccountTransaction::where('account_id', $bankAccount->linked_account_id)
                        ->where('reff_no', $statement['reference'] ?? null)
                        ->whereDate('operation_date', Carbon::parse($statement['date']))
                        ->where('amount', $statement['amount'])
                        ->first();

                    if ($transaction) {
                        $reconciliation = $this->reconcileTransactions(
                            $bankAccount,
                            [$transaction->id],
                            Carbon::parse($statement['date']),
                            $statement['reference'] ?? null
                        );
                        $imported[] = $reconciliation[0];
                    } else {
                        $errors[] = 'Transaction not found: ' . ($statement['reference'] ?? 'N/A');
                    }
                } catch (\Exception $e) {
                    $errors[] = 'Error importing statement: ' . $e->getMessage();
                }
            }

            DB::commit();
            
            return [
                'imported' => $imported,
                'errors' => $errors,
                'total' => count($statements),
                'success' => count($imported),
                'failed' => count($errors),
            ];
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('BankReconciliationService::importBankStatement - Error: ' . $e->getMessage());
            throw $e;
        }
    }
}

