<?php

namespace Modules\AccountingReports\Services;

use Modules\AccountingReports\Entities\BankAccount;
use App\Account;
use App\AccountTransaction;
use App\AccountType;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class BankAccountService
{
    /**
     * Create a new bank account with opening balance transaction
     */
    public function createBankAccount(array $data, int $businessId, int $userId): BankAccount
    {
        DB::beginTransaction();
        
        try {
            // Create the bank account
            $bankAccount = BankAccount::create([
                'business_id' => $businessId,
                'account_name' => $data['account_name'],
                'account_type' => $data['account_type'] ?? 'asset',
                'bank_name' => $data['bank_name'] ?? null,
                'account_number' => $data['account_number'] ?? null,
                'ifsc_code' => $data['ifsc_code'] ?? null,
                'branch_name' => $data['branch_name'] ?? null,
                'description' => $data['description'] ?? null,
                'linked_account_id' => $data['linked_account_id'] ?? null,
                'opening_balance' => $data['opening_balance'] ?? 0,
                'opening_date' => $data['opening_date'] ?? Carbon::now(),
                'is_active' => $data['is_active'] ?? true,
                'created_by' => $userId,
            ]);

            // Create opening balance transaction if amount is not zero
            if (isset($data['opening_balance']) && $data['opening_balance'] != 0 && $bankAccount->linked_account_id) {
                $this->createOpeningBalanceTransaction($bankAccount, $data['opening_balance'], $data['opening_date'] ?? Carbon::now());
            }

            DB::commit();
            return $bankAccount->fresh(['linkedAccount', 'creator']);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('BankAccountService::createBankAccount - Error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Update bank account
     */
    public function updateBankAccount(BankAccount $bankAccount, array $data, int $businessId): BankAccount
    {
        DB::beginTransaction();
        
        try {
            // Check if opening balance changed
            $openingBalanceChanged = isset($data['opening_balance']) && 
                                    $data['opening_balance'] != $bankAccount->opening_balance;
            
            $bankAccount->update([
                'account_name' => $data['account_name'] ?? $bankAccount->account_name,
                'account_type' => $data['account_type'] ?? $bankAccount->account_type,
                'bank_name' => $data['bank_name'] ?? $bankAccount->bank_name,
                'account_number' => $data['account_number'] ?? $bankAccount->account_number,
                'ifsc_code' => $data['ifsc_code'] ?? $bankAccount->ifsc_code,
                'branch_name' => $data['branch_name'] ?? $bankAccount->branch_name,
                'description' => $data['description'] ?? $bankAccount->description,
                'linked_account_id' => $data['linked_account_id'] ?? $bankAccount->linked_account_id,
                'opening_balance' => $data['opening_balance'] ?? $bankAccount->opening_balance,
                'opening_date' => $data['opening_date'] ?? $bankAccount->opening_date,
                'is_active' => $data['is_active'] ?? $bankAccount->is_active,
            ]);

            // Update opening balance transaction if changed
            if ($openingBalanceChanged && $bankAccount->linked_account_id) {
                $this->updateOpeningBalanceTransaction($bankAccount, $data['opening_balance'], $data['opening_date'] ?? $bankAccount->opening_date);
            }

            DB::commit();
            return $bankAccount->fresh(['linkedAccount', 'creator']);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('BankAccountService::updateBankAccount - Error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Delete bank account (soft delete)
     */
    public function deleteBankAccount(BankAccount $bankAccount): bool
    {
        // Check if account has transactions
        if ($bankAccount->linked_account_id) {
            $hasTransactions = AccountTransaction::where('account_id', $bankAccount->linked_account_id)
                ->where('id', '!=', $bankAccount->opening_balance_transaction_id)
                ->exists();
            
            if ($hasTransactions) {
                throw new \Exception('Cannot delete bank account with existing transactions. Please deactivate instead.');
            }
        }

        return $bankAccount->delete();
    }

    /**
     * Get bank account balance
     */
    public function getBalance(BankAccount $bankAccount, ?Carbon $asOfDate = null): float
    {
        if (!$bankAccount->linked_account_id) {
            return $bankAccount->opening_balance;
        }

        $asOfDate = $asOfDate ?? Carbon::now();
        
        $balance = AccountTransaction::where('account_id', $bankAccount->linked_account_id)
            ->whereDate('operation_date', '<=', $asOfDate)
            ->selectRaw('SUM(IF(type="credit", amount, -1 * amount)) as balance')
            ->value('balance') ?? 0;

        return (float) $balance;
    }

    /**
     * Get bank account statement
     */
    public function getStatement(BankAccount $bankAccount, Carbon $startDate, Carbon $endDate, ?int $locationId = null): array
    {
        if (!$bankAccount->linked_account_id) {
            return [
                'transactions' => [],
                'opening_balance' => $bankAccount->opening_balance,
                'closing_balance' => $bankAccount->opening_balance,
                'total_debit' => 0,
                'total_credit' => 0,
            ];
        }

        // Get opening balance
        $openingBalance = $this->getBalance($bankAccount, $startDate->copy()->subDay());

        // Get transactions
        $query = AccountTransaction::where('account_id', $bankAccount->linked_account_id)
            ->whereBetween('operation_date', [$startDate, $endDate])
            ->with(['transaction', 'transaction.location'])
            ->orderBy('operation_date', 'asc')
            ->orderBy('id', 'asc');

        if ($locationId) {
            $query->whereHas('transaction', function($q) use ($locationId) {
                $q->where('location_id', $locationId);
            });
        }

        $transactions = $query->get();

        $runningBalance = $openingBalance;
        $totalDebit = 0;
        $totalCredit = 0;

        $statement = [];
        foreach ($transactions as $transaction) {
            $amount = (float) $transaction->amount;
            if ($transaction->type === 'debit') {
                $runningBalance += $amount;
                $totalDebit += $amount;
            } else {
                $runningBalance -= $amount;
                $totalCredit += $amount;
            }

            $statement[] = [
                'id' => $transaction->id,
                'date' => $transaction->operation_date,
                'voucher_no' => $transaction->reff_no ?? 'TXN-' . $transaction->id,
                'description' => $transaction->note ?? '',
                'debit' => $transaction->type === 'debit' ? $amount : 0,
                'credit' => $transaction->type === 'credit' ? $amount : 0,
                'balance' => $runningBalance,
                'transaction' => $transaction->transaction,
                'location' => $transaction->transaction->location ?? null,
            ];
        }

        return [
            'transactions' => $statement,
            'opening_balance' => $openingBalance,
            'closing_balance' => $runningBalance,
            'total_debit' => $totalDebit,
            'total_credit' => $totalCredit,
        ];
    }

    /**
     * Create opening balance transaction
     */
    protected function createOpeningBalanceTransaction(BankAccount $bankAccount, float $amount, Carbon $date): AccountTransaction
    {
        // Delete existing opening balance transaction if any
        if ($bankAccount->opening_balance_transaction_id) {
            AccountTransaction::where('id', $bankAccount->opening_balance_transaction_id)->delete();
        }

        $transactionType = $bankAccount->account_type === 'liability' ? 'credit' : 'debit';
        
        $accountTransaction = AccountTransaction::create([
            'account_id' => $bankAccount->linked_account_id,
            'type' => $transactionType,
            'amount' => abs($amount),
            'operation_date' => $date,
            'note' => 'Opening Balance - ' . $bankAccount->account_name,
            'reff_no' => 'OB-' . $bankAccount->id,
        ]);

        $bankAccount->update([
            'opening_balance_transaction_id' => $accountTransaction->id
        ]);

        return $accountTransaction;
    }

    /**
     * Update opening balance transaction
     */
    protected function updateOpeningBalanceTransaction(BankAccount $bankAccount, float $amount, Carbon $date): AccountTransaction
    {
        if ($bankAccount->opening_balance_transaction_id) {
            $transaction = AccountTransaction::find($bankAccount->opening_balance_transaction_id);
            if ($transaction) {
                $transactionType = $bankAccount->account_type === 'liability' ? 'credit' : 'debit';
                $transaction->update([
                    'amount' => abs($amount),
                    'operation_date' => $date,
                ]);
                return $transaction;
            }
        }

        return $this->createOpeningBalanceTransaction($bankAccount, $amount, $date);
    }

    /**
     * Get available accounts for linking
     */
    public function getAvailableAccounts(int $businessId): array
    {
        // Get bank account type
        $bankAccountType = AccountType::where('name', 'like', '%bank%')
            ->orWhere('name', 'like', '%current%')
            ->orWhere('name', 'like', '%savings%')
            ->first();

        $accounts = Account::where('business_id', $businessId)
            ->where('is_closed', 0)
            ->when($bankAccountType, function($query) use ($bankAccountType) {
                $query->where('account_type_id', $bankAccountType->id);
            })
            ->orderBy('name', 'asc')
            ->get();

        return $accounts->pluck('name', 'id')->toArray();
    }

    /**
     * Get bank accounts summary
     */
    public function getSummary(int $businessId, ?Carbon $asOfDate = null): array
    {
        $asOfDate = $asOfDate ?? Carbon::now();
        
        $bankAccounts = BankAccount::where('business_id', $businessId)
            ->where('is_active', true)
            ->with('linkedAccount')
            ->get();

        $summary = [
            'total_accounts' => $bankAccounts->count(),
            'total_balance' => 0,
            'accounts' => [],
        ];

        foreach ($bankAccounts as $account) {
            $balance = $this->getBalance($account, $asOfDate);
            $summary['total_balance'] += $balance;
            
            $summary['accounts'][] = [
                'id' => $account->id,
                'name' => $account->account_name,
                'bank_name' => $account->bank_name,
                'account_number' => $account->account_number,
                'balance' => $balance,
                'account_type' => $account->account_type,
            ];
        }

        return $summary;
    }
}

