<?php

namespace Modules\AccountingReports\Services;

use App\Utils\TransactionUtil;
use App\Utils\BusinessUtil;
use App\Account;
use App\AccountTransaction;
use App\BusinessLocation;
use Modules\AccountingReports\Services\TrialBalanceService;
use Modules\AccountingReports\Entities\Loan;
use Modules\AccountingReports\Entities\FixedAsset;
use Modules\AccountingReports\Entities\CapitalAccount;
use DB;

class BalanceSheetService
{
    protected $transactionUtil;
    protected $businessUtil;
    protected $trialBalanceService;

    public function __construct(TransactionUtil $transactionUtil, BusinessUtil $businessUtil, TrialBalanceService $trialBalanceService)
    {
        $this->transactionUtil = $transactionUtil;
        $this->businessUtil = $businessUtil;
        $this->trialBalanceService = $trialBalanceService;
    }

    /**
     * Generate Balance Sheet data as of a specific date
     */
    public function getBalanceSheet($businessId, $asOfDate, $locationId = null)
    {
        try {
            $permitted_locations = auth()->user()->permitted_locations();
            
            // Get trial balance data
            $trialBalance = $this->trialBalanceService->getTrialBalance(
                $businessId, 
                $asOfDate, 
                $asOfDate, 
                $locationId
            );

            // Get account balances
            $accountBalances = $this->getAccountBalances($businessId, $asOfDate, $locationId);

            // Get closing stock
            $plData = $this->transactionUtil->getProfitLossDetails(
                $businessId, 
                $locationId, 
                $asOfDate, 
                $asOfDate, 
                null, 
                $permitted_locations
            );
            $closingStock = (float) ($plData['closing_stock'] ?? 0);

            // Get P&L for current period and opening balance
            $fy = $this->businessUtil->getCurrentFinancialYear($businessId);
            $plCurrentPeriod = $this->getCurrentPeriodProfitLoss($businessId, $fy['start'], $asOfDate, $locationId);
            $plOpening = $this->getOpeningProfitLoss($businessId, $fy['start'], $locationId);

            // Build Liabilities (Left Side)
            $liabilities = $this->buildLiabilities($accountBalances, $trialBalance, $businessId, $locationId, $asOfDate);

            // Build Assets (Right Side)
            $assets = $this->buildAssets($accountBalances, $trialBalance, $closingStock, $businessId, $locationId, $plOpening, $plCurrentPeriod);

            // Calculate totals
            $totalLiabilities = $this->calculateTotal($liabilities);
            $totalAssets = $this->calculateTotal($assets);

            // Calculate difference (for balancing)
            $difference = abs($totalLiabilities - $totalAssets);

            return [
                'liabilities' => $liabilities,
                'assets' => $assets,
                'total_liabilities' => $totalLiabilities,
                'total_assets' => $totalAssets,
                'difference' => $difference,
                'is_balanced' => abs($totalLiabilities - $totalAssets) < 0.01,
                'as_of_date' => $asOfDate
            ];
        } catch (\Exception $e) {
            \Log::error('BalanceSheetService getBalanceSheet Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            throw $e;
        }
    }

    /**
     * Build Liabilities section
     */
    protected function buildLiabilities($accountBalances, $trialBalance, $businessId, $locationId, $asOfDate = null)
    {
        $liabilities = [];
        $activeLoans = collect([]); // Initialize to avoid undefined variable

        // 1. Capital Account
        $capitalAccounts = [];
        $drawingsAccounts = [];
        $capitalTotal = 0;
        $drawingsTotal = 0;
        
        // Get capital accounts from Capital Account Management module
        $managedCapitalAccounts = CapitalAccount::where('business_id', $businessId)
            ->where('is_active', 1)
            ->get();
        
        foreach ($managedCapitalAccounts as $capitalAccount) {
            // Use current balance for balance sheet
            $currentBalance = (float) ($capitalAccount->current_balance ?? $capitalAccount->opening_balance ?? 0);
            
            if (abs($currentBalance) > 0.01) {
                $capitalAccounts[] = [
                    'name' => $capitalAccount->account_name . ' (' . $capitalAccount->account_type_label . ')',
                    'value' => abs($currentBalance),
                    'from_module' => true
                ];
                $capitalTotal += abs($currentBalance);
            }
        }
        
        // Also include capital accounts from account balances (for backward compatibility)
        // But exclude those that might already be in Capital Account module
        $managedAccountNames = $managedCapitalAccounts->pluck('account_name')->map(function($name) {
            return strtolower($name);
        })->toArray();
        
        foreach ($accountBalances as $accountId => $account) {
            $name = strtolower($account['name']);
            
            // Skip if this account name matches a managed capital account
            $isManagedAccount = false;
            foreach ($managedAccountNames as $managedName) {
                if (strpos($name, $managedName) !== false || strpos($managedName, $name) !== false) {
                    $isManagedAccount = true;
                    break;
                }
            }
            
            if (!$isManagedAccount) {
                $balance = (float) $account['balance'];
                
                // Check if it's a drawings account (negative equity)
                $isDrawings = strpos($name, 'drawing') !== false || 
                              strpos($name, 'drawings') !== false ||
                              (strtolower($account['type']) == 'equity' && $balance < 0);
                
                if ($isDrawings && abs($balance) > 0.01) {
                    // Drawings should be negative
                    $drawingsAccounts[] = [
                        'name' => $account['name'],
                        'value' => $balance, // Keep negative value
                        'from_module' => false
                    ];
                    $drawingsTotal += $balance; // This will be negative
                } elseif (!$isDrawings &&
                    (strtolower($account['type']) == 'equity' ||
                     strpos($name, 'capital') !== false ||
                     strpos($name, 'equity') !== false)
                ) {
                    if (abs($balance) > 0.01) {
                        $capitalAccounts[] = [
                            'name' => $account['name'],
                            'value' => abs($balance),
                            'from_module' => false
                        ];
                        $capitalTotal += abs($balance);
                    }
                }
            }
        }
        
        // Calculate net capital (capital - drawings)
        $netCapitalTotal = $capitalTotal + $drawingsTotal; // drawingsTotal is negative, so this subtracts
        
        if (!empty($capitalAccounts) || !empty($drawingsAccounts) || $netCapitalTotal > 0) {
            // Create sub-groups for Capital Account
            $subGroups = [];
            
            if (!empty($drawingsAccounts)) {
                $subGroups['Drawings of Owner'] = $drawingsAccounts;
            }
            
            if (!empty($capitalAccounts)) {
                // If there are drawings, group capital accounts under "Owner's Capital"
                // Otherwise, they can be direct items
                if (!empty($drawingsAccounts)) {
                    $subGroups["Owner's Capital"] = $capitalAccounts;
                }
            }
            
            $liabilities[] = [
                'label' => 'Capital Account',
                'items' => empty($subGroups) ? $capitalAccounts : [],
                'sub_groups' => $subGroups,
                'total' => $netCapitalTotal,
                'is_group' => true
            ];
        }

        // 2. Loans (Liability) - Get from Loan Management module
        $loans = [];
        $securedLoans = [];
        $unsecuredLoans = [];
        $loansTotal = 0;
        
        // First, get loans from Loan Management module
        $loanQuery = Loan::where('business_id', $businessId)
            ->where('status', '!=', 'closed')
            ->with('liabilityAccount');
            
        if ($locationId) {
            $loanQuery->where('location_id', $locationId);
        }
        
        $activeLoans = $loanQuery->get();
        
        foreach ($activeLoans as $loan) {
            if ($loan->liabilityAccount) {
                // Use outstanding principal from loan record (this is the authoritative source)
                // Also get account balance to verify
                $loanAccountBalance = $this->getAccountBalanceById($loan->liabilityAccount->id, $asOfDate);
                
                // Use the maximum of outstanding_principal and account balance to ensure we capture the loan
                // Outstanding principal is the primary source as it's maintained by the loan management system
                $balance = max(abs($loan->outstanding_principal), abs($loanAccountBalance));
                
                if ($balance > 0.01) {
                    $loanName = $loan->loan_name . ' (' . $loan->loan_number . ')';
                    
                    // Check if secured or unsecured based on loan name or other criteria
                    $name = strtolower($loanName);
                    if (strpos($name, 'secured') !== false || strpos($name, 'mortgage') !== false) {
                        $securedLoans[] = [
                            'name' => $loanName,
                            'value' => $balance
                        ];
                    } else {
                        $unsecuredLoans[] = [
                            'name' => $loanName,
                            'value' => $balance
                        ];
                    }
                    $loansTotal += $balance;
                }
            }
        }
        
        // Also include loan accounts from account balances that might not be in Loan Management
        foreach ($accountBalances as $accountId => $account) {
            $name = strtolower($account['name']);
            // Skip if already included from Loan Management
            $alreadyIncluded = false;
            foreach ($activeLoans as $loan) {
                if ($loan->liabilityAccount && $loan->liabilityAccount->id == $accountId) {
                    $alreadyIncluded = true;
                    break;
                }
            }
            
            if (!$alreadyIncluded &&
                strtolower($account['type']) == 'liability' &&
                (strpos($name, 'loan') !== false || 
                 strpos($name, 'borrowing') !== false ||
                 strpos($name, 'debt') !== false) &&
                $account['balance'] > 0
            ) {
                $balance = abs($account['balance']);
                if (strpos($name, 'secured') !== false || strpos($name, 'mortgage') !== false) {
                    $securedLoans[] = [
                        'name' => $account['name'],
                        'value' => $balance
                    ];
                } else {
                    $unsecuredLoans[] = [
                        'name' => $account['name'],
                        'value' => $balance
                    ];
                }
                $loansTotal += $balance;
            }
        }

        if ($loansTotal > 0) {
            // Only include items in sub_groups, not in main items array to avoid duplication
            $liabilities[] = [
                'label' => 'Loans (Liability)',
                'items' => [], // Empty - all items are in sub_groups
                'sub_groups' => [
                    'Secured Loans' => $securedLoans,
                    'Unsecured Loans' => $unsecuredLoans
                ],
                'total' => $loansTotal,
                'is_group' => true
            ];
        }

        // 3. Current Liabilities
        $currentLiabilities = [];
        $dutiesTaxes = [];
        $provisions = [];
        $sundryCreditors = [];
        $currentLiabilitiesTotal = 0;

        // Supplier due (Sundry Creditors)
        $supplierDue = abs($trialBalance['supplier_due'] ?? 0);
        if ($supplierDue > 0.01) {
            $sundryCreditors[] = [
                'name' => 'Sundry Creditors',
                'value' => $supplierDue
            ];
            $currentLiabilitiesTotal += $supplierDue;
        }

        // Other current liability accounts
        foreach ($accountBalances as $accountId => $account) {
            $name = strtolower($account['name']);
            // Skip if this is a loan account already included
            $isLoanAccount = false;
            if (isset($activeLoans)) {
                foreach ($activeLoans as $loan) {
                    if ($loan->liabilityAccount && $loan->liabilityAccount->id == $accountId) {
                        $isLoanAccount = true;
                        break;
                    }
                }
            }
            
            if (!$isLoanAccount &&
                strtolower($account['type']) == 'liability' &&
                $account['balance'] > 0 &&
                strpos($name, 'loan') === false &&
                strpos($name, 'borrowing') === false
            ) {
                $balance = abs($account['balance']);
                if (strpos($name, 'tax') !== false || strpos($name, 'duty') !== false || strpos($name, 'gst') !== false) {
                    $dutiesTaxes[] = [
                        'name' => $account['name'],
                        'value' => $balance
                    ];
                } elseif (strpos($name, 'provision') !== false) {
                    $provisions[] = [
                        'name' => $account['name'],
                        'value' => $balance
                    ];
                } else {
                    $sundryCreditors[] = [
                        'name' => $account['name'],
                        'value' => $balance
                    ];
                }
                $currentLiabilitiesTotal += $balance;
            }
        }

        if ($currentLiabilitiesTotal > 0) {
            // Only include items in sub_groups, not in main items array to avoid duplication
            $liabilities[] = [
                'label' => 'Current Liabilities',
                'items' => [], // Empty - all items are in sub_groups
                'sub_groups' => [
                    'Duties & Taxes' => $dutiesTaxes,
                    'Provisions' => $provisions,
                    'Sundry Creditors' => $sundryCreditors
                ],
                'total' => $currentLiabilitiesTotal,
                'is_group' => true
            ];
        }

        // 4. Branch / Divisions (if multi-location)
        $branches = [];
        $branchesTotal = 0;
        if ($locationId === null) {
            // Show all locations as branches
            $locations = BusinessLocation::where('business_id', $businessId)->get();
            foreach ($locations as $location) {
                // Get location-specific balances (simplified - you may want to enhance this)
                $locationBalance = 0; // Placeholder - calculate location-specific balance
                if (abs($locationBalance) > 0.01) {
                    $branches[] = [
                        'name' => $location->name,
                        'value' => abs($locationBalance)
                    ];
                    $branchesTotal += abs($locationBalance);
                }
            }
        }

        if ($branchesTotal > 0) {
            $liabilities[] = [
                'label' => 'Branch / Divisions',
                'items' => $branches,
                'total' => $branchesTotal,
                'is_group' => true
            ];
        }

        return $liabilities;
    }

    /**
     * Build Assets section
     */
    protected function buildAssets($accountBalances, $trialBalance, $closingStock, $businessId, $locationId, $plOpening = 0, $plCurrentPeriod = 0)
    {
        $assets = [];

        // 1. Fixed Assets
        $fixedAssets = [];
        $fixedAssetsTotal = 0;
        
        // Get fixed assets from Fixed Assets Management module
        $managedFixedAssets = FixedAsset::where('business_id', $businessId)
            ->where('is_active', 1)
            ->get();
        
        foreach ($managedFixedAssets as $asset) {
            // Use net book value (current book value - accumulated depreciation) for balance sheet
            $netBookValue = $asset->net_book_value;
            
            if ($netBookValue > 0.01) {
                $depreciationRate = null;
                if (!empty($asset->depreciation_percentage)) {
                    $depreciationRate = number_format($asset->depreciation_percentage, 2) . '%';
                }
                
                $fixedAssets[] = [
                    'name' => $asset->name,
                    'value' => $netBookValue,
                    'depreciation_rate' => $depreciationRate,
                    'from_module' => true // Flag to indicate it's from Fixed Assets module
                ];
                $fixedAssetsTotal += $netBookValue;
            }
        }
        
        // Also include fixed assets from account balances (for backward compatibility)
        // But exclude those that might already be in Fixed Assets module
        $managedAssetNames = $managedFixedAssets->pluck('name')->map(function($name) {
            return strtolower($name);
        })->toArray();
        
        foreach ($accountBalances as $accountId => $account) {
            $name = strtolower($account['name']);
            
            // Skip if this account name matches a managed fixed asset
            $isManagedAsset = false;
            foreach ($managedAssetNames as $managedName) {
                if (strpos($name, $managedName) !== false || strpos($managedName, $name) !== false) {
                    $isManagedAsset = true;
                    break;
                }
            }
            
            if (!$isManagedAsset &&
                (strpos($name, 'fixed asset') !== false ||
                 strpos($name, 'computer') !== false ||
                 strpos($name, 'furniture') !== false ||
                 strpos($name, 'land') !== false ||
                 strpos($name, 'building') !== false ||
                 strpos($name, 'machinery') !== false ||
                 strpos($name, 'equipment') !== false ||
                 strpos($name, 'vehicle') !== false) &&
                $account['balance'] > 0
            ) {
                $balance = abs($account['balance']);
                // Extract depreciation rate if in name (e.g., "Computer (18%)")
                $depreciationRate = null;
                if (preg_match('/\((\d+)%\)/', $account['name'], $matches)) {
                    $depreciationRate = $matches[1] . '%';
                }
                
                $fixedAssets[] = [
                    'name' => $account['name'],
                    'value' => $balance,
                    'depreciation_rate' => $depreciationRate,
                    'from_module' => false
                ];
                $fixedAssetsTotal += $balance;
            }
        }

        if ($fixedAssetsTotal > 0) {
            $assets[] = [
                'label' => 'Fixed Assets',
                'items' => $fixedAssets,
                'total' => $fixedAssetsTotal,
                'is_group' => true
            ];
        }

        // 2. Investments
        $investments = [];
        $investmentsTotal = 0;
        
        // Get investments from Investment Management module
        $managedInvestments = \Modules\AccountingReports\Entities\Investment::where('business_id', $businessId)
            ->where('is_active', 1)
            ->get();
        
        foreach ($managedInvestments as $investment) {
            // Use current value if available, otherwise use amount
            $value = $investment->balance_sheet_value;
            
            if ($value > 0.01) {
                $investments[] = [
                    'name' => $investment->name,
                    'value' => $value,
                    'from_module' => true // Flag to indicate it's from Investment module
                ];
                $investmentsTotal += $value;
            }
        }
        
        // Also include investments from account balances (for backward compatibility)
        // But exclude those that might already be in Investment module
        $managedInvestmentNames = $managedInvestments->pluck('name')->map(function($name) {
            return strtolower($name);
        })->toArray();
        
        foreach ($accountBalances as $accountId => $account) {
            $name = strtolower($account['name']);
            
            // Skip if this account name matches a managed investment
            $isManagedInvestment = false;
            foreach ($managedInvestmentNames as $managedName) {
                if (strpos($name, $managedName) !== false || strpos($managedName, $name) !== false) {
                    $isManagedInvestment = true;
                    break;
                }
            }
            
            if (!$isManagedInvestment &&
                (strpos($name, 'investment') !== false ||
                 strpos($name, 'share') !== false ||
                 strpos($name, 'mutual fund') !== false ||
                 strpos($name, 'insurance') !== false ||
                 strpos($name, 'commodities') !== false) &&
                $account['balance'] > 0
            ) {
                $balance = abs($account['balance']);
                $investments[] = [
                    'name' => $account['name'],
                    'value' => $balance,
                    'from_module' => false
                ];
                $investmentsTotal += $balance;
            }
        }

        if ($investmentsTotal > 0) {
            $assets[] = [
                'label' => 'Investments',
                'items' => $investments,
                'total' => $investmentsTotal,
                'is_group' => true
            ];
        }

        // 3. Current Assets
        $currentAssets = [];
        $currentAssetsItems = [];
        $currentAssetsTotal = 0;

        // Closing Stock
        if ($closingStock > 0.01) {
            $currentAssetsItems[] = [
                'name' => 'Closing Stock',
                'value' => $closingStock,
                'highlight' => true // Highlight as shown in image
            ];
            $currentAssetsTotal += $closingStock;
        }

        // Deposits (Asset)
        $deposits = [];
        foreach ($accountBalances as $accountId => $account) {
            $name = strtolower($account['name']);
            if (
                strpos($name, 'deposit') !== false &&
                strtolower($account['type']) == 'asset' &&
                $account['balance'] > 0
            ) {
                $balance = abs($account['balance']);
                $deposits[] = [
                    'name' => $account['name'],
                    'value' => $balance
                ];
                $currentAssetsTotal += $balance;
            }
        }
        $currentAssetsItems = array_merge($currentAssetsItems, $deposits);

        // Loans & Advances (Asset)
        $loansAdvances = [];
        foreach ($accountBalances as $accountId => $account) {
            $name = strtolower($account['name']);
            if (
                (strpos($name, 'loan') !== false || strpos($name, 'advance') !== false) &&
                strtolower($account['type']) == 'asset' &&
                $account['balance'] > 0
            ) {
                $balance = abs($account['balance']);
                $loansAdvances[] = [
                    'name' => $account['name'],
                    'value' => $balance
                ];
                $currentAssetsTotal += $balance;
            }
        }
        $currentAssetsItems = array_merge($currentAssetsItems, $loansAdvances);

        // Sundry Debtors (Customer Due)
        $customerDue = abs($trialBalance['customer_due'] ?? 0);
        if ($customerDue > 0.01) {
            $currentAssetsItems[] = [
                'name' => 'Sundry Debtors',
                'value' => $customerDue
            ];
            $currentAssetsTotal += $customerDue;
        }

        // Cash-in-Hand
        $cashInHand = 0;
        foreach ($accountBalances as $accountId => $account) {
            $name = strtolower($account['name']);
            if (
                (strpos($name, 'cash') !== false || strpos($name, 'cash-in-hand') !== false) &&
                strtolower($account['type']) == 'asset' &&
                $account['balance'] > 0
            ) {
                $cashInHand += abs($account['balance']);
            }
        }
        if ($cashInHand > 0.01) {
            $currentAssetsItems[] = [
                'name' => 'Cash-in-Hand',
                'value' => $cashInHand
            ];
            $currentAssetsTotal += $cashInHand;
        }

        // Bank Accounts
        $bankAccounts = [];
        $bankTotal = 0;
        foreach ($accountBalances as $accountId => $account) {
            $name = strtolower($account['name']);
            if (
                (strpos($name, 'bank') !== false || strpos($name, 'account') !== false) &&
                strtolower($account['type']) == 'asset' &&
                strpos($name, 'cash') === false &&
                $account['balance'] > 0
            ) {
                $balance = abs($account['balance']);
                $bankAccounts[] = [
                    'name' => $account['name'],
                    'value' => $balance
                ];
                $bankTotal += $balance;
            }
        }
        if ($bankTotal > 0.01) {
            $currentAssetsItems = array_merge($currentAssetsItems, $bankAccounts);
            $currentAssetsTotal += $bankTotal;
        }

        if ($currentAssetsTotal > 0) {
            $assets[] = [
                'label' => 'Current Assets',
                'items' => $currentAssetsItems,
                'total' => $currentAssetsTotal,
                'is_group' => true
            ];
        }

        // 4. Profit & Loss A/c
        $plTotal = $plOpening + $plCurrentPeriod;
        
        if (abs($plTotal) > 0.01 || abs($plOpening) > 0.01 || abs($plCurrentPeriod) > 0.01) {
            // Format items: show labels without values, then show total
            $plItems = [];
            if (abs($plOpening) > 0.01 || abs($plCurrentPeriod) > 0.01) {
                // Add Opening Balance label (no value shown)
                $plItems[] = [
                    'name' => 'Opening Balance',
                    'value' => 0, // Don't show value, just label
                    'show_value' => false
                ];
                // Add Current Period label (no value shown)
                $plItems[] = [
                    'name' => 'Current Period',
                    'value' => 0, // Don't show value, just label
                    'show_value' => false
                ];
            }
            
            $assets[] = [
                'label' => 'Profit & Loss A/c',
                'items' => $plItems,
                'total' => $plTotal,
                'is_group' => true,
                'show_items_without_values' => true // Flag to indicate items should be shown without values
            ];
        }

        // 5. Difference in opening balances (if any)
        $openingBalanceDiff = 0; // Calculate if needed
        if (abs($openingBalanceDiff) > 0.01) {
            $assets[] = [
                'label' => 'Difference in opening balances',
                'items' => [],
                'total' => $openingBalanceDiff,
                'is_group' => false
            ];
        }

        return $assets;
    }

    /**
     * Get account balance by account ID
     */
    protected function getAccountBalanceById($accountId, $asOfDate)
    {
        $balance = AccountTransaction::where('account_id', $accountId)
            ->whereNull('deleted_at')
            ->whereDate('operation_date', '<=', $asOfDate)
            ->select(DB::raw("COALESCE(SUM(IF(type='credit', amount, -1*amount)), 0) as balance"))
            ->value('balance');
            
        return (float) ($balance ?? 0);
    }

    /**
     * Get account balances
     */
    protected function getAccountBalances($businessId, $endDate, $locationId = null)
    {
        $query = Account::leftjoin(
            'account_transactions as AT',
            'AT.account_id',
            '=',
            'accounts.id'
        )
        ->whereNull('AT.deleted_at')
        ->where('accounts.business_id', $businessId)
        ->whereDate('AT.operation_date', '<=', $endDate);

        $permitted_locations = auth()->user()->permitted_locations();
        $account_ids = [];
        
        if ($permitted_locations != 'all') {
            $locations = BusinessLocation::where('business_id', $businessId)
                            ->whereIn('id', $permitted_locations)
                            ->get();

            foreach ($locations as $location) {
                if (!empty($location->default_payment_accounts)) {
                    $default_payment_accounts = json_decode($location->default_payment_accounts, true);
                    foreach ($default_payment_accounts as $key => $account) {
                        if (!empty($account['is_enabled']) && !empty($account['account'])) {
                            $account_ids[] = $account['account'];
                        }
                    }
                }
            }
            $account_ids = array_unique($account_ids);
        }

        if ($permitted_locations != 'all') {
            $query->whereIn('accounts.id', $account_ids);
        }

        if (!empty($locationId)) {
            $location = BusinessLocation::find($locationId);
            if (!empty($location->default_payment_accounts)) {
                $default_payment_accounts = json_decode($location->default_payment_accounts, true);
                $account_ids = [];
                foreach ($default_payment_accounts as $key => $account) {
                    if (!empty($account['is_enabled']) && !empty($account['account'])) {
                        $account_ids[] = $account['account'];
                    }
                }
                $query->whereIn('accounts.id', $account_ids);
            }
        }

        $hasAccountTypeColumn = \Schema::hasColumn('accounts', 'account_type');
        
        $selectFields = [
            'accounts.id',
            'accounts.name',
            'accounts.account_type_id',
            DB::raw("COALESCE(SUM(IF(AT.type='credit', AT.amount, -1*AT.amount)), 0) as balance")
        ];
        
        $groupByFields = ['accounts.id', 'accounts.name', 'accounts.account_type_id'];
        
        if ($hasAccountTypeColumn) {
            $selectFields[] = 'accounts.account_type as account_type_enum';
            $groupByFields[] = 'accounts.account_type';
        }

        $account_details = $query->select($selectFields)
        ->groupBy($groupByFields)
        ->get();

        $balances = [];
        foreach ($account_details as $account) {
            $accountType = 'asset';
            
            if ($account->account_type_id) {
                try {
                    $accountTypeModel = \App\AccountType::find($account->account_type_id);
                    if ($accountTypeModel) {
                        $typeName = strtolower($accountTypeModel->name ?? '');
                        if (strpos($typeName, 'asset') !== false || strpos($typeName, 'bank') !== false || strpos($typeName, 'cash') !== false) {
                            $accountType = 'asset';
                        } elseif (strpos($typeName, 'liability') !== false || strpos($typeName, 'payable') !== false || strpos($typeName, 'loan') !== false) {
                            $accountType = 'liability';
                        } elseif (strpos($typeName, 'equity') !== false || strpos($typeName, 'capital') !== false) {
                            $accountType = 'equity';
                        }
                    }
                } catch (\Exception $e) {
                }
            }
            
            if ($hasAccountTypeColumn && isset($account->account_type_enum)) {
                if ($account->account_type_enum == 'capital') {
                    $accountType = 'equity';
                } elseif ($account->account_type_enum == 'saving_current') {
                    $accountType = 'asset';
                }
            }
            
            $name = strtolower($account->name);
            if ($accountType == 'asset') {
                if (strpos($name, 'loan') !== false || strpos($name, 'borrowing') !== false || strpos($name, 'payable') !== false || strpos($name, 'liability') !== false) {
                    $accountType = 'liability';
                } elseif (strpos($name, 'capital') !== false || strpos($name, 'equity') !== false) {
                    $accountType = 'equity';
                }
            }
            
            $balances[$account->id] = [
                'name' => $account->name,
                'type' => $accountType,
                'balance' => (float) $account->balance
            ];
        }

        return $balances;
    }

    /**
     * Get opening Profit & Loss balance (from previous periods before financial year start)
     */
    protected function getOpeningProfitLoss($businessId, $fyStartDate, $locationId = null)
    {
        try {
            // Get P&L from all periods before the financial year start
            // We'll calculate it by getting P&L from a very early date to just before FY start
            $permitted_locations = auth()->user()->permitted_locations();
            
            // Use a date before FY start (e.g., 10 years ago to ensure we get all historical data)
            $startDate = date('Y-m-d', strtotime('-10 years'));
            $endDate = date('Y-m-d', strtotime($fyStartDate . ' -1 day'));
            
            $plData = $this->transactionUtil->getProfitLossDetails(
                $businessId, 
                $locationId, 
                $startDate, 
                $endDate, 
                null, 
                $permitted_locations
            );

            // Calculate net profit for opening period
            $sales = (float) ($plData['total_sell'] ?? 0);
            $salesDiscount = (float) ($plData['total_sell_discount'] ?? 0);
            $netSales = $sales - abs($salesDiscount);
            
            $totalPurchase = (float) ($plData['total_purchase'] ?? 0);
            $openingStock = (float) ($plData['opening_stock'] ?? 0);
            $closingStock = (float) ($plData['closing_stock'] ?? 0);
            $cogs = $openingStock + $totalPurchase - $closingStock;
            $grossProfit = $netSales - $cogs;
            
            $totalExpense = (float) ($plData['total_expense'] ?? 0);
            $netProfit = $grossProfit - $totalExpense;

            return $netProfit;
        } catch (\Exception $e) {
            \Log::error('BalanceSheetService getOpeningProfitLoss Error: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get current period Profit & Loss
     */
    protected function getCurrentPeriodProfitLoss($businessId, $startDate, $endDate, $locationId = null)
    {
        try {
            $permitted_locations = auth()->user()->permitted_locations();
            $plData = $this->transactionUtil->getProfitLossDetails(
                $businessId, 
                $locationId, 
                $startDate, 
                $endDate, 
                null, 
                $permitted_locations
            );

            // Calculate net profit
            $sales = (float) ($plData['total_sell'] ?? 0);
            $salesDiscount = (float) ($plData['total_sell_discount'] ?? 0);
            $netSales = $sales - abs($salesDiscount);
            
            $totalPurchase = (float) ($plData['total_purchase'] ?? 0);
            $openingStock = (float) ($plData['opening_stock'] ?? 0);
            $closingStock = (float) ($plData['closing_stock'] ?? 0);
            $cogs = $openingStock + $totalPurchase - $closingStock;
            $grossProfit = $netSales - $cogs;
            
            $totalExpense = (float) ($plData['total_expense'] ?? 0);
            $netProfit = $grossProfit - $totalExpense;

            return $netProfit;
        } catch (\Exception $e) {
            \Log::error('BalanceSheetService getCurrentPeriodProfitLoss Error: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Calculate total for a section
     */
    protected function calculateTotal($sections)
    {
        $total = 0;
        foreach ($sections as $section) {
            $total += (float) ($section['total'] ?? 0);
        }
        return $total;
    }
}

