@extends('layouts.app')

@section('title', __('accounting-reports::lang.trial_balance'))

@section('content')
<section class="content-header">
    <h1>@lang('accounting-reports::lang.trial_balance')</h1>
</section>

<section class="content">
    <div class="row">
        <div class="col-md-12">
            <div class="box box-solid">
                <div class="box-header with-border">
                    <h3 class="box-title">Filters</h3>
                </div>
                <div class="box-body">
                    <form id="filter-form">
                        <div class="row">
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label>@lang('accounting-reports::lang.location')</label>
                                    <select name="location_id" id="location_id" class="form-control select2" style="width:100%">
                                        <option value="">@lang('accounting-reports::lang.all_locations')</option>
                                        @foreach($locations as $locationId => $locationName)
                                            <option value="{{ $locationId }}">{{ $locationName }}</option>
                                        @endforeach
                                    </select>
                                </div>
                            </div>
                            <div class="col-sm-3 col-xs-6">
                                <div class="form-group">
                                    <label for="end_date">@lang('messages.filter_by_date'):</label>
                                    <div class="input-group">
                                        <span class="input-group-addon">
                                            <i class="fa fa-calendar"></i>
                                        </span>
                                        <input type="text" id="end_date" value="{{ @format_date('now') }}" class="form-control" readonly>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label>&nbsp;</label><br>
                                    <button type="button" id="generate-btn" class="btn btn-primary">
                                        <i class="fa fa-refresh"></i> @lang('accounting-reports::lang.view')
                                    </button>
                                    <button type="button" id="export-btn" class="btn btn-success">
                                        <i class="fa fa-download"></i> @lang('accounting-reports::lang.export')
                                    </button>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-md-12">
            <div class="box box-solid">
                <div class="box-header with-border">
                    <h3 class="box-title">Trial Balance</h3>
                </div>
                <div class="box-body">
                    <div id="loading" style="display:none;">
                        <i class="fa fa-spinner fa-spin"></i> Loading...
                    </div>
                    <div id="trial-balance-table"></div>
                </div>
            </div>
        </div>
    </div>
</section>
@endsection

@section('javascript')
<script>
$(document).ready(function() {
    // Initialize date picker (matching UltimatePOS)
    $('#end_date').datepicker({
        autoclose: true,
        format: datepicker_date_format
    });

    // Initialize select2
    $('#location_id').select2();

    // Load on page load
    update_trial_balance();

    $('#end_date').change(function() {
        update_trial_balance();
    });

    $('#location_id').change(function() {
        update_trial_balance();
    });

    $('#generate-btn').click(function() {
        update_trial_balance();
    });

    $('#export-btn').click(function() {
        var endDate = $('#end_date').val();
        var locationId = $('#location_id').val() || '';
        window.location.href = '{{ route("accounting-reports.trial-balance.export") }}?end_date=' + endDate + '&location_id=' + locationId + '&format=pdf';
    });

    function update_trial_balance() {
        $('#loading').show();
        $('#trial-balance-table').html('');
        
        var end_date = $('#end_date').val();
        var location_id = $('#location_id').val() || '';
        
        $.ajax({
            url: '{{ route("accounting-reports.trial-balance.get-data") }}',
            method: 'GET',
            data: {
                end_date: end_date,
                location_id: location_id
            },
            success: function(response) {
                if (response) {
                    renderTrialBalance(response);
                } else {
                    $('#trial-balance-table').html('<div class="alert alert-warning"><i class="fa fa-exclamation-triangle"></i> Invalid response from server.</div>');
                }
                $('#loading').hide();
            },
            error: function(xhr, status, error) {
                $('#loading').hide();
                console.error('Error:', xhr.responseText);
                var errorMsg = 'Error loading trial balance';
                if (xhr.responseJSON && xhr.responseJSON.error) {
                    errorMsg = xhr.responseJSON.error;
                } else if (xhr.responseText) {
                    try {
                        var response = JSON.parse(xhr.responseText);
                        if (response.message) {
                            errorMsg = response.message;
                        }
                    } catch(e) {
                        errorMsg = error + ' (Status: ' + xhr.status + ')';
                    }
                }
                $('#trial-balance-table').html('<div class="alert alert-danger"><i class="fa fa-exclamation-triangle"></i> ' + errorMsg + '</div>');
            }
        });
    }

    function renderTrialBalance(data) {
        var html = '<table class="table table-bordered table-striped" id="trial_balance_table">';
        html += '<thead><tr class="bg-gray">';
        html += '<th>@lang("account.trial_balance")</th>';
        html += '<th>@lang("account.debit")</th>';
        html += '<th>@lang("account.credit")</th>';
        html += '</tr></thead><tbody>';

        // Supplier Due (Accounts Payable) - matches UltimatePOS display logic
        var supplier_due_val = __currency_trans_from_en(data.supplier_due || 0);
        var supplier_due_display = __currency_trans_from_en(data.supplier_due || 0, true);
        html += '<tr>';
        html += '<th>@lang("account.supplier_due"):</th>';
        html += '<td>&nbsp;</td>';
        html += '<td>';
        html += '<input type="hidden" id="hidden_supplier_due" class="debit" value="' + supplier_due_val + '">';
        html += '<span id="supplier_due">' + supplier_due_display + '</span>';
        html += '</td>';
        html += '</tr>';

        // Customer Due (Accounts Receivable) - matches UltimatePOS display logic
        var customer_due_val = __currency_trans_from_en(data.customer_due || 0);
        var customer_due_display = __currency_trans_from_en(data.customer_due || 0, true);
        html += '<tr>';
        html += '<th>@lang("account.customer_due"):</th>';
        html += '<td>';
        html += '<input type="hidden" id="hidden_customer_due" class="credit" value="' + customer_due_val + '">';
        html += '<span id="customer_due">' + customer_due_display + '</span>';
        html += '</td>';
        html += '<td>&nbsp;</td>';
        html += '</tr>';

        // Account Balances
        html += '<tr><th colspan="3">@lang("account.account_balances"):</th></tr>';
        
        if (data.account_balances && Object.keys(data.account_balances).length > 0) {
            for (var accountName in data.account_balances) {
                var balance = data.account_balances[accountName];
                var balance_val = __currency_trans_from_en(balance);
                var balance_display = __currency_trans_from_en(balance, true);
                
                html += '<tr>';
                html += '<td class="pl-20-td">' + accountName + ':</td>';
                
                if (balance >= 0) {
                    // Positive balance = Credit
                    html += '<td>&nbsp;</td>';
                    html += '<td>';
                    html += '<input type="hidden" class="credit" value="' + balance_val + '">';
                    html += '<span>' + balance_display + '</span>';
                    html += '</td>';
                } else {
                    // Negative balance = Debit
                    html += '<td>';
                    html += '<input type="hidden" class="debit" value="' + Math.abs(balance_val) + '">';
                    html += '<span>' + __currency_trans_from_en(Math.abs(balance), true) + '</span>';
                    html += '</td>';
                    html += '<td>&nbsp;</td>';
                }
                html += '</tr>';
            }
        } else {
            html += '<tr><td colspan="3" class="text-center">No account balances found</td></tr>';
        }

        html += '</tbody><tfoot>';
        html += '<tr class="bg-gray">';
        html += '<th>@lang("sale.total")</th>';
        html += '<td><span class="remote-data" id="total_credit">0.00</span></td>';
        html += '<td><span class="remote-data" id="total_debit">0.00</span></td>';
        html += '</tr>';
        html += '</tfoot></table>';

        $('#trial-balance-table').html(html);

        // Calculate totals (matching UltimatePOS logic)
        var total_debit = 0;
        var total_credit = 0;
        $('#trial_balance_table input.debit').each(function() {
            total_debit += __read_number($(this));
        });
        $('#trial_balance_table input.credit').each(function() {
            total_credit += __read_number($(this));
        });

        $('#trial_balance_table #total_debit').text(__currency_trans_from_en(total_debit, true));
        $('#trial_balance_table #total_credit').text(__currency_trans_from_en(total_credit, true));

        if (!data.is_balanced) {
            $('#trial-balance-table').after('<div class="alert alert-warning"><strong>Warning:</strong> Trial Balance is not balanced! Debit: ' + __currency_trans_from_en(total_debit, true) + ', Credit: ' + __currency_trans_from_en(total_credit, true) + '</div>');
        }
    }

    function formatNumber(num) {
        return parseFloat(num).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2});
    }

    // Load on page load
    loadTrialBalance();
});
</script>
@endsection


