@extends('layouts.app')

@section('title', __('accounting-reports::lang.receivables'))

@section('content')

<!-- Content Header (Page header) -->
<section class="content-header">
    <h1>@lang('accounting-reports::lang.receivables')
        <small class="text-muted">@lang('accounting-reports::lang.tally_style')</small>
    </h1>
</section>

<!-- Main content -->
<section class="content">
    <div class="print_section">
        <h2 class="text-center">{{ session()->get('business.name') }} - @lang('accounting-reports::lang.receivables')</h2>
        <p class="text-center text-muted" id="ar_period"></p>
    </div>

    <div class="row no-print">
        <div class="col-md-3 col-xs-12">
            <div class="input-group">
                <span class="input-group-addon bg-light-blue"><i class="fa fa-map-marker"></i></span>
                <select class="form-control select2" id="ar_location_filter" style="width:100%">
                    <option value="">@lang('messages.all')</option>
                    @foreach ($locations as $key => $value)
                        <option value="{{ $key }}">{{ $value }}</option>
                    @endforeach
                </select>
            </div>
        </div>
    
        <div class="col-md-3 col-xs-12">
            <div class="form-group">
                <div class="input-group">
                    <span class="input-group-addon">
                        <i class="fa fa-calendar"></i>
                    </span>
                    <input type="text" id="ar_as_on_date" value="{{ @format_date('now') }}" class="form-control" readonly placeholder="@lang('accounting-reports::lang.as_on_date')">
                </div>
            </div>
        </div>
        
        <div class="col-md-3 col-xs-12">
            <button type="button" class="btn btn-primary btn-block" id="generate_ar_btn">
                <i class="fa fa-refresh"></i> @lang('accounting-reports::lang.generate')
            </button>
        </div>

        <div class="col-md-3 col-xs-12">
            <button class="btn btn-primary btn-block" aria-label="Print" onclick="window.print();">
                <i class="fa fa-print"></i> @lang('messages.print')
            </button>
        </div>
    </div>

    <div class="row">
        <div id="ar_data_div" class="col-md-12">
            <div class="alert alert-info text-center">
                <i class="fa fa-info-circle"></i> @lang('accounting-reports::lang.select_as_on_date')
            </div>
        </div>
    </div>

</section>
<!-- /.content -->

@endsection

@section('javascript')
<script type="text/javascript">
    $(document).ready(function() {
        // Initialize date picker (matching Trial Balance and Balance Sheet)
        $('#ar_as_on_date').datepicker({
            autoclose: true,
            format: datepicker_date_format
        });

        // Initialize select2
        $('#ar_location_filter').select2();

        // Load on page load
        loadReceivables();

        // Load receivables on date change
        $('#ar_as_on_date').change(function() {
            loadReceivables();
        });

        // Load receivables on location change or button click
        $('#ar_location_filter').change(function() {
            loadReceivables();
        });

        $('#generate_ar_btn').click(function() {
            loadReceivables();
        });

        function loadReceivables() {
            var as_on_date = $('#ar_as_on_date').val();
            var location_id = $('#ar_location_filter').val() || '';
            
            if (!as_on_date) {
                $('#ar_data_div').html('<div class="alert alert-warning"><i class="fa fa-exclamation-triangle"></i> @lang("accounting-reports::lang.please_select_as_on_date")</div>');
                return;
            }

            $('#ar_data_div').html('<div class="text-center"><i class="fa fa-spinner fa-spin fa-3x"></i><br>Loading...</div>');

            $.ajax({
                url: '{{ route("accounting-reports.receivables.get-data") }}',
                method: 'GET',
                data: {
                    as_on_date: as_on_date,
                    location_id: location_id
                },
                success: function(response) {
                    if (response && response.success) {
                        renderTallyReceivables(response);
                    } else {
                        var errorMsg = response.error || 'Error loading data';
                        $('#ar_data_div').html('<div class="alert alert-danger"><i class="fa fa-exclamation-triangle"></i> ' + errorMsg + '</div>');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Error:', xhr.responseText);
                    var errorMsg = 'Error: Internal Server Error';
                    try {
                        if (xhr.responseJSON && xhr.responseJSON.error) {
                            errorMsg = xhr.responseJSON.error;
                        } else if (xhr.responseText) {
                            var response = JSON.parse(xhr.responseText);
                            if (response.error) {
                                errorMsg = response.error;
                            }
                        }
                    } catch(e) {
                        errorMsg = 'Error: ' + error + ' (Status: ' + xhr.status + ')';
                    }
                    $('#ar_data_div').html('<div class="alert alert-danger"><i class="fa fa-exclamation-triangle"></i> ' + errorMsg + '</div>');
                }
            });
        }

        function renderTallyReceivables(data) {
            var html = '';
            var businessName = '{{ session()->get("business.name") }}';
            var asOnDate = data.as_on_date || $('#ar_as_on_date').val();
            
            // Update period display
            $('#ar_period').text('As on: ' + asOnDate);
            
            // Summary card
            html += '<div class="row">';
            html += '<div class="col-md-12">';
            html += '<div class="box box-primary">';
            html += '<div class="box-header with-border">';
            html += '<h3 class="box-title">@lang("accounting-reports::lang.summary")</h3>';
            html += '</div>';
            html += '<div class="box-body">';
            html += '<div class="row">';
            html += '<div class="col-md-3">';
            html += '<div class="info-box">';
            html += '<span class="info-box-icon bg-blue"><i class="fa fa-users"></i></span>';
            html += '<div class="info-box-content">';
            html += '<span class="info-box-text">@lang("accounting-reports::lang.total_customers")</span>';
            html += '<span class="info-box-number">' + (data.summary ? data.summary.total_customers : 0) + '</span>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '<div class="col-md-3">';
            html += '<div class="info-box">';
            html += '<span class="info-box-icon bg-green"><i class="fa fa-file-text"></i></span>';
            html += '<div class="info-box-content">';
            html += '<span class="info-box-text">@lang("accounting-reports::lang.total_invoices")</span>';
            html += '<span class="info-box-number">' + (data.summary ? data.summary.total_invoices : 0) + '</span>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '<div class="col-md-3">';
            html += '<div class="info-box">';
            html += '<span class="info-box-icon bg-yellow"><i class="fa fa-money"></i></span>';
            html += '<div class="info-box-content">';
            html += '<span class="info-box-text">@lang("accounting-reports::lang.total_receivables")</span>';
            html += '<span class="info-box-number">' + __currency_trans_from_en(data.total_receivables || 0, true) + '</span>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '<div class="col-md-3">';
            html += '<div class="info-box">';
            html += '<span class="info-box-icon bg-red"><i class="fa fa-clock-o"></i></span>';
            html += '<div class="info-box-content">';
            html += '<span class="info-box-text">@lang("accounting-reports::lang.as_on_date")</span>';
            html += '<span class="info-box-number" style="font-size:14px;">' + asOnDate + '</span>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            
            // Aging Summary
            html += '<div class="row">';
            html += '<div class="col-md-12">';
            html += '<div class="box box-info">';
            html += '<div class="box-header with-border">';
            html += '<h3 class="box-title">@lang("accounting-reports::lang.aging_summary")</h3>';
            html += '</div>';
            html += '<div class="box-body">';
            html += '<div class="table-responsive">';
            html += '<table class="table table-bordered table-striped">';
            html += '<thead>';
            html += '<tr>';
            html += '<th>@lang("accounting-reports::lang.current")</th>';
            html += '<th>@lang("accounting-reports::lang.b0_7_days")</th>';
            html += '<th>@lang("accounting-reports::lang.b8_30_days")</th>';
            html += '<th>@lang("accounting-reports::lang.b31_60_days")</th>';
            html += '<th>@lang("accounting-reports::lang.b61_90_days")</th>';
            html += '<th>@lang("accounting-reports::lang.b90_plus_days")</th>';
            html += '<th><strong>@lang("accounting-reports::lang.total")</strong></th>';
            html += '</tr>';
            html += '</thead>';
            html += '<tbody>';
            html += '<tr>';
            html += '<td>' + __currency_trans_from_en(data.total_current || 0, true) + '</td>';
            html += '<td>' + __currency_trans_from_en(data.total_b0_7 || 0, true) + '</td>';
            html += '<td>' + __currency_trans_from_en(data.total_b8_30 || 0, true) + '</td>';
            html += '<td>' + __currency_trans_from_en(data.total_b31_60 || 0, true) + '</td>';
            html += '<td>' + __currency_trans_from_en(data.total_b61_90 || 0, true) + '</td>';
            html += '<td>' + __currency_trans_from_en(data.total_b90_plus || 0, true) + '</td>';
            html += '<td><strong>' + __currency_trans_from_en(data.total_receivables || 0, true) + '</strong></td>';
            html += '</tr>';
            html += '</tbody>';
            html += '</table>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            
            // Month-wise Aging Summary
            html += '<div class="row">';
            html += '<div class="col-md-12">';
            html += '<div class="box box-warning">';
            html += '<div class="box-header with-border">';
            html += '<h3 class="box-title">@lang("accounting-reports::lang.month_wise_aging_summary")</h3>';
            html += '</div>';
            html += '<div class="box-body">';
            html += '<div class="table-responsive">';
            html += '<table class="table table-bordered table-striped">';
            html += '<thead>';
            html += '<tr>';
            html += '<th>@lang("accounting-reports::lang.current_month")</th>';
            html += '<th>@lang("accounting-reports::lang.month_1")</th>';
            html += '<th>@lang("accounting-reports::lang.month_2")</th>';
            html += '<th>@lang("accounting-reports::lang.month_3")</th>';
            html += '<th>@lang("accounting-reports::lang.month_4_6")</th>';
            html += '<th>@lang("accounting-reports::lang.month_7_12")</th>';
            html += '<th>@lang("accounting-reports::lang.month_12_plus")</th>';
            html += '<th><strong>@lang("accounting-reports::lang.total")</strong></th>';
            html += '</tr>';
            html += '</thead>';
            html += '<tbody>';
            html += '<tr>';
            html += '<td>' + __currency_trans_from_en(data.total_current_month || 0, true) + '</td>';
            html += '<td>' + __currency_trans_from_en(data.total_month_1 || 0, true) + '</td>';
            html += '<td>' + __currency_trans_from_en(data.total_month_2 || 0, true) + '</td>';
            html += '<td>' + __currency_trans_from_en(data.total_month_3 || 0, true) + '</td>';
            html += '<td>' + __currency_trans_from_en(data.total_month_4_6 || 0, true) + '</td>';
            html += '<td>' + __currency_trans_from_en(data.total_month_7_12 || 0, true) + '</td>';
            html += '<td>' + __currency_trans_from_en(data.total_month_12_plus || 0, true) + '</td>';
            html += '<td><strong>' + __currency_trans_from_en(data.total_receivables || 0, true) + '</strong></td>';
            html += '</tr>';
            html += '</tbody>';
            html += '</table>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            
            // Tally-style Receivables (Grouped by Customer)
            html += '<div class="row">';
            html += '<div class="col-md-12">';
            html += '<div class="box box-primary">';
            html += '<div class="box-header with-border">';
            html += '<h3 class="box-title">@lang("accounting-reports::lang.sundry_debtors") - @lang("accounting-reports::lang.receivables")</h3>';
            html += '</div>';
            html += '<div class="box-body">';
            html += '<div class="table-responsive">';
            html += '<table class="table table-bordered table-striped tally-receivables-table">';
            html += '<thead>';
            html += '<tr>';
            html += '<th width="20%">@lang("accounting-reports::lang.customer")</th>';
            html += '<th width="15%">@lang("accounting-reports::lang.invoice_no")</th>';
            html += '<th width="10%">@lang("accounting-reports::lang.invoice_date")</th>';
            html += '<th width="10%">@lang("accounting-reports::lang.due_date")</th>';
            html += '<th width="10%">@lang("accounting-reports::lang.invoice_amount")</th>';
            html += '<th width="10%">@lang("accounting-reports::lang.paid_amount")</th>';
            html += '<th width="10%">@lang("accounting-reports::lang.outstanding")</th>';
            html += '<th width="10%">@lang("accounting-reports::lang.ageing")</th>';
            html += '<th width="5%">@lang("accounting-reports::lang.days")</th>';
            html += '</tr>';
            html += '</thead>';
            html += '<tbody>';
            
            if (data.customers && data.customers.length > 0) {
                data.customers.forEach(function(customer) {
                    // Customer header row
                    html += '<tr class="customer-header-row">';
                    html += '<td colspan="9">';
                    html += '<strong>' + (customer.contact_code || '') + ' - ' + customer.contact_name + '</strong>';
                    html += '<span class="pull-right">';
                    html += '<strong>@lang("accounting-reports::lang.total"): ' + __currency_trans_from_en(customer.total_outstanding, true) + '</strong>';
                    html += '</span>';
                    html += '</td>';
                    html += '</tr>';
                    
                    // Invoice rows for this customer
                    if (customer.invoices && customer.invoices.length > 0) {
                        customer.invoices.forEach(function(invoice) {
                            html += '<tr class="invoice-row">';
                            html += '<td></td>'; // Empty cell under customer name
                            html += '<td>' + (invoice.invoice_no || '-') + '</td>';
                            html += '<td>' + invoice.transaction_date + '</td>';
                            html += '<td>' + invoice.due_date + '</td>';
                            html += '<td class="text-right">' + __currency_trans_from_en(invoice.invoice_amount, true) + '</td>';
                            html += '<td class="text-right">' + __currency_trans_from_en(invoice.paid_amount, true) + '</td>';
                            html += '<td class="text-right"><strong>' + __currency_trans_from_en(invoice.outstanding, true) + '</strong></td>';
                            
                            // Aging bucket with color coding
                            var agingClass = '';
                            var agingLabel = '';
                            switch(invoice.aging_bucket) {
                                case 'current':
                                    agingClass = 'label-success';
                                    agingLabel = 'Current';
                                    break;
                                case 'b0_7':
                                    agingClass = 'label-info';
                                    agingLabel = '0-7 Days';
                                    break;
                                case 'b8_30':
                                    agingClass = 'label-warning';
                                    agingLabel = '8-30 Days';
                                    break;
                                case 'b31_60':
                                    agingClass = 'label-primary';
                                    agingLabel = '31-60 Days';
                                    break;
                                case 'b61_90':
                                    agingClass = 'label-danger';
                                    agingLabel = '61-90 Days';
                                    break;
                                case 'b90_plus':
                                    agingClass = 'label-danger';
                                    agingLabel = '90+ Days';
                                    break;
                                default:
                                    agingClass = 'label-default';
                                    agingLabel = 'Current';
                            }
                            html += '<td><span class="label ' + agingClass + '">' + agingLabel + '</span></td>';
                            html += '<td class="text-right">' + invoice.days_past_due + '</td>';
                            html += '</tr>';
                        });
                    }
                });
            } else {
                html += '<tr>';
                html += '<td colspan="9" class="text-center">';
                html += '<div class="alert alert-info">@lang("accounting-reports::lang.no_receivables_found")</div>';
                html += '</td>';
                html += '</tr>';
            }
            
            // Grand Total Row
            html += '<tr class="grand-total-row">';
            html += '<td colspan="6" class="text-right"><strong>@lang("accounting-reports::lang.grand_total"):</strong></td>';
            html += '<td class="text-right"><strong>' + __currency_trans_from_en(data.total_receivables || 0, true) + '</strong></td>';
            html += '<td colspan="2"></td>';
            html += '</tr>';
            
            html += '</tbody>';
            html += '</table>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            
            $('#ar_data_div').html(html);
            
            // Apply currency conversion
            __currency_convert_recursively($('#ar_data_div'));
        }

    });
</script>
<style>
    .tally-receivables-table {
        font-size: 13px;
        margin-bottom: 0;
    }
    .tally-receivables-table th {
        background-color: #f8f9fa;
        font-weight: bold;
        border: 1px solid #dee2e6;
        text-align: center;
    }
    .tally-receivables-table td {
        border: 1px solid #dee2e6;
        padding: 8px;
    }
    .customer-header-row {
        background-color: #e9ecef !important;
        font-weight: bold;
    }
    .customer-header-row td {
        padding: 12px 8px;
        font-weight: bold;
    }
    .invoice-row td {
        padding-left: 30px;
    }
    .grand-total-row {
        background-color: #f8f9fa !important;
        font-weight: bold;
        border-top: 2px solid #333;
    }
    .grand-total-row td {
        padding: 12px 8px;
        font-weight: bold;
    }
    @media print {
        .no-print { display: none !important; }
        .print_section { display: block !important; }
    }
</style>
@endsection
