@extends('layouts.app')

@section('title', __('accounting-reports::lang.ratio_analysis'))

@section('content')
<section class="content-header">
    <h1>@lang('accounting-reports::lang.ratio_analysis')</h1>
</section>

<section class="content">
    <div class="row">
        <div class="col-md-12">
            <div class="box box-solid">
                <div class="box-header with-border">
                    <h3 class="box-title">Filters</h3>
                </div>
                <div class="box-body">
                    <form id="filter-form">
                        <div class="row">
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label>@lang('accounting-reports::lang.location')</label>
                                    <select name="location_id" id="location_id" class="form-control select2" style="width:100%">
                                        <option value="">@lang('accounting-reports::lang.all_locations')</option>
                                        @foreach($locations as $locationId => $locationName)
                                            <option value="{{ $locationId }}">{{ $locationName }}</option>
                                        @endforeach
                                    </select>
                                </div>
                            </div>
                            <div class="col-sm-3 col-xs-6">
                                <div class="form-group">
                                    <label for="start_date">@lang('accounting-reports::lang.start_date'):</label>
                                    <div class="input-group">
                                        <span class="input-group-addon">
                                            <i class="fa fa-calendar"></i>
                                        </span>
                                        <input type="text" id="start_date" value="{{ @format_date('now') }}" class="form-control" readonly>
                                    </div>
                                </div>
                            </div>
                            <div class="col-sm-3 col-xs-6">
                                <div class="form-group">
                                    <label for="end_date">@lang('accounting-reports::lang.end_date'):</label>
                                    <div class="input-group">
                                        <span class="input-group-addon">
                                            <i class="fa fa-calendar"></i>
                                        </span>
                                        <input type="text" id="end_date" value="{{ @format_date('now') }}" class="form-control" readonly>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label>&nbsp;</label><br>
                                    <button type="button" id="generate-btn" class="btn btn-primary">
                                        <i class="fa fa-refresh"></i> @lang('accounting-reports::lang.view')
                                    </button>
                                    <button type="button" id="export-btn" class="btn btn-success">
                                        <i class="fa fa-download"></i> @lang('accounting-reports::lang.export')
                                    </button>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-md-12">
            <div class="box box-solid">
                <div class="box-header with-border">
                    <h3 class="box-title">Principal Ratios</h3>
                </div>
                <div class="box-body">
                    <div id="loading" style="display:none; text-align:center; padding:20px;">
                        <i class="fa fa-spinner fa-spin fa-3x"></i>
                        <p>Loading ratio analysis...</p>
                    </div>
                    <div id="ratio-analysis-content" style="display:none;">
                        <div class="row">
                            <div class="col-md-12">
                                <h4 id="business-name" style="text-align:center; margin-bottom:10px;"></h4>
                                <p id="period-range" style="text-align:center; margin-bottom:20px;"></p>
                            </div>
                        </div>
                        <div class="table-responsive">
                            <table class="table table-bordered table-striped" id="ratios-table">
                                <thead>
                                    <tr>
                                        <th width="30%">Ratio Name</th>
                                        <th width="20%">Value</th>
                                        <th width="50%">Formula</th>
                                    </tr>
                                </thead>
                                <tbody id="ratios-tbody">
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <div id="error-message" style="display:none;" class="alert alert-danger"></div>
                </div>
            </div>
        </div>
    </div>
</section>
@endsection

@section('javascript')
<script>
$(document).ready(function() {
    // Initialize date pickers
    $('#start_date').datepicker({
        autoclose: true,
        format: datepicker_date_format
    });

    $('#end_date').datepicker({
        autoclose: true,
        format: datepicker_date_format
    });

    // Initialize select2
    $('#location_id').select2();

    // Load on page load
    update_ratio_analysis();

    $('#start_date, #end_date, #location_id').change(function() {
        update_ratio_analysis();
    });

    $('#generate-btn').click(function() {
        update_ratio_analysis();
    });

    $('#export-btn').click(function() {
        export_ratio_analysis();
    });

    function update_ratio_analysis() {
        var location_id = $('#location_id').val();
        var start_date = $('#start_date').val();
        var end_date = $('#end_date').val();

        if (!start_date || !end_date) {
            toastr.error('Please select start date and end date');
            return;
        }

        $('#loading').show();
        $('#ratio-analysis-content').hide();
        $('#error-message').hide();

        $.ajax({
            url: "{{ action([\Modules\AccountingReports\Http\Controllers\RatioAnalysisController::class, 'getData']) }}",
            data: {
                location_id: location_id,
                start_date: start_date,
                end_date: end_date
            },
            dataType: 'json',
            success: function(response) {
                $('#loading').hide();
                
                if (response.success === false) {
                    $('#error-message').text(response.error || 'Error loading ratio analysis').show();
                    return;
                }

                if (response.ratios && response.ratios.length > 0) {
                    display_ratios(response);
                    $('#ratio-analysis-content').show();
                } else {
                    $('#error-message').text('No data available for the selected period').show();
                }
            },
            error: function(xhr, status, error) {
                $('#loading').hide();
                var errorMsg = 'Error loading ratio analysis';
                if (xhr.responseJSON && xhr.responseJSON.error) {
                    errorMsg = xhr.responseJSON.error;
                }
                $('#error-message').text(errorMsg).show();
            }
        });
    }

    function display_ratios(data) {
        // Set business name and period
        $('#business-name').text(data.business_name || '');
        
        var startDate = data.period.start_date;
        var endDate = data.period.end_date;
        var formattedStart = moment(startDate).format('DD-MMM-YY');
        var formattedEnd = moment(endDate).format('DD-MMM-YY');
        $('#period-range').text(formattedStart + ' to ' + formattedEnd);

        // Clear existing rows
        $('#ratios-tbody').empty();

        // Add ratio rows
        $.each(data.ratios, function(index, ratio) {
            var row = $('<tr>');
            
            // Highlight Quick Ratio row (as shown in image)
            if (ratio.name === 'Quick Ratio') {
                row.css('background-color', '#fffacd'); // Light yellow
            }

            // Ratio Name
            row.append($('<td>').text(ratio.name));

            // Value
            var valueCell = $('<td>').text(ratio.value);
            valueCell.css('font-weight', 'bold');
            row.append(valueCell);

            // Formula
            var formulaCell = $('<td>');
            if (ratio.formula) {
                formulaCell.text(ratio.formula);
            } else {
                formulaCell.html('&nbsp;');
            }
            row.append(formulaCell);

            $('#ratios-tbody').append(row);
        });
    }

    function export_ratio_analysis() {
        var location_id = $('#location_id').val();
        var start_date = $('#start_date').val();
        var end_date = $('#end_date').val();

        if (!start_date || !end_date) {
            toastr.error('Please select start date and end date');
            return;
        }

        var url = "{{ action([\Modules\AccountingReports\Http\Controllers\RatioAnalysisController::class, 'export']) }}";
        url += '?location_id=' + location_id + '&start_date=' + start_date + '&end_date=' + end_date;
        
        window.location.href = url;
    }
});
</script>
@endsection
