@extends('layouts.app')

@section('title', __('accounting-reports::lang.fixed_asset_revaluations'))

@section('content')

<!-- Content Header (Page header) -->
<section class="content-header">
    <h1>@lang('accounting-reports::lang.fixed_asset_revaluations')</h1>
</section>

<!-- Main content -->
<section class="content">
    @component('components.widget', ['class' => 'box-primary'])
        <div class="row">
            <div class="col-md-12">
                <div class="btn-group">
                    <button type="button" class="btn btn-success btn-modal" data-href="{{ action([\Modules\AccountingReports\Http\Controllers\FixedAssetRevaluationController::class, 'create']) }}" data-container=".revaluation_modal">
                        <i class="fa fa-plus"></i> @lang('accounting-reports::lang.add_revaluation')
                    </button>
                </div>
            </div>
        </div>
        <br>
        
        <div class="row">
            <div class="col-md-3">
                <div class="form-group">
                    <label>@lang('accounting-reports::lang.filter_by_asset')</label>
                    <select name="filter_asset" id="filter_asset" class="form-control select2" style="width:100%">
                        <option value="">@lang('accounting-reports::lang.all_assets')</option>
                        @foreach($assets as $assetId => $assetName)
                            <option value="{{ $assetId }}">{{ $assetName }}</option>
                        @endforeach
                    </select>
                </div>
            </div>
        </div>
        
        <div class="table-responsive">
            <table class="table table-bordered table-striped table-hover" id="revaluations_table" style="width:100%;">
                <thead>
                    <tr>
                        <th>@lang('accounting-reports::lang.asset_name')</th>
                        <th>@lang('accounting-reports::lang.revaluation_date')</th>
                        <th>@lang('accounting-reports::lang.previous_value')</th>
                        <th>@lang('accounting-reports::lang.revaluation_amount')</th>
                        <th>@lang('accounting-reports::lang.revaluation_difference')</th>
                        <th>@lang('accounting-reports::lang.reason')</th>
                        <th>@lang('accounting-reports::lang.created_at')</th>
                        <th>@lang('messages.action')</th>
                    </tr>
                </thead>
            </table>
        </div>
    @endcomponent
    
    <div class="modal fade revaluation_modal" tabindex="-1" role="dialog"></div>
</section>

@endsection

@section('javascript')
<script type="text/javascript">
    $(document).ready(function() {
        $('#filter_asset').select2();
        
        var revaluations_table = $('#revaluations_table').DataTable({
            processing: true,
            serverSide: true,
            ajax: {
                url: '{{ action([\Modules\AccountingReports\Http\Controllers\FixedAssetRevaluationController::class, "getData"]) }}',
                data: function(d) {
                    d.fixed_asset_id = $('#filter_asset').val();
                }
            },
            columns: [
                { data: 'asset_name', name: 'asset_name' },
                { data: 'revaluation_date', name: 'revaluation_date' },
                { data: 'previous_value', name: 'previous_value' },
                { data: 'revaluation_amount', name: 'revaluation_amount' },
                { data: 'revaluation_difference', name: 'revaluation_difference' },
                { data: 'reason', name: 'reason' },
                { data: 'created_at', name: 'created_at' },
                { data: 'action', name: 'action', orderable: false, searchable: false }
            ],
        });

        $('#filter_asset').change(function() {
            revaluations_table.ajax.reload();
        });

        $(document).on('click', '.view_revaluation', function(e) {
            e.preventDefault();
            $('div.revaluation_modal').load($(this).data('href'), function() {
                $(this).modal('show');
            });
        });

        $(document).on('click', '.delete_revaluation', function() {
            var url = $(this).data('href');
            swal({
                title: LANG.sure,
                text: '{{ __("accounting-reports::lang.confirm_delete_revaluation") }}',
                icon: "warning",
                buttons: true,
                dangerMode: true,
            }).then((confirmed) => {
                if (confirmed) {
                    $.ajax({
                        method: 'DELETE',
                        url: url,
                        dataType: 'json',
                        success: function(response) {
                            if (response.success) {
                                toastr.success(response.msg);
                                revaluations_table.ajax.reload();
                            } else {
                                toastr.error(response.msg);
                            }
                        }
                    });
                }
            });
        });

        $(document).on('submit', 'form#add_revaluation_form', function(e) {
            e.preventDefault();
            $(this).validate();
            if ($(this).valid()) {
                $.ajax({
                    method: 'POST',
                    url: $(this).attr('action'),
                    data: $(this).serialize(),
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            $('div.revaluation_modal').modal('hide');
                            toastr.success(response.msg);
                            revaluations_table.ajax.reload();
                        } else {
                            toastr.error(response.msg);
                        }
                    }
                });
            }
        });
        
        // Initialize modal
        $(document).on('click', '.btn-modal', function(e) {
            e.preventDefault();
            $('div.revaluation_modal').load($(this).data('href'), function() {
                $(this).modal('show');
                $('.date-picker').datepicker({
                    autoclose: true,
                    format: datepicker_date_format
                });
                $('.select2').select2();
                
                // Store asset data and current book value
                var assetData = null;
                var currentBookValue = 0;
                
                // Load asset details when asset is selected (store data but don't show yet)
                $('#fixed_asset_id').on('change', function() {
                    var assetId = $(this).val();
                    if (assetId) {
                        $.ajax({
                            url: "{{ route('accounting-reports.fixed-asset-revaluations.get-asset-details', '') }}/" + assetId,
                            method: 'GET',
                            dataType: 'json',
                            success: function(response) {
                                if (response.success) {
                                    // Store asset data
                                    assetData = response;
                                    currentBookValue = parseFloat(response.current_book_value);
                                    
                                    // Check if date is selected, then show info and auto-fill amount
                                    var revaluationDate = $('#revaluation_date').val();
                                    if (revaluationDate) {
                                        showAssetInfoAndAutoFill();
                                    }
                                }
                            },
                            error: function() {
                                toastr.error('Error loading asset details');
                                assetData = null;
                            }
                        });
                    } else {
                        assetData = null;
                        $('#asset_info_section').slideUp();
                        $('#revaluation_amount').val('');
                    }
                });
                
                // Show asset info and auto-fill amount when date is selected
                $('#revaluation_date').on('changeDate', function() {
                    if (assetData && $(this).val()) {
                        showAssetInfoAndAutoFill();
                    }
                });
                
                // Also trigger on manual date input
                $('#revaluation_date').on('change', function() {
                    if (assetData && $(this).val()) {
                        showAssetInfoAndAutoFill();
                    }
                });
                
                // Function to show asset info and auto-fill revaluation amount
                function showAssetInfoAndAutoFill() {
                    if (!assetData) return;
                    
                    // Display asset information with currency formatting
                    $('#display_original_amount').text(assetData.original_amount);
                    $('#display_current_book_value').text(assetData.current_book_value);
                    $('#display_net_book_value').text(assetData.net_book_value);
                    $('#display_depreciation_percentage').text(assetData.depreciation_percentage + '%');
                    
                    // Convert currency display
                    __currency_convert_recursively($('#asset_info_section'));
                    
                    // Show asset info section
                    $('#asset_info_section').slideDown();
                    
                    // Auto-fill revaluation amount with current book value
                    $('#revaluation_amount').val(assetData.current_book_value);
                    
                    // Trigger currency formatting for the input
                    __currency_convert_recursively($('#revaluation_amount'));
                    
                    // Calculate and show difference
                    var revaluationAmount = parseFloat(assetData.current_book_value);
                    if (currentBookValue > 0 && revaluationAmount > 0) {
                        var difference = revaluationAmount - currentBookValue;
                        $('#revaluation_difference_value').text(difference);
                        __currency_convert_recursively($('#revaluation_difference_value'));
                        $('#revaluation_difference_display').show();
                        
                        if (difference >= 0) {
                            $('#revaluation_difference_value').removeClass('text-danger').addClass('text-success');
                        } else {
                            $('#revaluation_difference_value').removeClass('text-success').addClass('text-danger');
                        }
                    }
                }
                
                // Calculate and display revaluation difference in real-time
                $('#revaluation_amount').on('keyup blur', function() {
                    var revaluationAmount = parseFloat($(this).val()) || 0;
                    if (currentBookValue > 0 && revaluationAmount > 0) {
                        var difference = revaluationAmount - currentBookValue;
                        $('#revaluation_difference_value').text(difference);
                        __currency_convert_recursively($('#revaluation_difference_value'));
                        $('#revaluation_difference_display').show();
                        
                        // Color code: green for increase, red for decrease
                        if (difference >= 0) {
                            $('#revaluation_difference_value').removeClass('text-danger').addClass('text-success');
                        } else {
                            $('#revaluation_difference_value').removeClass('text-success').addClass('text-danger');
                        }
                    } else {
                        $('#revaluation_difference_display').hide();
                    }
                });
            });
        });
    });
</script>
@endsection

