@extends('layouts.app')

@section('title', __('accounting-reports::lang.bank_book'))

@section('content')
<section class="content-header">
    <h1>@lang('accounting-reports::lang.bank_book')</h1>
</section>

<section class="content">
    <div class="box box-solid">
        <div class="box-header with-border">
            <h3 class="box-title">@lang('accounting-reports::lang.bank_book') @lang('accounting-reports::lang.tally_style')</h3>
            <div class="box-tools">
                <a href="{{ route('accounting-reports.cheque-book.index') }}" class="btn btn-sm btn-warning">
                    <i class="fa fa-book"></i> @lang('accounting-reports::lang.cheque_book')
                </a>
                <button type="button" class="btn btn-sm btn-success" id="export_excel">
                    <i class="fa fa-file-excel-o"></i> @lang('daybook::lang.excel')
                </button>
                <button type="button" class="btn btn-sm btn-info" onclick="window.print()">
                    <i class="fa fa-print"></i> @lang('messages.print')
                </button>
            </div>
        </div>

        <div class="box-body">
            <div class="row">
                <!-- Date Range Picker -->
                <div class="col-md-4">
                    <div class="form-group">
                        {!! Form::label('date_range', __('report.date_range') . ':') !!}
                        <div class="input-group">
                            {!! Form::text('date_range', null, [
                                'class' => 'form-control', 
                                'id' => 'date_range', 
                                'readonly', 
                                'placeholder' => __('lang_v1.select_a_date_range')
                            ]) !!}
                            <span class="input-group-addon">
                                <i class="fa fa-calendar"></i>
                            </span>
                        </div>
                    </div>
                </div>
                <!-- Quick Date Filters -->
                <div class="col-md-8">
                    <div class="form-group">
                        {!! Form::label('quick_dates', __('daybook::lang.quick_filters') . ':') !!}
                        <div class="btn-group" style="display: flex; flex-wrap: wrap; gap: 5px;">
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="today">@lang('daybook::lang.today')</button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="yesterday">@lang('daybook::lang.yesterday')</button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="this_month">@lang('daybook::lang.this_month')</button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="last_month">@lang('daybook::lang.last_month')</button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="last_7_days">@lang('daybook::lang.last_7_days')</button>
                            <button type="button" class="btn btn-sm btn-default quick-date" data-range="last_30_days">@lang('daybook::lang.last_30_days')</button>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Hidden date fields for backend compatibility -->
            {!! Form::hidden('start_date', '', ['id' => 'start_date']) !!}
            {!! Form::hidden('end_date', '', ['id' => 'end_date']) !!}
            
            <div class="row">
                <div class="col-md-3">
                    <div class="form-group">
                        {!! Form::label('account_id', __('accounting-reports::lang.bank_account') . ':') !!}
                        {!! Form::select('account_id', $accounts, null, 
                            ['class' => 'form-control select2', 'id' => 'account_id', 'placeholder' => __('messages.all')]) 
                        !!}
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="form-group">
                        {!! Form::label('location_id', __('business.business_location') . ':') !!}
                        {!! Form::select('location_id', $locations, null, 
                            ['class' => 'form-control select2', 'id' => 'location_id', 'placeholder' => __('messages.all')]) 
                        !!}
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="form-group">
                        {!! Form::label('user_id', __('daybook::lang.user') . ':') !!}
                        {!! Form::select('user_id', $users, null, 
                            ['class' => 'form-control select2', 'id' => 'user_id', 'placeholder' => __('messages.all')]) 
                        !!}
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="form-group">
                        <br>
                        <button type="button" class="btn btn-primary" id="filter_btn">
                            <i class="fa fa-filter"></i> @lang('daybook::lang.filter')
                        </button>
                        <button type="button" class="btn btn-default" id="reset_btn">
                            <i class="fa fa-refresh"></i> @lang('daybook::lang.reset')
                        </button>
                    </div>
                </div>
            </div>

            <style>
                /* Excel-like minimal table styling */
                #bankbook_table {
                    border-collapse: collapse;
                    font-size: 12px;
                }
                #bankbook_table thead th {
                    background-color: #f2f2f2;
                    border: 1px solid #d0d0d0;
                    padding: 4px 6px;
                    font-weight: 600;
                    text-align: center;
                    font-size: 11px;
                }
                #bankbook_table tbody td {
                    padding: 3px 6px;
                    border: 1px solid #e0e0e0;
                    vertical-align: middle;
                }
                #bankbook_table tbody tr:hover {
                    background-color: #f9f9f9;
                }
                #bankbook_table tfoot th {
                    padding: 4px 6px;
                    border: 1px solid #d0d0d0;
                    font-size: 11px;
                }
                #bankbook_table tfoot tr:last-child {
                    border-top: 2px solid #333;
                }
            </style>
            
            <div class="table-responsive" style="margin-top: 15px;">
                <table class="table" id="bankbook_table" style="width: 100%; border-collapse: collapse;">
                    <thead>
                        <tr>
                            <th>@lang('daybook::lang.datetime')</th>
                            <th>@lang('daybook::lang.voucher_no')</th>
                            <th>@lang('daybook::lang.module')</th>
                            <th>@lang('daybook::lang.party')</th>
                            <th>@lang('daybook::lang.location')</th>
                            <th>@lang('daybook::lang.account')</th>
                            <th class="text-right">@lang('accounting-reports::lang.deposit') (Dr)</th>
                            <th class="text-right">@lang('accounting-reports::lang.withdrawal') (Cr)</th>
                            <th>@lang('accounting-reports::lang.cheque_no')</th>
                            <th>@lang('accounting-reports::lang.reconciled')</th>
                            <th>@lang('daybook::lang.narration')</th>
                            <th>@lang('daybook::lang.user')</th>
                        </tr>
                    </thead>
                    <tbody></tbody>
                    <tfoot>
                        <tr>
                            <th colspan="6" class="text-right">@lang('daybook::lang.total'):</th>
                            <th class="text-right" id="total_debit">0.00</th>
                            <th class="text-right" id="total_credit">0.00</th>
                            <th colspan="4"></th>
                        </tr>
                        <tr>
                            <th colspan="6" class="text-right" style="font-weight: 600;">Opening Balance:</th>
                            <th class="text-right" id="opening_balance_debit" style="font-weight: 600;">0.00</th>
                            <th class="text-right" id="opening_balance_credit" style="font-weight: 600;">-</th>
                            <th colspan="4"></th>
                        </tr>
                        <tr>
                            <th colspan="6" class="text-right" style="font-weight: 600;">Current Total:</th>
                            <th class="text-right" id="current_total_debit" style="font-weight: 600;">0.00</th>
                            <th class="text-right" id="current_total_credit" style="font-weight: 600;">0.00</th>
                            <th colspan="4"></th>
                        </tr>
                        <tr style="border-top: 2px solid #333;">
                            <th colspan="6" class="text-right" style="font-weight: 700;">Closing Balance:</th>
                            <th class="text-right" id="closing_balance_debit" style="font-weight: 700; font-size: 12px;">0.00</th>
                            <th class="text-right" id="closing_balance_credit" style="font-weight: 700; font-size: 12px;">-</th>
                            <th colspan="4"></th>
                        </tr>
                    </tfoot>
                </table>
            </div>
        </div>
    </div>
</section>
@endsection

@section('javascript')
<script type="text/javascript">
$(document).ready(function() {
    var bankbook_table;
    
    // Get account_id from URL parameter if present
    var urlParams = new URLSearchParams(window.location.search);
    var urlAccountId = urlParams.get('account_id');
    
    // Set account_id in dropdown if present in URL
    if (urlAccountId) {
        // Wait for select2 to be initialized
        setTimeout(function() {
            $('#account_id').val(urlAccountId).trigger('change.select2');
        }, 100);
    }
    
    // Initialize date range picker (same logic as cash book)
    if (typeof dateRangeSettings !== 'undefined' && $('#date_range').length && typeof moment !== 'undefined') {
        var today = moment();
        var todayFormatted = today.format(moment_date_format);
        $('#date_range').val(todayFormatted + ' ~ ' + todayFormatted);
        $('#start_date').val(todayFormatted);
        $('#end_date').val(todayFormatted);
        
        $('#date_range').daterangepicker(
            {
                ...dateRangeSettings,
                startDate: today,
                endDate: today,
                autoUpdateInput: true
            },
            function (start, end) {
                var startFormatted = start.format(moment_date_format);
                var endFormatted = end.format(moment_date_format);
                $('#date_range').val(startFormatted + ' ~ ' + endFormatted);
                $('#start_date').val(startFormatted);
                $('#end_date').val(endFormatted);
                if (typeof bankbook_table !== 'undefined' && bankbook_table) {
                    bankbook_table.ajax.reload();
                }
            }
        );
    }
    
    // Quick date filters
    $('.quick-date').click(function() {
        var range = $(this).data('range');
        var start, end;
        if (typeof moment === 'undefined') return;
        
        switch(range) {
            case 'today': start = end = moment(); break;
            case 'yesterday': start = end = moment().subtract(1, 'days'); break;
            case 'this_month': start = moment().startOf('month'); end = moment().endOf('month'); break;
            case 'last_month': start = moment().subtract(1, 'month').startOf('month'); end = moment().subtract(1, 'month').endOf('month'); break;
            case 'last_7_days': start = moment().subtract(6, 'days'); end = moment(); break;
            case 'last_30_days': start = moment().subtract(29, 'days'); end = moment(); break;
            default: start = end = moment();
        }
        
        if ($('#date_range').length && $('#date_range').data('daterangepicker')) {
            $('#date_range').data('daterangepicker').setStartDate(start);
            $('#date_range').data('daterangepicker').setEndDate(end);
            $('#date_range').val(start.format(moment_date_format) + ' ~ ' + end.format(moment_date_format));
        }
        $('#start_date').val(start.format(moment_date_format));
        $('#end_date').val(end.format(moment_date_format));
        bankbook_table.ajax.reload();
    });
    
    // Initialize DataTable
    bankbook_table = $('#bankbook_table').DataTable({
        processing: true,
        serverSide: true,
        ajax: {
            url: '{{ route("accounting-reports.bank-book.get-data") }}',
            type: 'GET',
            data: function(d) {
                d.start_date = $('#start_date').val();
                d.end_date = $('#end_date').val();
                d.account_id = $('#account_id').val() || urlAccountId || null;
                d.location_id = $('#location_id').val();
                d.user_id = $('#user_id').val();
            },
            dataSrc: function(json) {
                // Store opening balance for footer callback
                if (json.opening_balance !== undefined) {
                    bankbook_table.opening_balance = json.opening_balance;
                }
                if (json.current_period_debit !== undefined) {
                    bankbook_table.current_period_debit = json.current_period_debit;
                }
                if (json.current_period_credit !== undefined) {
                    bankbook_table.current_period_credit = json.current_period_credit;
                }
                if (json.closing_balance !== undefined) {
                    bankbook_table.closing_balance = json.closing_balance;
                }
                return json.data;
            },
            error: function(xhr, error, thrown) {
                console.error('Bank Book AJAX Error:', error);
                console.error('Response:', xhr.responseText);
                alert('Error loading bank book data.');
            }
        },
        columns: [
            { data: 'datetime', name: 'datetime' },
            { data: 'voucher_no', name: 'voucher_no' },
            { data: 'module', name: 'module' },
            { data: 'party', name: 'party' },
            { data: 'location', name: 'location' },
            { data: 'account', name: 'account' },
            { data: 'debit', name: 'debit', className: 'text-right' },
            { data: 'credit', name: 'credit', className: 'text-right' },
            { data: 'cheque_no', name: 'cheque_no' },
            { data: 'is_reconciled', name: 'is_reconciled' },
            { data: 'narration', name: 'narration' },
            { data: 'user', name: 'user' }
        ],
        order: [[0, 'asc']],
        paging: false,
        pageLength: -1,
        footerCallback: function (row, data, start, end, display) {
            var api = this.api();
            var json = api.ajax.json();
            
            // Try multiple ways to get the opening balance
            var opening_balance = json.opening_balance || 
                                 (bankbook_table.opening_balance !== undefined ? bankbook_table.opening_balance : 0) ||
                                 0;
            var current_period_debit = json.current_period_debit || 
                                      (bankbook_table.current_period_debit !== undefined ? bankbook_table.current_period_debit : 0) ||
                                      0;
            var current_period_credit = json.current_period_credit || 
                                       (bankbook_table.current_period_credit !== undefined ? bankbook_table.current_period_credit : 0) ||
                                       0;
            var closing_balance = json.closing_balance || 
                                 (bankbook_table.closing_balance !== undefined ? bankbook_table.closing_balance : 0) ||
                                 0;
            
            // Debug logging
            console.log('Bank Book Footer Callback:', {
                opening_balance: opening_balance,
                current_period_debit: current_period_debit,
                current_period_credit: current_period_credit,
                closing_balance: closing_balance,
                json: json
            });
            
            var total_debit = 0, total_credit = 0;
            
            api.column(6, {page: 'current'}).data().each(function (value) {
                if (value && value !== '-') {
                    var num = parseFloat(String(value).replace(/<[^>]*>/g, '').replace(/[^\d.-]/g, '')) || 0;
                    total_debit += Math.abs(num);
                }
            });
            
            api.column(7, {page: 'current'}).data().each(function (value) {
                if (value && value !== '-') {
                    var num = parseFloat(String(value).replace(/<[^>]*>/g, '').replace(/[^\d.-]/g, '')) || 0;
                    total_credit += Math.abs(num);
                }
            });
            
            $('#total_debit').html(__currency_trans_from_en(total_debit, true));
            $('#total_credit').html(__currency_trans_from_en(total_credit, true));
            $('#opening_balance_debit').html(__currency_trans_from_en(Math.abs(opening_balance), true));
            $('#opening_balance_credit').html('-');
            $('#current_total_debit').html(__currency_trans_from_en(current_period_debit || total_debit, true));
            $('#current_total_credit').html(__currency_trans_from_en(current_period_credit || total_credit, true));
            $('#closing_balance_debit').html(__currency_trans_from_en(Math.abs(closing_balance), true));
            $('#closing_balance_credit').html('-');
        },
        dom: 'Bfrtip',
        buttons: [
            { extend: 'excel', text: '<i class="fa fa-file-excel-o"></i> Excel', className: 'btn-success' },
            { extend: 'print', text: '<i class="fa fa-print"></i> Print', className: 'btn-info' }
        ]
    });
    
    $('#filter_btn').click(function() {
        var startVal = $('#start_date').val().trim();
        var endVal = $('#end_date').val().trim();
        if (!startVal && !endVal && typeof moment !== 'undefined') {
            var today = moment();
            var todayFormatted = today.format(moment_date_format);
            $('#start_date').val(todayFormatted);
            $('#end_date').val(todayFormatted);
            if ($('#date_range').length && $('#date_range').data('daterangepicker')) {
                $('#date_range').data('daterangepicker').setStartDate(today);
                $('#date_range').data('daterangepicker').setEndDate(today);
                $('#date_range').val(todayFormatted + ' ~ ' + todayFormatted);
            }
        }
        bankbook_table.ajax.reload();
    });
    
    $('#reset_btn').click(function() {
        if (typeof moment !== 'undefined') {
            var today = moment();
            var todayFormatted = today.format(moment_date_format);
            $('#start_date').val(todayFormatted);
            $('#end_date').val(todayFormatted);
            if ($('#date_range').length && $('#date_range').data('daterangepicker')) {
                $('#date_range').data('daterangepicker').setStartDate(today);
                $('#date_range').data('daterangepicker').setEndDate(today);
                $('#date_range').val(todayFormatted + ' ~ ' + todayFormatted);
            }
        }
        $('#account_id, #location_id, #user_id').val('').trigger('change');
        bankbook_table.ajax.reload();
    });
    
    $('#account_id, #location_id, #user_id').change(function() {
        bankbook_table.ajax.reload();
    });
    
    $('#export_excel').click(function() {
        bankbook_table.button('.buttons-excel').trigger();
    });
});
</script>
@endsection
