@extends('layouts.app')

@section('title', __('accounting-reports::lang.balance_sheet'))

@section('content')
<section class="content-header">
    <h1>@lang('accounting-reports::lang.balance_sheet')</h1>
</section>

<section class="content">
    <div class="row">
        <div class="col-md-12">
            <div class="box box-solid">
                <div class="box-header with-border">
                    <h3 class="box-title">Filters</h3>
                </div>
                <div class="box-body">
                    <form id="filter-form">
                        <div class="row">
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label>@lang('accounting-reports::lang.location')</label>
                                    <select name="location_id" id="location_id" class="form-control select2" style="width:100%">
                                        <option value="">@lang('accounting-reports::lang.all_locations')</option>
                                        @foreach($locations as $locationId => $locationName)
                                            <option value="{{ $locationId }}">{{ $locationName }}</option>
                                        @endforeach
                                    </select>
                                </div>
                            </div>
                            <div class="col-sm-3 col-xs-6">
                                <div class="form-group">
                                    <label for="end_date">@lang('accounting-reports::lang.as_on_date'):</label>
                                    <div class="input-group">
                                        <span class="input-group-addon">
                                            <i class="fa fa-calendar"></i>
                                        </span>
                                        <input type="text" id="end_date" value="{{ @format_date('now') }}" class="form-control" readonly>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label>&nbsp;</label><br>
                                    <button type="button" id="generate-btn" class="btn btn-primary">
                                        <i class="fa fa-refresh"></i> @lang('accounting-reports::lang.view')
                                    </button>
                                    <button type="button" id="export-btn" class="btn btn-success">
                                        <i class="fa fa-download"></i> @lang('accounting-reports::lang.export')
                                    </button>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-md-12">
            <div class="box box-solid">
                <div class="box-header with-border">
                    <h3 class="box-title">Balance Sheet</h3>
                </div>
                <div class="box-body">
                    <div id="loading" style="display:none; text-align:center; padding:20px;">
                        <i class="fa fa-spinner fa-spin fa-3x"></i>
                        <p>Loading balance sheet...</p>
                    </div>
                    <div id="balance-sheet-content" style="display:none;">
                        <div class="row">
                            <div class="col-md-12">
                                <h3 id="business-name" style="text-align:center; margin-bottom:5px; color:#0066cc; font-weight:bold;"></h3>
                                <h4 style="text-align:center; margin-top:0; margin-bottom:5px; color:#0066cc; font-weight:bold;">Balance Sheet</h4>
                                <p id="as-on-date" style="text-align:center; margin-bottom:20px; font-size:14px; color:#666;"></p>
                            </div>
                        </div>
                        <div class="table-responsive" style="border: 1px solid #ddd;">
                            <table class="table table-bordered" id="balance-sheet-table" style="margin-bottom:0; border-collapse: collapse;">
                                <thead>
                                    <tr>
                                        <th width="50%" style="text-align:center; background-color:#4a90e2; color:#fff; font-weight:bold; padding:12px; font-size:15px; border: 1px solid #357abd;">Liabilities</th>
                                        <th width="50%" style="text-align:center; background-color:#4a90e2; color:#fff; font-weight:bold; padding:12px; font-size:15px; border: 1px solid #357abd;">Assets</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td id="liabilities-column" style="vertical-align:top; padding:0; border: 1px solid #ddd;">
                                            <table class="table" style="margin:0; border:0; width:100%;">
                                                <tbody id="liabilities-tbody"></tbody>
                                            </table>
                                        </td>
                                        <td id="assets-column" style="vertical-align:top; padding:0; border: 1px solid #ddd;">
                                            <table class="table" style="margin:0; border:0; width:100%;">
                                                <tbody id="assets-tbody"></tbody>
                                            </table>
                                        </td>
                                    </tr>
                                </tbody>
                                <tfoot>
                                    <tr>
                                        <td style="text-align:right; font-weight:bold; padding:12px; background-color:#e8f4f8; border-top: 2px solid #4a90e2; font-size:15px;">
                                            <span id="total-liabilities"></span>
                                        </td>
                                        <td style="text-align:right; font-weight:bold; padding:12px; background-color:#e8f4f8; border-top: 2px solid #4a90e2; font-size:15px;">
                                            <span id="total-assets"></span>
                                        </td>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>
                    </div>
                    <div id="error-message" style="display:none;" class="alert alert-danger"></div>
                </div>
            </div>
        </div>
    </div>
</section>
@endsection

@section('javascript')
<script>
$(document).ready(function() {
    // Initialize date picker
    $('#end_date').datepicker({
        autoclose: true,
        format: datepicker_date_format
    });

    // Initialize select2
    $('#location_id').select2();

    // Load on page load
    loadBalanceSheet();

    $('#end_date, #location_id').change(function() {
        loadBalanceSheet();
    });

    $('#generate-btn').click(function() {
        loadBalanceSheet();
    });

    $('#export-btn').click(function() {
        exportBalanceSheet();
    });

    function loadBalanceSheet() {
        var location_id = $('#location_id').val();
        var end_date = $('#end_date').val();

        if (!end_date) {
            toastr.error('Please select as on date');
            return;
        }

        $('#loading').show();
        $('#balance-sheet-content').hide();
        $('#error-message').hide();

        $.ajax({
            url: "{{ action([\Modules\AccountingReports\Http\Controllers\BalanceSheetController::class, 'getData']) }}",
            data: {
                location_id: location_id,
                end_date: end_date
            },
            dataType: 'json',
            success: function(response) {
                $('#loading').hide();
                
                if (response.success === false) {
                    $('#error-message').text(response.error || 'Error loading balance sheet').show();
                    return;
                }

                if (response.liabilities && response.assets) {
                    displayBalanceSheet(response);
                    $('#balance-sheet-content').show();
                } else {
                    $('#error-message').text('No data available for the selected date').show();
                }
            },
            error: function(xhr, status, error) {
                $('#loading').hide();
                var errorMsg = 'Error loading balance sheet';
                if (xhr.responseJSON && xhr.responseJSON.error) {
                    errorMsg = xhr.responseJSON.error;
                }
                $('#error-message').text(errorMsg).show();
            }
        });
    }

    function displayBalanceSheet(data) {
        // Set business name and date
        $('#business-name').text(data.business_name || '');
        
        var asOnDate = data.as_of_date;
        var formattedDate = moment(asOnDate).format('DD-MMM-YYYY');
        $('#as-on-date').text('As at ' + formattedDate);

        // Clear existing content
        $('#liabilities-tbody').empty();
        $('#assets-tbody').empty();

        // Render Liabilities (Left Column)
        renderSection(data.liabilities, 'liabilities-tbody', true);

        // Render Assets (Right Column)
        renderSection(data.assets, 'assets-tbody', false);

        // Display totals
        $('#total-liabilities').html('<strong>Total: ' + formatCurrency(data.total_liabilities) + '</strong>');
        $('#total-assets').html('<strong>Total: ' + formatCurrency(data.total_assets) + '</strong>');

        // Show balance status
        var tfoot = $('#balance-sheet-table tfoot');
        tfoot.find('tr.balance-status').remove(); // Remove existing status row
        
        if (data.is_balanced) {
            tfoot.append('<tr class="balance-status"><td colspan="2" style="text-align:center; color:#28a745; font-weight:bold; padding:8px; background-color:#d4edda; border-top: 2px solid #28a745;">✓ Balance Sheet is Balanced</td></tr>');
        } else {
            var diff = Math.abs(data.total_liabilities - data.total_assets);
            tfoot.append('<tr class="balance-status"><td colspan="2" style="text-align:center; color:#dc3545; font-weight:bold; padding:8px; background-color:#f8d7da; border-top: 2px solid #dc3545;">⚠ Difference: ' + formatCurrency(diff) + '</td></tr>');
        }
    }

    function renderSection(sections, tbodyId, isLiabilities) {
        var tbody = $('#' + tbodyId);
        
        $.each(sections, function(index, section) {
            // Section header (Main Group)
            var headerRow = $('<tr>');
            headerRow.append($('<td>').html('<strong>' + section.label + '</strong>').css({
                'font-weight': 'bold',
                'background-color': '#f5f5f5',
                'padding': '10px 8px',
                'border-bottom': '2px solid #4a90e2',
                'border-top': index === 0 ? '1px solid #ddd' : '2px solid #4a90e2',
                'font-size': '14px',
                'color': '#333'
            }));
            headerRow.append($('<td>').html('<strong>' + formatCurrency(section.total) + '</strong>').css({
                'text-align': 'right',
                'font-weight': 'bold',
                'background-color': '#f5f5f5',
                'padding': '10px 8px',
                'border-bottom': '2px solid #4a90e2',
                'border-top': index === 0 ? '1px solid #ddd' : '2px solid #4a90e2',
                'font-size': '14px',
                'color': '#333'
            }));
            tbody.append(headerRow);

            // Track which items are already in sub-groups
            var itemsInSubGroups = [];
            if (section.sub_groups) {
                $.each(section.sub_groups, function(subGroupName, subGroupItems) {
                    if (subGroupItems && subGroupItems.length > 0) {
                        $.each(subGroupItems, function(idx, subItem) {
                            itemsInSubGroups.push(subItem.name);
                        });
                    }
                });
            }

            // Display Sub-groups first (e.g., Secured Loans, Unsecured Loans, Duties & Taxes, etc.)
            if (section.sub_groups) {
                $.each(section.sub_groups, function(subGroupName, subGroupItems) {
                    if (subGroupItems && subGroupItems.length > 0) {
                        // Calculate sub-group total (preserve negative values)
                        var subTotal = subGroupItems.reduce(function(sum, item) { 
                            return sum + (parseFloat(item.value) || 0); 
                        }, 0);
                        
                        // Sub-group header
                        var subHeaderRow = $('<tr>');
                        var subGroupDisplayName = subGroupName;
                        var subGroupDisplayValue = subTotal;
                        var isSubGroupNegative = subTotal < 0;
                        
                        if (isSubGroupNegative) {
                            subGroupDisplayName = '(-)' + subGroupName;
                            subGroupDisplayValue = Math.abs(subTotal);
                        }
                        
                        subHeaderRow.append($('<td>').html('&nbsp;&nbsp;&nbsp;&nbsp;<strong>' + subGroupDisplayName + '</strong>').css({
                            'font-weight': 'bold',
                            'padding': '8px 5px',
                            'background-color': '#fafafa',
                            'border-left': '3px solid #4a90e2',
                            'font-size': '13px'
                        }));
                        
                        var subTotalCell = $('<td>').html('<strong>' + formatCurrency(subGroupDisplayValue) + '</strong>').css({
                            'text-align': 'right',
                            'font-weight': 'bold',
                            'padding': '8px 5px',
                            'background-color': '#fafafa',
                            'font-size': '13px'
                        });
                        
                        if (isSubGroupNegative) {
                            subTotalCell.css('color', '#d9534f');
                        }
                        
                        subHeaderRow.append(subTotalCell);
                        tbody.append(subHeaderRow);

                        // Sub-group items with proper indentation
                        $.each(subGroupItems, function(subItemIndex, subItem) {
                            var subItemRow = $('<tr>');
                            
                            // Item name with deeper indentation
                            var itemName = subItem.name;
                            var itemValue = parseFloat(subItem.value) || 0;
                            
                            // Handle negative values
                            var displayValue = itemValue;
                            var isNegative = itemValue < 0;
                            if (isNegative) {
                                displayValue = Math.abs(itemValue);
                                itemName = '(-)' + itemName;
                            }
                            
                            var nameCell = $('<td>').html('&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;' + itemName).css({
                                'padding': '6px 5px',
                                'border-left': '3px solid #ddd',
                                'font-size': '12px'
                            });
                            
                            // Highlight closing stock if needed
                            if (subItem.highlight) {
                                nameCell.css('background-color', '#fff5f5');
                            }
                            
                            subItemRow.append(nameCell);
                            
                            // Item value
                            var valueCell = $('<td>').html(formatCurrency(displayValue)).css({
                                'text-align': 'right',
                                'padding': '6px 5px',
                                'font-size': '12px'
                            });
                            
                            if (isNegative) {
                                valueCell.css('color', '#d9534f');
                            }
                            
                            if (subItem.highlight) {
                                valueCell.css('background-color', '#fff5f5');
                            }
                            
                            subItemRow.append(valueCell);
                            tbody.append(subItemRow);
                        });
                    }
                });
            }

            // Display items that are NOT in sub-groups (direct items under the main group)
            if (section.items && section.items.length > 0) {
                $.each(section.items, function(itemIndex, item) {
                    // Skip if this item is already in a sub-group
                    if (itemsInSubGroups.indexOf(item.name) !== -1) {
                        return true; // continue
                    }
                    
                    var itemRow = $('<tr>');
                    
                    // Item name with indentation
                    var itemName = item.name;
                    var itemValue = parseFloat(item.value) || 0;
                    
                    // Special handling for P&L items that should not show values
                    var showValue = item.show_value !== false;
                    
                    // Handle negative values
                    var displayValue = itemValue;
                    var isNegative = itemValue < 0;
                    if (isNegative) {
                        displayValue = Math.abs(itemValue);
                        itemName = '(-)' + itemName;
                    }
                    
                    var nameCell = $('<td>').html('&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;' + itemName).css({
                        'padding': '6px 5px',
                        'border-left': '3px solid #ddd',
                        'font-size': '12px'
                    });
                    
                    // Show depreciation rate if available
                    if (item.depreciation_rate) {
                        nameCell.html('&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;' + itemName + ' (' + item.depreciation_rate + ')');
                    }
                    
                    // Highlight closing stock if needed
                    if (item.highlight) {
                        nameCell.css('background-color', '#fff5f5');
                    }
                    
                    itemRow.append(nameCell);
                    
                    // Item value - only show if showValue is true
                    var valueCell = $('<td>').html(showValue ? formatCurrency(displayValue) : '').css({
                        'text-align': 'right',
                        'padding': '6px 5px',
                        'font-size': '12px'
                    });
                    
                    if (isNegative && showValue) {
                        valueCell.css('color', '#d9534f');
                    }
                    
                    if (item.highlight) {
                        valueCell.css('background-color', '#fff5f5');
                    }
                    
                    itemRow.append(valueCell);
                    tbody.append(itemRow);
                });
            }
            
            // Special handling for P&L: Show total value after items if show_items_without_values flag is set
            if (section.show_items_without_values && section.items && section.items.length > 0) {
                var totalRow = $('<tr>');
                var totalNameCell = $('<td>').html('&nbsp;').css({
                    'padding': '6px 5px',
                    'border-left': '3px solid #ddd',
                    'font-size': '12px'
                });
                totalRow.append(totalNameCell);
                
                var totalValueCell = $('<td>').html(formatCurrency(section.total)).css({
                    'text-align': 'right',
                    'padding': '6px 5px',
                    'font-size': '12px',
                    'font-weight': 'bold'
                });
                totalRow.append(totalValueCell);
                tbody.append(totalRow);
            }

            // Empty row for spacing between groups (only if not last section)
            if (index < sections.length - 1) {
                tbody.append($('<tr>').append($('<td>').html('&nbsp;').attr('colspan', 2).css({
                    'height': '8px',
                    'padding': '0',
                    'border': 'none'
                })));
            }
        });
    }

    function formatCurrency(amount) {
        if (typeof amount === 'undefined' || amount === null) {
            return '0.00';
        }
        var num = parseFloat(amount);
        if (isNaN(num)) {
            return '0.00';
        }
        // Format with commas and 2 decimal places
        return num.toLocaleString('en-IN', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
    }

    function exportBalanceSheet() {
        var location_id = $('#location_id').val();
        var end_date = $('#end_date').val();

        if (!end_date) {
            toastr.error('Please select as on date');
            return;
        }

        var url = "{{ action([\Modules\AccountingReports\Http\Controllers\BalanceSheetController::class, 'export']) }}";
        url += '?location_id=' + location_id + '&end_date=' + end_date;
        
        window.location.href = url;
    }
});
</script>
@endsection
