<?php

namespace Modules\AccountingReports\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StoreBankAccountRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return auth()->user()->can('accounting.view_all');
    }

    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $businessId = auth()->user()->business_id;

        return [
            'account_name' => [
                'required',
                'string',
                'max:255',
                'unique:ar_bank_accounts,account_name,NULL,id,business_id,' . $businessId . ',deleted_at,NULL'
            ],
            'account_type' => 'required|in:liability,asset',
            'bank_name' => 'nullable|string|max:255',
            'account_number' => 'nullable|string|max:255',
            'ifsc_code' => 'nullable|string|max:50',
            'branch_name' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'linked_account_id' => 'nullable|exists:accounts,id',
            'opening_balance' => 'nullable|numeric|min:0',
            'opening_date' => 'nullable|date',
            'is_active' => 'nullable|boolean',
        ];
    }

    /**
     * Get custom messages for validator errors.
     */
    public function messages(): array
    {
        return [
            'account_name.required' => 'Account name is required.',
            'account_name.unique' => 'Account name already exists for this business.',
            'account_type.required' => 'Account type is required.',
            'account_type.in' => 'Account type must be either liability or asset.',
            'linked_account_id.exists' => 'Selected account does not exist.',
            'opening_balance.numeric' => 'Opening balance must be a number.',
            'opening_balance.min' => 'Opening balance cannot be negative.',
            'opening_date.date' => 'Opening date must be a valid date.',
        ];
    }
}

