<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Services\TrialBalanceService;
use Modules\AccountingReports\Exports\TrialBalanceExport;
use App\Utils\TransactionUtil;

class TrialBalanceController extends Controller
{
    protected $trialBalanceService;
    protected $transactionUtil;

    public function __construct(TrialBalanceService $trialBalanceService, TransactionUtil $transactionUtil)
    {
        $this->trialBalanceService = $trialBalanceService;
        $this->transactionUtil = $transactionUtil;
    }

    /**
     * Display trial balance page
     */
    public function index()
    {
        if (!auth()->user()->can('accounting.view_trial_balance')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $locations = \App\BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::trial-balance.index', compact('locations'));
    }

    /**
     * Get trial balance data via AJAX (matching UltimatePOS logic)
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_trial_balance')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $businessId = auth()->user()->business_id;
            
            // Use end_date (like UltimatePOS) or fallback to to_date/from_date
            $endDate = null;
            if ($request->has('end_date') && !empty($request->input('end_date'))) {
                try {
                    $endDate = $this->transactionUtil->uf_date($request->input('end_date'));
                    if (!$endDate) {
                        // Fallback: try parsing as Y-m-d format
                        $endDate = \Carbon\Carbon::createFromFormat('Y-m-d', $request->input('end_date'))->format('Y-m-d');
                    }
                } catch (\Exception $e) {
                    // If date parsing fails, use current date
                    $endDate = \Carbon\Carbon::now()->format('Y-m-d');
                }
            } elseif ($request->has('to_date') && !empty($request->input('to_date'))) {
                $endDate = $request->input('to_date');
            } else {
                $endDate = \Carbon\Carbon::now()->format('Y-m-d');
            }
            
            $locationId = $request->input('location_id');

            // Pass end_date as both from and to (UltimatePOS uses single end_date)
            $data = $this->trialBalanceService->getTrialBalance($businessId, $endDate, $endDate, $locationId);

            return response()->json($data);
        } catch (\Exception $e) {
            \Log::error('TrialBalance getData Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'error' => 'Error loading trial balance: ' . $e->getMessage(),
                'supplier_due' => 0,
                'customer_due' => 0,
                'account_balances' => [],
                'total_debit' => 0,
                'total_credit' => 0,
                'is_balanced' => true,
            ], 500);
        }
    }

    /**
     * Export trial balance
     */
    public function export(Request $request)
    {
        if (!auth()->user()->can('accounting.export_reports')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        
        // Use end_date (like UltimatePOS)
        $endDate = $request->input('end_date') 
            ? $this->transactionUtil->uf_date($request->input('end_date'))
            : \Carbon\Carbon::now()->format('Y-m-d');
        
        $locationId = $request->input('location_id');
        $format = $request->input('format', 'pdf');

        $data = $this->trialBalanceService->getTrialBalance($businessId, $endDate, $endDate, $locationId);

        $export = new TrialBalanceExport($data, $endDate, $endDate);

        if ($format === 'excel') {
            return $export->download('trial-balance-' . date('Y-m-d') . '.xlsx');
        } elseif ($format === 'csv') {
            return $export->download('trial-balance-' . date('Y-m-d') . '.csv');
        } else {
            return $export->download('trial-balance-' . date('Y-m-d') . '.pdf');
        }
    }
}

