<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Services\RatioAnalysisService;
use App\Utils\BusinessUtil;
use App\Utils\TransactionUtil;
use App\BusinessLocation;

class RatioAnalysisController extends Controller
{
    protected $ratioAnalysisService;
    protected $businessUtil;
    protected $transactionUtil;

    public function __construct(RatioAnalysisService $ratioAnalysisService, BusinessUtil $businessUtil, TransactionUtil $transactionUtil)
    {
        $this->ratioAnalysisService = $ratioAnalysisService;
        $this->businessUtil = $businessUtil;
        $this->transactionUtil = $transactionUtil;
    }

    public function index()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::ratio-analysis.index', compact('locations'));
    }

    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $businessId = auth()->user()->business_id;
            
            $fy = $this->businessUtil->getCurrentFinancialYear($businessId);
            
            $locationId = !empty($request->input('location_id')) ? $request->input('location_id') : null;
            
            // Parse dates using UltimatePOS date utility
            $startDate = !empty($request->input('start_date')) 
                ? $this->transactionUtil->uf_date($request->input('start_date'), false) 
                : $fy['start'];
            $endDate = !empty($request->input('end_date')) 
                ? $this->transactionUtil->uf_date($request->input('end_date'), false) 
                : $fy['end'];

            // Ensure dates are in Y-m-d format
            if ($startDate instanceof \Carbon\Carbon) {
                $startDate = $startDate->format('Y-m-d');
            }
            if ($endDate instanceof \Carbon\Carbon) {
                $endDate = $endDate->format('Y-m-d');
            }

            $data = $this->ratioAnalysisService->calculateRatios($businessId, $startDate, $endDate, $locationId);

            // Get business name
            $business = \App\Business::find($businessId);
            $data['business_name'] = $business->name ?? '';

            return response()->json($data);
        } catch (\Exception $e) {
            \Log::error('RatioAnalysis getData Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'error' => 'Error generating Ratio Analysis: ' . $e->getMessage(),
                'ratios' => [],
                'period' => [],
                'summary' => []
            ], 500);
        }
    }

    public function export(Request $request)
    {
        if (!auth()->user()->can('accounting.export_reports')) {
            abort(403, 'Unauthorized action.');
        }

        // TODO: Implement export to PDF/Excel
        return redirect()->back();
    }
}


