<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Entities\Loan;
use Modules\AccountingReports\Entities\LoanTransaction;
use App\BusinessLocation;
use App\Account;
use App\AccountTransaction;
use App\AccountType;
use App\Utils\Util;
use App\Utils\TransactionUtil;
use App\Utils\ModuleUtil;
use DB;
use Yajra\DataTables\Facades\DataTables;

class LoanController extends Controller
{
    protected $commonUtil;
    protected $transactionUtil;
    protected $moduleUtil;

    public function __construct(Util $commonUtil, TransactionUtil $transactionUtil, ModuleUtil $moduleUtil)
    {
        $this->commonUtil = $commonUtil;
        $this->transactionUtil = $transactionUtil;
        $this->moduleUtil = $moduleUtil;
    }

    /**
     * Display a listing of loans
     */
    public function index()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $locations = BusinessLocation::forDropdown($business_id, true);

        return view('accounting-reports::loans.index', compact('locations'));
    }

    /**
     * Get loans data for DataTables
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;

        $query = Loan::with(['location', 'liabilityAccount', 'creator'])
            ->where('business_id', $business_id);

        // Apply filters
        if ($request->has('location_id') && !empty($request->location_id)) {
            $query->where('location_id', $request->location_id);
        }

        if ($request->has('status') && !empty($request->status)) {
            $query->where('status', $request->status);
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $html = '<div class="btn-group">';
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-info btn-xs view_loan" data-href="' . 
                        action([self::class, 'show'], [$row->id]) . '"><i class="fa fa-eye"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-success btn-xs ledger_loan" data-href="' . 
                        action([self::class, 'ledger'], [$row->id]) . '"><i class="fa fa-book"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-primary btn-xs edit_loan" data-href="' . 
                        action([self::class, 'edit'], [$row->id]) . '"><i class="fa fa-edit"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-danger btn-xs delete_loan" data-href="' . 
                        action([self::class, 'destroy'], [$row->id]) . '"><i class="fa fa-trash"></i></button>';
                }
                
                $html .= '</div>';
                return $html;
            })
            ->editColumn('loan_date', function ($row) {
                return $this->commonUtil->format_date($row->loan_date, true);
            })
            ->editColumn('loan_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true">' . $row->loan_amount . '</span>';
            })
            ->editColumn('outstanding_principal', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true">' . $row->outstanding_principal . '</span>';
            })
            ->editColumn('status', function ($row) {
                $statusColors = [
                    'active' => 'success',
                    'closed' => 'default',
                    'defaulted' => 'danger'
                ];
                $color = $statusColors[$row->status] ?? 'default';
                return '<span class="label label-' . $color . '">' . ucfirst($row->status) . '</span>';
            })
            ->editColumn('liability_account', function ($row) {
                return $row->liabilityAccount ? $row->liabilityAccount->name : '-';
            })
            ->rawColumns(['action', 'loan_amount', 'outstanding_principal', 'status'])
            ->toJson();
    }

    /**
     * Show the form for creating a new loan
     */
    public function create()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $locations = BusinessLocation::forDropdown($business_id, true);
        $payment_accounts = [];
        
        if ($this->moduleUtil->isModuleEnabled('account')) {
            $payment_accounts = Account::forDropdown($business_id, true, false, true);
        }

        return view('accounting-reports::loans.create', compact('locations', 'payment_accounts'));
    }

    /**
     * Store a newly created loan
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $user_id = auth()->user()->id;

            $request->validate([
                'loan_name' => 'required|string|max:255',
                'loan_amount' => 'required|numeric|min:0',
                'loan_date' => 'required|date',
                'lender_name' => 'required|string|max:255',
                'location_id' => 'nullable|exists:business_locations,id',
                'payment_account_id' => 'nullable|exists:accounts,id',
                'interest_rate' => 'nullable|numeric|min:0|max:100',
                'maturity_date' => 'nullable|date|after:loan_date',
                'document' => 'nullable|file|max:' . (config('constants.document_size_limit', 10000) / 1000),
            ]);

            DB::beginTransaction();

            // Generate loan number
            $loanNumber = $this->generateLoanNumber($business_id);

            // Create liability account for the loan
            $liabilityAccount = $this->createLiabilityAccount($business_id, $loanNumber, $request->loan_name, $user_id);

            $data = $request->only([
                'loan_name', 'description', 'loan_amount', 'loan_date', 
                'maturity_date', 'interest_rate', 'interest_type',
                'lender_name', 'lender_contact', 'lender_address',
                'location_id', 'payment_account_id', 'reference_no'
            ]);

            $data['business_id'] = $business_id;
            $data['created_by'] = $user_id;
            $data['loan_number'] = $loanNumber;
            $data['liability_account_id'] = $liabilityAccount->id;
            $data['outstanding_principal'] = $this->commonUtil->num_uf($data['loan_amount']);
            
            // Format dates
            $data['loan_date'] = $this->commonUtil->uf_date($data['loan_date'], false);
            if (!empty($data['maturity_date'])) {
                $data['maturity_date'] = $this->commonUtil->uf_date($data['maturity_date'], false);
            }
            
            // Format amounts
            $data['loan_amount'] = $this->commonUtil->num_uf($data['loan_amount']);
            $data['interest_rate'] = $request->has('interest_rate') ? $this->commonUtil->num_uf($request->interest_rate) : 0;

            // Upload document if provided
            if ($request->hasFile('document')) {
                $document_name = $this->commonUtil->uploadFile($request, 'document', 'documents', 'document');
                if (!empty($document_name)) {
                    $data['document'] = $document_name;
                }
            }

            $loan = Loan::create($data);

            DB::commit();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.loan_added_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Display the specified loan
     */
    public function show($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $loan = Loan::where('business_id', $business_id)
            ->with(['location', 'liabilityAccount', 'paymentAccount', 'creator', 'transactions'])
            ->findOrFail($id);

        return view('accounting-reports::loans.show', compact('loan'));
    }

    /**
     * Show the form for editing the specified loan
     */
    public function edit($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $loan = Loan::where('business_id', $business_id)->findOrFail($id);
        
        $locations = BusinessLocation::forDropdown($business_id, true);
        $payment_accounts = [];
        
        if ($this->moduleUtil->isModuleEnabled('account')) {
            $payment_accounts = Account::forDropdown($business_id, true, false, true);
        }

        return view('accounting-reports::loans.edit', compact('loan', 'locations', 'payment_accounts'));
    }

    /**
     * Update the specified loan
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $loan = Loan::where('business_id', $business_id)->findOrFail($id);

            $request->validate([
                'loan_name' => 'required|string|max:255',
                'loan_amount' => 'required|numeric|min:0',
                'loan_date' => 'required|date',
                'lender_name' => 'required|string|max:255',
                'location_id' => 'nullable|exists:business_locations,id',
                'payment_account_id' => 'nullable|exists:accounts,id',
                'interest_rate' => 'nullable|numeric|min:0|max:100',
                'maturity_date' => 'nullable|date|after:loan_date',
                'document' => 'nullable|file|max:' . (config('constants.document_size_limit', 10000) / 1000),
            ]);

            $data = $request->only([
                'loan_name', 'description', 'loan_amount', 'loan_date', 
                'maturity_date', 'interest_rate', 'interest_type',
                'lender_name', 'lender_contact', 'lender_address',
                'location_id', 'payment_account_id', 'reference_no', 'status'
            ]);

            // Format dates
            $data['loan_date'] = $this->commonUtil->uf_date($data['loan_date'], false);
            if (!empty($data['maturity_date'])) {
                $data['maturity_date'] = $this->commonUtil->uf_date($data['maturity_date'], false);
            }
            
            // Format amounts
            $data['loan_amount'] = $this->commonUtil->num_uf($data['loan_amount']);
            $data['interest_rate'] = $request->has('interest_rate') ? $this->commonUtil->num_uf($request->interest_rate) : 0;

            // Upload document if provided
            if ($request->hasFile('document')) {
                // Delete old document if exists
                if ($loan->document && file_exists(public_path('uploads/documents/' . $loan->document))) {
                    @unlink(public_path('uploads/documents/' . $loan->document));
                }

                $document_name = $this->commonUtil->uploadFile($request, 'document', 'documents', 'document');
                if (!empty($document_name)) {
                    $data['document'] = $document_name;
                }
            }

            $loan->update($data);

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.loan_updated_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Remove the specified loan
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $loan = Loan::where('business_id', $business_id)->findOrFail($id);

            DB::beginTransaction();

            // Check if loan has transactions
            if ($loan->transactions()->count() > 0) {
                throw new \Exception('Cannot delete loan with existing transactions');
            }

            // Delete associated account transactions if any
            if ($loan->liability_account_id) {
                AccountTransaction::where('account_id', $loan->liability_account_id)
                    ->where('sub_type', 'loan')
                    ->delete();
            }

            // Delete document if exists
            if ($loan->document && file_exists(public_path('uploads/documents/' . $loan->document))) {
                @unlink(public_path('uploads/documents/' . $loan->document));
            }

            $loan->delete();

            DB::commit();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.loan_deleted_success')
            ];

            return response()->json($output);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => $e->getMessage() ?: __('messages.something_went_wrong')
            ];

            return response()->json($output);
        }
    }

    /**
     * Show loan ledger
     */
    public function ledger($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $loan = Loan::where('business_id', $business_id)
            ->with(['location', 'liabilityAccount', 'paymentAccount'])
            ->findOrFail($id);

        $transactions = LoanTransaction::where('loan_id', $loan->id)
            ->with(['paymentAccount', 'accountTransaction', 'creator'])
            ->orderBy('transaction_date', 'desc')
            ->orderBy('id', 'desc')
            ->get();

        // Format transactions for DataTable
        $formattedTransactions = [];
        foreach ($transactions as $transaction) {
            $formattedTransactions[] = [
                'id' => $transaction->id,
                'transaction_date' => $transaction->transaction_date,
                'transaction_number' => $transaction->transaction_number,
                'transaction_type' => $transaction->transaction_type,
                'principal_amount' => $transaction->principal_amount,
                'interest_amount' => $transaction->interest_amount,
                'total_amount' => $transaction->total_amount,
                'payment_account' => $transaction->paymentAccount,
                'reference_no' => $transaction->reference_no,
            ];
        }

        // Get payment accounts
        $payment_accounts = [];
        if ($this->moduleUtil->isModuleEnabled('account')) {
            $payment_accounts = Account::forDropdown($business_id, true, false, true);
        }

        // Get payment types
        $payment_types = $this->transactionUtil->payment_types(null, false, $business_id);

        return view('accounting-reports::loans.ledger', compact('loan', 'transactions', 'formattedTransactions', 'payment_accounts', 'payment_types'));
    }

    /**
     * Record loan receipt (take loan)
     */
    public function takeLoan(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $loan = Loan::where('business_id', $business_id)->findOrFail($id);

            $request->validate([
                'transaction_date' => 'required|date',
                'principal_amount' => 'required|numeric|min:0',
                'payment_account_id' => 'required|exists:accounts,id',
                'payment_method' => 'nullable|string',
                'reference_no' => 'nullable|string',
            ]);

            DB::beginTransaction();

            $user_id = auth()->user()->id;
            $transactionDate = $this->commonUtil->uf_date($request->transaction_date, false);
            $principalAmount = $this->commonUtil->num_uf($request->principal_amount);

            // Generate transaction number
            $transactionNumber = $this->generateTransactionNumber($business_id, 'LOAN-REC');

            // Create loan transaction
            $loanTransaction = LoanTransaction::create([
                'loan_id' => $loan->id,
                'business_id' => $business_id,
                'transaction_number' => $transactionNumber,
                'transaction_type' => 'loan_receipt',
                'transaction_date' => $transactionDate,
                'principal_amount' => $principalAmount,
                'interest_amount' => 0,
                'total_amount' => $principalAmount,
                'payment_account_id' => $request->payment_account_id,
                'payment_method' => $request->payment_method,
                'reference_no' => $request->reference_no,
                'notes' => $request->notes,
                'created_by' => $user_id,
            ]);

            // Create account transactions - Double entry accounting
            if ($this->moduleUtil->isModuleEnabled('account', $business_id)) {
                // 1. Credit payment account (Cash in Hand, Bank, etc.) - Money coming in
                if (!empty($request->payment_account_id)) {
                    $payment_account_data = [
                        'amount' => $principalAmount,
                        'account_id' => $request->payment_account_id,
                        'type' => 'credit', // Loan receipt = credit to asset account (increases cash/bank)
                        'operation_date' => $transactionDate,
                        'created_by' => $user_id,
                        'sub_type' => 'loan_receipt',
                        'note' => 'Loan Receipt: ' . $loan->loan_name . ' - ' . $transactionNumber,
                    ];
                    
                    AccountTransaction::createAccountTransaction($payment_account_data);
                }

                // 2. Credit liability account (loan received) - Increasing loan liability
                $account_transaction_data = [
                    'amount' => $principalAmount,
                    'account_id' => $loan->liability_account_id,
                    'type' => 'credit', // Loan received = credit to liability (increases liability)
                    'operation_date' => $transactionDate,
                    'created_by' => $user_id,
                    'sub_type' => 'loan',
                    'note' => 'Loan Receipt: ' . $loan->loan_name . ' - ' . $transactionNumber,
                ];
                
                $accountTransaction = AccountTransaction::createAccountTransaction($account_transaction_data);
                $loanTransaction->account_transaction_id = $accountTransaction->id;
                $loanTransaction->save();
            }

            // Update loan outstanding
            $loan->outstanding_principal += $principalAmount;
            $loan->save();

            DB::commit();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.loan_receipt_recorded_success')
            ];

            return response()->json($output);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            return response()->json($output);
        }
    }

    /**
     * Record loan repayment
     */
    public function repayment(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $loan = Loan::where('business_id', $business_id)->findOrFail($id);

            $request->validate([
                'transaction_date' => 'required|date',
                'principal_amount' => 'nullable|numeric|min:0',
                'interest_amount' => 'nullable|numeric|min:0',
                'payment_account_id' => 'required|exists:accounts,id',
                'payment_method' => 'nullable|string',
                'reference_no' => 'nullable|string',
            ]);

            $principalAmount = $this->commonUtil->num_uf($request->principal_amount ?? 0);
            $interestAmount = $this->commonUtil->num_uf($request->interest_amount ?? 0);
            $totalAmount = $principalAmount + $interestAmount;

            if ($totalAmount <= 0) {
                throw new \Exception('Total repayment amount must be greater than zero');
            }

            // Check if repayment exceeds outstanding
            if ($principalAmount > $loan->outstanding_principal) {
                throw new \Exception('Principal repayment cannot exceed outstanding principal');
            }

            DB::beginTransaction();

            $user_id = auth()->user()->id;
            $transactionDate = $this->commonUtil->uf_date($request->transaction_date, false);

            // Determine transaction type
            $transactionType = 'principal_repayment';
            if ($interestAmount > 0 && $principalAmount > 0) {
                $transactionType = 'full_repayment';
            } elseif ($interestAmount > 0) {
                $transactionType = 'interest_payment';
            }

            // Generate transaction number
            $transactionNumber = $this->generateTransactionNumber($business_id, 'LOAN-PAY');

            // Create loan transaction
            $loanTransaction = LoanTransaction::create([
                'loan_id' => $loan->id,
                'business_id' => $business_id,
                'transaction_number' => $transactionNumber,
                'transaction_type' => $transactionType,
                'transaction_date' => $transactionDate,
                'principal_amount' => $principalAmount,
                'interest_amount' => $interestAmount,
                'total_amount' => $totalAmount,
                'payment_account_id' => $request->payment_account_id,
                'payment_method' => $request->payment_method,
                'cheque_number' => $request->cheque_number,
                'bank_name' => $request->bank_name,
                'reference_no' => $request->reference_no,
                'notes' => $request->notes,
                'created_by' => $user_id,
            ]);

            // Create account transactions - Double entry accounting
            if ($this->moduleUtil->isModuleEnabled('account', $business_id)) {
                // 1. Debit payment account (Cash in Hand, Bank, etc.) - Money going out
                // In UltimatePOS: balance = SUM(IF(type='credit', amount, -1*amount))
                // So: debit = -amount (reduces balance), credit = +amount (increases balance)
                if (!empty($request->payment_account_id)) {
                    $payment_account_data = [
                        'amount' => $totalAmount,
                        'account_id' => $request->payment_account_id,
                        'type' => 'debit', // Payment = debit to asset account (reduces cash/bank balance)
                        'operation_date' => $transactionDate,
                        'created_by' => $user_id,
                        'sub_type' => 'loan_payment',
                        'note' => 'Loan Repayment: ' . $loan->loan_name . ' - ' . $transactionNumber,
                    ];
                    
                    AccountTransaction::createAccountTransaction($payment_account_data);
                }

                // 2. Debit liability account (loan repayment) - Reducing loan liability
                if ($principalAmount > 0) {
                    $account_transaction_data = [
                        'amount' => $principalAmount,
                        'account_id' => $loan->liability_account_id,
                        'type' => 'debit', // Loan repayment = debit to liability (reduces liability)
                        'operation_date' => $transactionDate,
                        'created_by' => $user_id,
                        'sub_type' => 'loan',
                        'note' => 'Loan Repayment (Principal): ' . $loan->loan_name . ' - ' . $transactionNumber,
                    ];
                    
                    AccountTransaction::createAccountTransaction($account_transaction_data);
                }

                // 3. Interest payment (if any) - Debit interest expense account or liability
                if ($interestAmount > 0) {
                    // For now, debit the liability account. You may want to create a separate interest expense account
                    $account_transaction_data = [
                        'amount' => $interestAmount,
                        'account_id' => $loan->liability_account_id,
                        'type' => 'debit',
                        'operation_date' => $transactionDate,
                        'created_by' => $user_id,
                        'sub_type' => 'loan_interest',
                        'note' => 'Loan Interest Payment: ' . $loan->loan_name . ' - ' . $transactionNumber,
                    ];
                    
                    AccountTransaction::createAccountTransaction($account_transaction_data);
                }
            }

            // Update loan outstanding
            $loan->outstanding_principal -= $principalAmount;
            $loan->principal_paid += $principalAmount;
            $loan->interest_paid += $interestAmount;
            
            // Check if loan is fully paid
            if ($loan->isFullyPaid()) {
                $loan->status = 'closed';
            }
            
            $loan->save();

            DB::commit();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.loan_repayment_recorded_success')
            ];

            return response()->json($output);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => $e->getMessage() ?: __('messages.something_went_wrong')
            ];

            return response()->json($output);
        }
    }

    /**
     * Generate unique loan number
     */
    protected function generateLoanNumber($businessId)
    {
        $prefix = 'LOAN';
        $year = date('Y');
        
        $lastLoan = Loan::where('business_id', $businessId)
            ->where('loan_number', 'like', $prefix . '-' . $year . '%')
            ->orderBy('id', 'desc')
            ->first();
        
        if ($lastLoan) {
            $lastNum = (int) substr($lastLoan->loan_number, -4);
            $nextNum = $lastNum + 1;
        } else {
            $nextNum = 1;
        }
        
        return $prefix . '-' . $year . '-' . str_pad($nextNum, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Generate unique transaction number
     */
    protected function generateTransactionNumber($businessId, $prefix)
    {
        $year = date('Y');
        
        $lastTransaction = LoanTransaction::where('business_id', $businessId)
            ->where('transaction_number', 'like', $prefix . '-' . $year . '%')
            ->orderBy('id', 'desc')
            ->first();
        
        if ($lastTransaction) {
            $lastNum = (int) substr($lastTransaction->transaction_number, -4);
            $nextNum = $lastNum + 1;
        } else {
            $nextNum = 1;
        }
        
        return $prefix . '-' . $year . '-' . str_pad($nextNum, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Create liability account for loan
     */
    protected function createLiabilityAccount($businessId, $loanNumber, $loanName, $userId)
    {
        // Find or create liability account type
        $accountType = AccountType::where('business_id', $businessId)
            ->where('name', 'LIKE', '%Loan%')
            ->orWhere('name', 'LIKE', '%Liability%')
            ->first();

        $accountData = [
            'business_id' => $businessId,
            'name' => $loanName . ' (' . $loanNumber . ')',
            'account_number' => $loanNumber,
            'account_type_id' => $accountType ? $accountType->id : null,
            'note' => 'Liability account for loan: ' . $loanNumber,
            'created_by' => $userId,
        ];

        $account = Account::create($accountData);

        return $account;
    }

    /**
     * Show loan transaction details
     */
    public function showTransaction($transaction_id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $transaction = LoanTransaction::where('business_id', $business_id)
            ->with(['loan', 'paymentAccount', 'accountTransaction', 'creator'])
            ->findOrFail($transaction_id);

        return view('accounting-reports::loans.show-transaction', compact('transaction'));
    }
}

