<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Entities\Investment;
use App\Utils\Util;
use DB;
use Yajra\DataTables\Facades\DataTables;

class InvestmentController extends Controller
{
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of investments
     */
    public function index()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting-reports::investments.index');
    }

    /**
     * Get investments data for DataTables
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;

        $query = Investment::with(['creator'])
            ->where('business_id', $business_id);

        // Filter by active status if provided
        if ($request->has('is_active') && $request->is_active !== '') {
            $query->where('is_active', $request->is_active);
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $html = '<div class="btn-group">';
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-info btn-xs view_investment" data-href="' . 
                        action([self::class, 'show'], [$row->id]) . '"><i class="fa fa-eye"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-primary btn-xs edit_investment" data-href="' . 
                        action([self::class, 'edit'], [$row->id]) . '"><i class="fa fa-edit"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-danger btn-xs delete_investment" data-href="' . 
                        action([self::class, 'destroy'], [$row->id]) . '"><i class="fa fa-trash"></i></button>';
                }
                
                $html .= '</div>';
                return $html;
            })
            ->editColumn('is_active', function ($row) {
                if ($row->is_active) {
                    return '<span class="label label-success">' . __('accounting-reports::lang.active') . '</span>';
                } else {
                    return '<span class="label label-danger">' . __('accounting-reports::lang.inactive') . '</span>';
                }
            })
            ->editColumn('description', function ($row) {
                return !empty($row->description) ? $row->description : '-';
            })
            ->editColumn('investment_date', function ($row) {
                return !empty($row->investment_date) ? $this->commonUtil->format_date($row->investment_date, false) : '-';
            })
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true">' . $row->amount . '</span>';
            })
            ->editColumn('current_value', function ($row) {
                $value = $row->current_value ?? $row->amount;
                return '<span class="display_currency" data-currency_symbol="true">' . $value . '</span>';
            })
            ->editColumn('investment_type', function ($row) {
                return !empty($row->investment_type) ? $row->investment_type : '-';
            })
            ->editColumn('created_at', function ($row) {
                return $this->commonUtil->format_date($row->created_at, true);
            })
            ->rawColumns(['action', 'is_active', 'amount', 'current_value'])
            ->toJson();
    }

    /**
     * Show the form for creating a new investment
     */
    public function create()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting-reports::investments.create');
    }

    /**
     * Store a newly created investment
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $user_id = auth()->user()->id;

            $request->validate([
                'name' => [
                    'required',
                    'string',
                    'max:255',
                    'unique:ar_investments,name,NULL,id,business_id,' . $business_id . ',deleted_at,NULL'
                ],
                'description' => 'nullable|string',
                'investment_date' => 'nullable|date',
                'amount' => 'nullable|numeric|min:0',
                'current_value' => 'nullable|numeric|min:0',
                'investment_type' => 'nullable|string|max:255',
            ]);

            $data = $request->only(['name', 'description', 'investment_date', 'amount', 'current_value', 'investment_type']);
            $data['business_id'] = $business_id;
            $data['created_by'] = $user_id;
            $data['is_active'] = $request->has('is_active') ? 1 : 1; // Default to active
            
            // Format investment_date if provided
            if (!empty($data['investment_date'])) {
                $data['investment_date'] = $this->commonUtil->uf_date($data['investment_date'], false);
            }

            $investment = Investment::create($data);

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.investment_added_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Display the specified investment
     */
    public function show($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $investment = Investment::where('business_id', $business_id)
            ->with(['creator'])
            ->findOrFail($id);

        return view('accounting-reports::investments.show', compact('investment'));
    }

    /**
     * Show the form for editing the specified investment
     */
    public function edit($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $investment = Investment::where('business_id', $business_id)->findOrFail($id);

        return view('accounting-reports::investments.edit', compact('investment'));
    }

    /**
     * Update the specified investment
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $investment = Investment::where('business_id', $business_id)->findOrFail($id);

            $request->validate([
                'name' => [
                    'required',
                    'string',
                    'max:255',
                    'unique:ar_investments,name,' . $id . ',id,business_id,' . $business_id . ',deleted_at,NULL'
                ],
                'description' => 'nullable|string',
                'investment_date' => 'nullable|date',
                'amount' => 'nullable|numeric|min:0',
                'current_value' => 'nullable|numeric|min:0',
                'investment_type' => 'nullable|string|max:255',
            ]);

            $data = $request->only(['name', 'description', 'investment_date', 'amount', 'current_value', 'investment_type']);
            $data['is_active'] = $request->has('is_active') ? 1 : 0;
            
            // Format investment_date if provided
            if (!empty($data['investment_date'])) {
                $data['investment_date'] = $this->commonUtil->uf_date($data['investment_date'], false);
            }

            $investment->update($data);

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.investment_updated_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Remove the specified investment
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $investment = Investment::where('business_id', $business_id)->findOrFail($id);

            $investment->delete();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.investment_deleted_success')
            ];

            return response()->json($output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            return response()->json($output);
        }
    }
}

