<?php

namespace Modules\AccountingReports\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\System;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\AccountingReports\Database\Seeders\AccountingReportsDatabaseSeeder;
use Spatie\Permission\Models\Permission;

class InstallController extends Controller
{
    /**
     * Install the module
     */
    public function index()
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            ini_set('max_execution_time', 0);
            ini_set('memory_limit', '512M');

            // Clear cache & config
            config(['app.debug' => true]);
            Artisan::call('config:clear');
            Artisan::call('cache:clear');

            // Check if already installed
            $is_installed = System::getProperty('accountingreports_version');
            if (!empty($is_installed)) {
                return redirect()->back()->with('status', [
                    'success' => 0,
                    'msg' => 'Module is already installed.'
                ]);
            }

            DB::beginTransaction();
            DB::statement('SET default_storage_engine=INNODB;');

            // Always run ALL migrations - module:migrate will run all pending migrations
            // This ensures all migration files are executed during installation
            \Log::info('AccountingReports: Running all migrations...');
            
            try {
                // Use module:migrate which will run all pending migrations
                // It respects the migrations table, so already-run migrations are skipped
                // But all new/pending migrations will be executed
                Artisan::call('module:migrate', [
                    'module' => 'AccountingReports',
                    '--force' => true,
                ]);
                
                $migrationOutput = Artisan::output();
                \Log::info('AccountingReports: Migration completed. Output: ' . $migrationOutput);
                
                // Verify all key tables exist after migration
                $keyTables = [
                    'ar_chart_of_accounts',
                    'ar_journal_entry_headers',
                    'ar_journal_entry_lines',
                    'ar_ledger_rollups',
                    'ar_fifo_layers',
                    'ar_receivables_payables',
                    'ar_period_locks',
                    'ar_bank_reconciliation',
                    'ar_audit_log',
                    'ar_ratio_snapshots',
                    'ar_direct_expenses',
                    'ar_direct_expense_names',
                    'ar_direct_income_names',
                    'ar_direct_incomes',
                    'ar_cheque_book_entries',
                    'ar_cheque_books',
                    'ar_loans',
                    'ar_loan_transactions',
                    'ar_fixed_assets',
                    'ar_fixed_asset_revaluations',
                    'ar_capital_accounts',
                    'ar_investments',
                    'ar_bank_accounts'
                ];
                
                $missingTables = [];
                foreach ($keyTables as $table) {
                    if (!Schema::hasTable($table)) {
                        $missingTables[] = $table;
                    }
                }
                
                if (!empty($missingTables)) {
                    \Log::warning('AccountingReports: Some tables still missing after migration: ' . implode(', ', $missingTables));
                    throw new \Exception('Some required tables are missing after migration. Missing tables: ' . implode(', ', $missingTables) . '. Please check migration errors above.');
                } else {
                    \Log::info('AccountingReports: All key tables verified successfully');
                }
                
            } catch (\Exception $migrationError) {
                // Log the error but check if tables exist anyway
                \Log::error('AccountingReports: Migration error: ' . $migrationError->getMessage());
                
                // Check which tables are missing
                $keyTables = [
                    'ar_chart_of_accounts',
                    'ar_journal_entry_headers',
                    'ar_journal_entry_lines',
                    'ar_ledger_rollups',
                    'ar_fifo_layers',
                    'ar_receivables_payables',
                    'ar_period_locks',
                    'ar_bank_reconciliation',
                    'ar_audit_log',
                    'ar_ratio_snapshots',
                    'ar_direct_expenses',
                    'ar_direct_expense_names',
                    'ar_direct_income_names',
                    'ar_direct_incomes',
                    'ar_cheque_book_entries',
                    'ar_cheque_books',
                    'ar_loans',
                    'ar_loan_transactions',
                    'ar_fixed_assets',
                    'ar_fixed_asset_revaluations',
                    'ar_capital_accounts',
                    'ar_investments',
                    'ar_bank_accounts'
                ];
                
                $missingTables = [];
                foreach ($keyTables as $table) {
                    if (!Schema::hasTable($table)) {
                        $missingTables[] = $table;
                    }
                }
                
                if (!empty($missingTables)) {
                    \Log::error('AccountingReports: Critical - Missing tables after migration: ' . implode(', ', $missingTables));
                    throw new \Exception('Migration failed. Missing tables: ' . implode(', ', $missingTables) . '. Error: ' . $migrationError->getMessage());
                } else {
                    // All tables exist, so migration might have partially succeeded
                    // If error is about "table already exists", it's okay
                    if (strpos($migrationError->getMessage(), 'already exists') !== false) {
                        \Log::warning('AccountingReports: Some tables already exist, but all required tables are present. Continuing...');
                    } else {
                        // Other errors should be thrown
                        throw $migrationError;
                    }
                }
            }
            
            // Run seeders (only if tables were just created or if Chart of Accounts is empty)
            try {
                $seeder = new AccountingReportsDatabaseSeeder();
                $seeder->run();
            } catch (\Exception $seederError) {
                // If seeder fails (e.g., accounts already exist), log but continue
                \Log::warning('AccountingReports: Seeder warning: ' . $seederError->getMessage());
            }
            
            // Store module version in system table (required for menu display)
            // Check both config keys for compatibility with ModuleUtil
            $moduleVersion = config('accounting-reports.module_version') 
                         ?? config('accountingreports.module_version') 
                         ?? '1.0.3';
            System::addProperty('accountingreports_version', $moduleVersion);
            
            // Create permissions if they don't exist
            $this->createPermissions();

            DB::commit();
            
            \Log::info('AccountingReports: Module installed successfully - Version ' . $moduleVersion);
            
            return redirect()->back()->with('status', [
                'success' => 1,
                'msg' => __('accounting-reports::lang.installation_success')
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency("AccountingReports Installation Error - File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            \Log::emergency("AccountingReports Installation Trace: " . $e->getTraceAsString());
            
            return redirect()->back()->with('status', [
                'success' => 0,
                'msg' => __('accounting-reports::lang.installation_error') . ': ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Update the module
     */
    public function update()
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            ini_set('max_execution_time', 0);
            ini_set('memory_limit', '512M');

            // Clear cache & config
            config(['app.debug' => true]);
            Artisan::call('config:clear');
            Artisan::call('cache:clear');

            DB::beginTransaction();
            DB::statement('SET default_storage_engine=INNODB;');

            \Log::info('AccountingReports: Starting module update - Running all pending migrations');

            // Always run ALL pending migrations during update
            // This ensures any new migration files are executed
            try {
                Artisan::call('module:migrate', [
                    'module' => 'AccountingReports',
                    '--force' => true,
                ]);
                
                $migrationOutput = Artisan::output();
                \Log::info('AccountingReports: Migration completed. Output: ' . $migrationOutput);
                
                // Verify all key tables still exist after update
                $keyTables = [
                    'ar_chart_of_accounts',
                    'ar_journal_entry_headers',
                    'ar_journal_entry_lines',
                    'ar_ledger_rollups',
                    'ar_fifo_layers',
                    'ar_receivables_payables',
                    'ar_period_locks',
                    'ar_bank_reconciliation',
                    'ar_audit_log',
                    'ar_ratio_snapshots',
                    'ar_direct_expenses',
                    'ar_direct_expense_names',
                    'ar_direct_income_names',
                    'ar_direct_incomes',
                    'ar_cheque_book_entries',
                    'ar_cheque_books',
                    'ar_loans',
                    'ar_loan_transactions',
                    'ar_fixed_assets',
                    'ar_fixed_asset_revaluations',
                    'ar_capital_accounts',
                    'ar_investments',
                    'ar_bank_accounts'
                ];
                
                $missingTables = [];
                foreach ($keyTables as $table) {
                    if (!Schema::hasTable($table)) {
                        $missingTables[] = $table;
                    }
                }
                
                if (!empty($missingTables)) {
                    \Log::error('AccountingReports: Some tables missing after update: ' . implode(', ', $missingTables));
                    throw new \Exception('Update failed. Missing tables: ' . implode(', ', $missingTables));
                } else {
                    \Log::info('AccountingReports: All tables verified after update');
                }
                
            } catch (\Exception $migrationError) {
                // If migration fails due to table exists, check if all tables are present
                if (strpos($migrationError->getMessage(), 'already exists') !== false) {
                    \Log::warning('AccountingReports: Some migrations failed (tables exist), verifying all tables...');
                    
                    // Quick check - if all key tables exist, continue
                    $allTablesExist = Schema::hasTable('ar_chart_of_accounts') && 
                                     Schema::hasTable('ar_journal_entry_headers') &&
                                     Schema::hasTable('ar_capital_accounts');
                    
                    if ($allTablesExist) {
                        \Log::info('AccountingReports: Key tables exist, continuing update despite migration warning');
                    } else {
                        throw $migrationError;
                    }
                } else {
                    throw $migrationError;
                }
            }
            
            // Update module version - check both config keys for compatibility
            $moduleVersion = config('accounting-reports.module_version') 
                         ?? config('accountingreports.module_version') 
                         ?? '1.0.3';
            System::setProperty('accountingreports_version', $moduleVersion);

            DB::commit();
            
            \Log::info('AccountingReports: Module updated successfully to version ' . $moduleVersion);
            
            return redirect()->back()->with('status', [
                'success' => 1,
                'msg' => __('accounting-reports::lang.update_success')
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency("AccountingReports Update Error - File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            \Log::emergency("AccountingReports Update Trace: " . $e->getTraceAsString());
            
            return redirect()->back()->with('status', [
                'success' => 0,
                'msg' => __('accounting-reports::lang.update_error') . ': ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Uninstall the module
     */
    public function uninstall()
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            // Remove version property
            System::removeProperty('accountingreports_version');
            
            // Optionally drop tables - be careful with this!
            // Artisan::call('migrate:rollback', [
            //     '--path' => 'Modules/AccountingReports/Database/Migrations',
            //     '--force' => true,
            // ]);
            
            return redirect()->back()->with('status', [
                'success' => 1,
                'msg' => __('accounting-reports::lang.uninstall_success')
            ]);
        } catch (\Exception $e) {
            \Log::emergency("File:" . $e->getFile() . "Line:" . $e->getLine() . "Message:" . $e->getMessage());
            
            return redirect()->back()->with('status', [
                'success' => 0,
                'msg' => __('accounting-reports::lang.uninstall_error') . ': ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Record migrations for tables that already exist
     * This prevents "table already exists" errors during migration
     */
    protected function recordExistingMigrations()
    {
        $migrationMap = [
            '2024_01_01_000001_create_chart_of_accounts_table' => 'ar_chart_of_accounts',
            '2024_01_01_000002_create_journal_entry_headers_table' => 'ar_journal_entry_headers',
            '2024_01_01_000003_create_journal_entry_lines_table' => 'ar_journal_entry_lines',
            '2024_01_01_000004_create_ledger_rollups_table' => 'ar_ledger_rollups',
            '2024_01_01_000005_create_fifo_layers_table' => 'ar_fifo_layers',
            '2024_01_01_000006_create_account_receivables_payables_table' => 'ar_receivables_payables',
            '2024_01_01_000007_create_period_locks_table' => 'ar_period_locks',
            '2024_01_01_000008_create_bank_reconciliation_table' => 'ar_bank_reconciliation',
            '2024_01_01_000009_create_audit_log_table' => 'ar_audit_log',
            '2024_01_01_000010_create_ratio_snapshots_table' => 'ar_ratio_snapshots',
            '2024_01_01_000011_create_direct_expenses_table' => 'ar_direct_expenses',
            '2024_01_01_000012_create_direct_expense_names_table' => 'ar_direct_expense_names',
            '2024_01_01_000013_create_direct_income_names_table' => 'ar_direct_income_names',
            '2024_01_01_000014_create_direct_incomes_table' => 'ar_direct_incomes',
            '2024_01_01_000015_create_cheque_book_entries_table' => 'ar_cheque_book_entries',
            '2024_01_01_000016_create_cheque_books_table' => 'ar_cheque_books',
            '2024_01_01_000017_create_loans_table' => 'ar_loans',
            '2024_01_01_000018_create_loan_transactions_table' => 'ar_loan_transactions',
            '2024_01_01_000019_create_fixed_assets_table' => 'ar_fixed_assets',
            '2024_01_01_000020_add_fields_to_fixed_assets_table' => 'ar_fixed_assets', // Check same table
            '2024_01_01_000021_add_revaluation_fields_to_fixed_assets_table' => 'ar_fixed_assets', // Check same table
            '2024_01_01_000022_create_fixed_asset_revaluations_table' => 'ar_fixed_asset_revaluations',
            '2024_01_01_000023_create_capital_accounts_table' => 'ar_capital_accounts',
            '2024_01_01_000024_create_investments_table' => 'ar_investments',
            '2024_01_01_000025_create_bank_accounts_table' => 'ar_bank_accounts',
            '2024_01_01_000026_add_opening_balance_transaction_to_bank_accounts' => 'ar_bank_accounts', // Check same table
            '2025_01_15_000001_add_account_transaction_id_to_bank_reconciliation' => 'ar_bank_reconciliation', // Check same table
            '2025_01_15_000002_add_bank_account_id_to_cheque_book_entries' => 'ar_cheque_book_entries', // Check same table
            '2025_11_03_030547_add_category_id_to_direct_expenses_table' => 'ar_direct_expenses', // Check same table
        ];

        $migrationsPath = module_path('AccountingReports', 'Database/Migrations');
        $migrationFiles = glob($migrationsPath . '/*.php');
        
        $batch = DB::table('migrations')->max('batch') ?? 0;
        $newBatch = $batch + 1;
        
        $recorded = 0;
        foreach ($migrationFiles as $file) {
            $migrationName = basename($file, '.php');
            
            // Check if migration is already recorded
            $exists = DB::table('migrations')
                ->where('migration', $migrationName)
                ->exists();
            
            if ($exists) {
                continue; // Already recorded
            }
            
            // Check if the table exists
            $tableName = $migrationMap[$migrationName] ?? null;
            if ($tableName && Schema::hasTable($tableName)) {
                // Table exists but migration not recorded - record it
                DB::table('migrations')->insert([
                    'migration' => $migrationName,
                    'batch' => $newBatch,
                ]);
                $recorded++;
                \Log::info("AccountingReports: Recorded existing migration: {$migrationName} (table: {$tableName})");
            }
        }
        
        if ($recorded > 0) {
            \Log::info("AccountingReports: Recorded {$recorded} existing migrations");
        }
    }

    /**
     * Create module permissions
     */
    protected function createPermissions()
    {
        $permissions = [
            'accounting.view_all',
            'accounting.view_trial_balance',
            'accounting.view_balance_sheet',
            'accounting.view_pl',
            'accounting.view_cashflow',
            'accounting.view_ar',
            'accounting.view_ap',
            'accounting.view_daybook',
            'accounting.view_cashbook',
            'accounting.view_bankbook',
            'accounting.export_reports',
            'accounting.print_multi_account',
        ];

        foreach ($permissions as $permission) {
            Permission::firstOrCreate(
                ['name' => $permission, 'guard_name' => 'web']
            );
        }
    }
}


