<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Entities\FixedAsset;
use App\Utils\Util;
use App\BusinessLocation;
use Carbon\Carbon;

class FixedAssetsReportController extends Controller
{
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display the fixed assets report
     */
    public function index()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::fixed-assets-report.index', compact('locations'));
    }

    /**
     * Get fixed assets report data
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $businessId = auth()->user()->business_id;
            
            // Parse date
            $asOfDate = !empty($request->input('as_of_date')) 
                ? $this->commonUtil->uf_date($request->input('as_of_date'), false) 
                : date('Y-m-d');

            if ($asOfDate instanceof \Carbon\Carbon) {
                $asOfDate = $asOfDate->format('Y-m-d');
            }

            $assets = FixedAsset::where('business_id', $businessId)
                ->where('is_active', 1)
                ->with(['revaluations'])
                ->get();

            $reportData = [];
            $totalOriginalCost = 0;
            $totalRevaluation = 0;
            $totalBookValue = 0;
            $totalAccumulatedDepreciation = 0;
            $totalNetBookValue = 0;
            $totalAnnualDepreciation = 0;

            foreach ($assets as $asset) {
                $originalCost = (float) $asset->amount;
                $bookValue = (float) ($asset->current_book_value ?? $asset->amount);
                $accumulatedDepreciation = (float) ($asset->accumulated_depreciation ?? 0);
                $netBookValue = $asset->net_book_value;
                $annualDepreciation = $asset->annual_depreciation;
                
                // Calculate revaluation total
                $revaluationTotal = 0;
                if ($asset->revaluations) {
                    $revaluationTotal = $asset->revaluations->sum('revaluation_difference');
                }

                // Calculate depreciation for period (if needed)
                $depreciationForPeriod = 0;
                if (!empty($asset->depreciation_percentage) && $asset->depreciation_percentage > 0) {
                    // Calculate based on days from purchase/revaluation to as_of_date
                    $startDate = $asset->purchase_date;
                    if ($asset->revaluations && $asset->revaluations->count() > 0) {
                        $lastRevaluation = $asset->revaluations->first();
                        $startDate = $lastRevaluation->revaluation_date;
                    }
                    
                    $days = Carbon::parse($startDate)->diffInDays(Carbon::parse($asOfDate));
                    $years = $days / 365;
                    $depreciationForPeriod = ($bookValue * $asset->depreciation_percentage / 100) * $years;
                }

                $reportData[] = [
                    'id' => $asset->id,
                    'name' => $asset->name,
                    'description' => $asset->description,
                    'purchase_date' => $asset->purchase_date ? $this->commonUtil->format_date($asset->purchase_date, false) : '-',
                    'original_cost' => $originalCost,
                    'revaluation_total' => $revaluationTotal,
                    'current_book_value' => $bookValue,
                    'depreciation_percentage' => $asset->depreciation_percentage ?? 0,
                    'annual_depreciation' => $annualDepreciation,
                    'accumulated_depreciation' => $accumulatedDepreciation,
                    'net_book_value' => $netBookValue,
                    'revaluation_count' => $asset->revaluations ? $asset->revaluations->count() : 0,
                ];

                $totalOriginalCost += $originalCost;
                $totalRevaluation += $revaluationTotal;
                $totalBookValue += $bookValue;
                $totalAccumulatedDepreciation += $accumulatedDepreciation;
                $totalNetBookValue += $netBookValue;
                $totalAnnualDepreciation += $annualDepreciation;
            }

            // Get business name
            $business = \App\Business::find($businessId);

            return response()->json([
                'success' => true,
                'data' => $reportData,
                'totals' => [
                    'original_cost' => $totalOriginalCost,
                    'revaluation_total' => $totalRevaluation,
                    'current_book_value' => $totalBookValue,
                    'accumulated_depreciation' => $totalAccumulatedDepreciation,
                    'net_book_value' => $totalNetBookValue,
                    'annual_depreciation' => $totalAnnualDepreciation,
                ],
                'business_name' => $business->name ?? '',
                'as_of_date' => $asOfDate,
            ]);

        } catch (\Exception $e) {
            \Log::error('FixedAssetsReport getData Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'error' => 'Error generating report: ' . $e->getMessage(),
                'data' => [],
                'totals' => []
            ], 500);
        }
    }

    /**
     * Export report
     */
    public function export(Request $request)
    {
        if (!auth()->user()->can('accounting.export_reports')) {
            abort(403, 'Unauthorized action.');
        }

        // TODO: Implement export to PDF/Excel
        return redirect()->back();
    }
}

