<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Entities\FixedAsset;
use Modules\AccountingReports\Entities\FixedAssetRevaluation;
use App\Utils\Util;
use DB;
use Yajra\DataTables\Facades\DataTables;

class FixedAssetRevaluationController extends Controller
{
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of revaluations
     */
    public function index()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $assets = FixedAsset::where('business_id', $business_id)
            ->where('is_active', 1)
            ->pluck('name', 'id');

        return view('accounting-reports::fixed-asset-revaluations.index', compact('assets'));
    }

    /**
     * Get revaluations data for DataTables
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;

        $query = FixedAssetRevaluation::with(['fixedAsset', 'creator'])
            ->where('business_id', $business_id);

        // Filter by asset if provided
        if ($request->has('fixed_asset_id') && $request->fixed_asset_id) {
            $query->where('fixed_asset_id', $request->fixed_asset_id);
        }

        return DataTables::of($query)
            ->addColumn('asset_name', function ($row) {
                return $row->fixedAsset->name ?? '-';
            })
            ->editColumn('revaluation_date', function ($row) {
                return $this->commonUtil->format_date($row->revaluation_date, false);
            })
            ->editColumn('previous_value', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true">' . $row->previous_value . '</span>';
            })
            ->editColumn('revaluation_amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true">' . $row->revaluation_amount . '</span>';
            })
            ->editColumn('revaluation_difference', function ($row) {
                $class = $row->revaluation_difference >= 0 ? 'text-success' : 'text-danger';
                return '<span class="display_currency ' . $class . '" data-currency_symbol="true">' . $row->revaluation_difference . '</span>';
            })
            ->editColumn('created_at', function ($row) {
                return $this->commonUtil->format_date($row->created_at, true);
            })
            ->addColumn('action', function ($row) {
                $html = '<div class="btn-group">';
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-info btn-xs view_revaluation" data-href="' . 
                        action([self::class, 'show'], [$row->id]) . '"><i class="fa fa-eye"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-danger btn-xs delete_revaluation" data-href="' . 
                        action([self::class, 'destroy'], [$row->id]) . '"><i class="fa fa-trash"></i></button>';
                }
                
                $html .= '</div>';
                return $html;
            })
            ->rawColumns(['action', 'previous_value', 'revaluation_amount', 'revaluation_difference'])
            ->toJson();
    }

    /**
     * Show the form for creating a new revaluation
     */
    public function create()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $assets = FixedAsset::where('business_id', $business_id)
            ->where('is_active', 1)
            ->get();

        return view('accounting-reports::fixed-asset-revaluations.create', compact('assets'));
    }

    /**
     * Get asset details for auto-population
     */
    public function getAssetDetails($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $asset = FixedAsset::where('business_id', $business_id)
                ->findOrFail($id);

            $currentBookValue = $asset->current_book_value ?? $asset->amount;
            $originalAmount = $asset->amount;
            $netBookValue = $asset->net_book_value;
            $depreciationPercentage = $asset->depreciation_percentage ?? 0;

            return response()->json([
                'success' => true,
                'current_book_value' => $currentBookValue,
                'original_amount' => $originalAmount,
                'net_book_value' => $netBookValue,
                'depreciation_percentage' => $depreciationPercentage,
                'asset_name' => $asset->name,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Asset not found'
            ], 404);
        }
    }

    /**
     * Store a newly created revaluation
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $user_id = auth()->user()->id;

            $request->validate([
                'fixed_asset_id' => 'required|exists:ar_fixed_assets,id',
                'revaluation_date' => 'required|date',
                'revaluation_amount' => 'required|numeric|min:0',
                'reason' => 'nullable|string',
                'notes' => 'nullable|string',
            ]);

            $asset = FixedAsset::where('business_id', $business_id)
                ->findOrFail($request->fixed_asset_id);

            $previousValue = $asset->current_book_value ?? $asset->amount;
            $revaluationAmount = (float) $request->revaluation_amount;
            $revaluationDifference = $revaluationAmount - $previousValue;

            // Format date
            $revaluationDate = $this->commonUtil->uf_date($request->revaluation_date, false);

            DB::beginTransaction();

            // Create revaluation record
            $revaluation = FixedAssetRevaluation::create([
                'fixed_asset_id' => $request->fixed_asset_id,
                'business_id' => $business_id,
                'revaluation_date' => $revaluationDate,
                'previous_value' => $previousValue,
                'revaluation_amount' => $revaluationAmount,
                'revaluation_difference' => $revaluationDifference,
                'reason' => $request->reason,
                'notes' => $request->notes,
                'created_by' => $user_id,
            ]);

            // Update asset's current book value
            $asset->current_book_value = $revaluationAmount;
            $asset->save();

            DB::commit();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.revaluation_added_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Display the specified revaluation
     */
    public function show($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $revaluation = FixedAssetRevaluation::where('business_id', $business_id)
            ->with(['fixedAsset', 'creator'])
            ->findOrFail($id);

        return view('accounting-reports::fixed-asset-revaluations.show', compact('revaluation'));
    }

    /**
     * Remove the specified revaluation
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $revaluation = FixedAssetRevaluation::where('business_id', $business_id)
                ->with('fixedAsset')
                ->findOrFail($id);

            DB::beginTransaction();

            // Restore previous book value
            $asset = $revaluation->fixedAsset;
            if ($asset) {
                // Get the previous revaluation or use original amount
                $previousRevaluation = FixedAssetRevaluation::where('fixed_asset_id', $asset->id)
                    ->where('id', '!=', $id)
                    ->orderBy('revaluation_date', 'desc')
                    ->first();

                if ($previousRevaluation) {
                    $asset->current_book_value = $previousRevaluation->revaluation_amount;
                } else {
                    $asset->current_book_value = $asset->amount;
                }
                $asset->save();
            }

            $revaluation->delete();

            DB::commit();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.revaluation_deleted_success')
            ];

            return response()->json($output);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            return response()->json($output);
        }
    }
}

