<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Entities\FixedAsset;
use App\Utils\Util;
use DB;
use Yajra\DataTables\Facades\DataTables;

class FixedAssetController extends Controller
{
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of fixed assets
     */
    public function index()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting-reports::fixed-assets.index');
    }

    /**
     * Get fixed assets data for DataTables
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;

        $query = FixedAsset::with(['creator'])
            ->where('business_id', $business_id);

        // Filter by active status if provided
        if ($request->has('is_active') && $request->is_active !== '') {
            $query->where('is_active', $request->is_active);
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $html = '<div class="btn-group">';
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-info btn-xs view_asset" data-href="' . 
                        action([self::class, 'show'], [$row->id]) . '"><i class="fa fa-eye"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-primary btn-xs edit_asset" data-href="' . 
                        action([self::class, 'edit'], [$row->id]) . '"><i class="fa fa-edit"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-danger btn-xs delete_asset" data-href="' . 
                        action([self::class, 'destroy'], [$row->id]) . '"><i class="fa fa-trash"></i></button>';
                }
                
                $html .= '</div>';
                return $html;
            })
            ->editColumn('is_active', function ($row) {
                if ($row->is_active) {
                    return '<span class="label label-success">' . __('accounting-reports::lang.active') . '</span>';
                } else {
                    return '<span class="label label-danger">' . __('accounting-reports::lang.inactive') . '</span>';
                }
            })
            ->editColumn('description', function ($row) {
                return !empty($row->description) ? $row->description : '-';
            })
            ->editColumn('purchase_date', function ($row) {
                return !empty($row->purchase_date) ? $this->commonUtil->format_date($row->purchase_date, false) : '-';
            })
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true">' . $row->amount . '</span>';
            })
            ->editColumn('depreciation_percentage', function ($row) {
                return !empty($row->depreciation_percentage) ? number_format($row->depreciation_percentage, 2) . '%' : '-';
            })
            ->editColumn('created_at', function ($row) {
                return $this->commonUtil->format_date($row->created_at, true);
            })
            ->rawColumns(['action', 'is_active', 'amount'])
            ->toJson();
    }

    /**
     * Show the form for creating a new fixed asset
     */
    public function create()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting-reports::fixed-assets.create');
    }

    /**
     * Store a newly created fixed asset
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $user_id = auth()->user()->id;

            $request->validate([
                'name' => [
                    'required',
                    'string',
                    'max:255',
                    'unique:ar_fixed_assets,name,NULL,id,business_id,' . $business_id . ',deleted_at,NULL'
                ],
                'description' => 'nullable|string',
                'purchase_date' => 'nullable|date',
                'amount' => 'nullable|numeric|min:0',
                'depreciation_percentage' => 'nullable|numeric|min:0|max:100',
            ]);

            $data = $request->only(['name', 'description', 'purchase_date', 'amount', 'depreciation_percentage']);
            $data['business_id'] = $business_id;
            $data['created_by'] = $user_id;
            $data['is_active'] = $request->has('is_active') ? 1 : 1; // Default to active
            
            // Format purchase_date if provided
            if (!empty($data['purchase_date'])) {
                $data['purchase_date'] = $this->commonUtil->uf_date($data['purchase_date'], false);
            }
            
            // Initialize current_book_value with amount if not set
            if (empty($data['current_book_value']) && !empty($data['amount'])) {
                $data['current_book_value'] = $data['amount'];
            }

            $fixed_asset = FixedAsset::create($data);

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.fixed_asset_added_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Display the specified fixed asset
     */
    public function show($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $fixed_asset = FixedAsset::where('business_id', $business_id)
            ->with(['creator'])
            ->findOrFail($id);

        return view('accounting-reports::fixed-assets.show', compact('fixed_asset'));
    }

    /**
     * Show the form for editing the specified fixed asset
     */
    public function edit($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $fixed_asset = FixedAsset::where('business_id', $business_id)->findOrFail($id);

        return view('accounting-reports::fixed-assets.edit', compact('fixed_asset'));
    }

    /**
     * Update the specified fixed asset
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $fixed_asset = FixedAsset::where('business_id', $business_id)->findOrFail($id);

            $request->validate([
                'name' => [
                    'required',
                    'string',
                    'max:255',
                    'unique:ar_fixed_assets,name,' . $id . ',id,business_id,' . $business_id . ',deleted_at,NULL'
                ],
                'description' => 'nullable|string',
                'purchase_date' => 'nullable|date',
                'amount' => 'nullable|numeric|min:0',
                'depreciation_percentage' => 'nullable|numeric|min:0|max:100',
            ]);

            $data = $request->only(['name', 'description', 'purchase_date', 'amount', 'depreciation_percentage']);
            $data['is_active'] = $request->has('is_active') ? 1 : 0;
            
            // Format purchase_date if provided
            if (!empty($data['purchase_date'])) {
                $data['purchase_date'] = $this->commonUtil->uf_date($data['purchase_date'], false);
            }

            $fixed_asset->update($data);

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.fixed_asset_updated_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Remove the specified fixed asset
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $fixed_asset = FixedAsset::where('business_id', $business_id)->findOrFail($id);

            $fixed_asset->delete();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.fixed_asset_deleted_success')
            ];

            return response()->json($output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            return response()->json($output);
        }
    }
}

