<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\BusinessLocation;
use App\Account;
use App\Contact;
use App\User;
use App\Utils\TransactionUtil;
use Modules\AccountingReports\Services\DaybookQuery;
use DB;

class DayBookController extends Controller
{
    protected $transactionUtil;
    protected $daybookQuery;

    public function __construct(TransactionUtil $transactionUtil, DaybookQuery $daybookQuery)
    {
        $this->transactionUtil = $transactionUtil;
        $this->daybookQuery = $daybookQuery;
    }

    public function index()
    {
        if (!auth()->user()->can('accounting.view_daybook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $locations = BusinessLocation::forDropdown($businessId, true);
        $accounts = Account::forDropdown($businessId, true, false, false);
        $customers = Contact::where('business_id', $businessId)->where('type', 'customer')->pluck('name', 'id');
        $suppliers = Contact::where('business_id', $businessId)->where('type', 'supplier')->pluck('name', 'id');
        $users = User::forDropdown($businessId, false, true, false);

        // Get payment methods
        $payment_methods = [
            'cash' => __('lang_v1.cash'),
            'card' => __('lang_v1.card'),
            'cheque' => __('lang_v1.cheque'),
            'bank_transfer' => __('lang_v1.bank_transfer'),
            'other' => __('lang_v1.other'),
        ];

        return view('accounting-reports::day-book.index', compact(
            'locations',
            'accounts',
            'customers',
            'suppliers',
            'users',
            'payment_methods'
        ));
    }

    /**
     * Get Day Book data in Tally-style format
     * Chronological listing of all vouchers with debit/credit entries
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_daybook')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        
        // Handle date range input - convert from user format to Y-m-d if needed
        $start_date = null;
        $end_date = null;
        
        if ($request->has('start_date') && !empty($request->input('start_date'))) {
            try {
                $start_date = $this->transactionUtil->uf_date($request->input('start_date'));
                if (!$start_date) {
                    $start_date = \Carbon\Carbon::createFromFormat('Y-m-d', $request->input('start_date'))->format('Y-m-d');
                }
            } catch (\Exception $e) {
                $start_date = \Carbon\Carbon::now()->format('Y-m-d');
            }
        } else {
            $start_date = \Carbon\Carbon::now()->format('Y-m-d');
        }

        if ($request->has('end_date') && !empty($request->input('end_date'))) {
            try {
                $end_date = $this->transactionUtil->uf_date($request->input('end_date'));
                if (!$end_date) {
                    $end_date = \Carbon\Carbon::createFromFormat('Y-m-d', $request->input('end_date'))->format('Y-m-d');
                }
            } catch (\Exception $e) {
                $end_date = \Carbon\Carbon::now()->format('Y-m-d');
            }
        } else {
            $end_date = \Carbon\Carbon::now()->format('Y-m-d');
        }

        $permitted_locations = auth()->user()->permitted_locations();

        // Build filters array
        $filters = [
            'location_id' => $request->input('location_id'),
            'user_id' => $request->input('user_id'),
            'account_id' => $request->input('account_id'),
            'module_filter' => $request->input('module_filter'),
            'scope' => $request->input('scope', 'cash_bank'), // cash, bank, cash_bank, all
            'party_id' => $request->input('party_id'),
            'payment_method' => $request->input('payment_method'),
            'currency' => $request->input('currency'),
            'use_created_date' => $request->input('use_created_date', false),
            'permitted_locations' => $permitted_locations,
        ];

        try {
            // Use DaybookQuery service to get entries
            $daybook_data = $this->daybookQuery->getDaybookEntries(
                $business_id,
                $start_date,
                $end_date,
                $filters
            );

            // Log view for audit trail
            \Log::info('Day Book viewed', [
                'user_id' => auth()->id(),
                'business_id' => $business_id,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'filters' => $filters,
                'entry_count' => count($daybook_data['entries']),
            ]);

            return response()->json([
                'success' => true,
                'entries' => $daybook_data['entries'],
                'summary' => $daybook_data['summary'],
                'start_date' => $start_date,
                'end_date' => $end_date,
                'opening_balance' => $daybook_data['summary']['opening_balance'],
                'closing_balance' => $daybook_data['summary']['closing_balance'],
                'total_debit' => $daybook_data['summary']['total_debit'],
                'total_credit' => $daybook_data['summary']['total_credit'],
                'count_by_module' => $daybook_data['summary']['count_by_module'],
                'is_scoped' => $daybook_data['is_scoped'],
            ]);

        } catch (\Exception $e) {
            \Log::error('Day Book getData Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            return response()->json([
                'success' => false,
                'error' => 'Error generating Day Book: ' . $e->getMessage(),
                'entries' => [],
                'summary' => []
            ], 500);
        }
    }


    /**
     * Export Day Book as PDF
     */
    public function exportPdf(Request $request)
    {
        if (!auth()->user()->can('accounting.view_daybook')) {
            abort(403, 'Unauthorized action.');
        }

        // Similar logic to getData but return PDF view
        // Implementation will depend on PDF library (Snappy/Dompdf)
        // For now, redirect to print view
        
        return redirect()->route('accounting-reports.day-book.print', $request->all());
    }

    /**
     * Export Day Book as CSV/Excel
     */
    public function exportCsv(Request $request)
    {
        if (!auth()->user()->can('accounting.view_daybook')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        
        // Get date range
        $start_date = $request->input('start_date') ? 
            $this->transactionUtil->uf_date($request->input('start_date')) : 
            \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = $request->input('end_date') ? 
            $this->transactionUtil->uf_date($request->input('end_date')) : 
            \Carbon\Carbon::now()->format('Y-m-d');

        $permitted_locations = auth()->user()->permitted_locations();

        $filters = [
            'location_id' => $request->input('location_id'),
            'user_id' => $request->input('user_id'),
            'account_id' => $request->input('account_id'),
            'module_filter' => $request->input('module_filter'),
            'scope' => $request->input('scope', 'cash_bank'),
            'party_id' => $request->input('party_id'),
            'payment_method' => $request->input('payment_method'),
            'currency' => $request->input('currency'),
            'use_created_date' => $request->input('use_created_date', false),
            'permitted_locations' => $permitted_locations,
        ];

        $daybook_data = $this->daybookQuery->getDaybookEntries(
            $business_id,
            $start_date,
            $end_date,
            $filters
        );

        // Log export for audit trail
        \Log::info('Day Book exported as CSV', [
            'user_id' => auth()->id(),
            'business_id' => $business_id,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'filters' => $filters,
        ]);

        // Generate CSV
        $filename = 'daybook_' . $start_date . '_to_' . $end_date . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($daybook_data) {
            $file = fopen('php://output', 'w');
            
            // Header row
            fputcsv($file, [
                'Date & Time',
                'Voucher No',
                'Voucher Type',
                'Module',
                'Party',
                'Location',
                'Payment Account',
                'Debit',
                'Credit',
                'Running Balance',
                'Narration',
                'User',
            ]);

            // Data rows
            foreach ($daybook_data['entries'] as $entry) {
                fputcsv($file, [
                    $entry['tx_datetime'],
                    $entry['voucher_no'],
                    $entry['voucher_type'],
                    $entry['module'],
                    $entry['party_name'],
                    $entry['location_name'],
                    $entry['payment_account_name'],
                    $entry['debit'],
                    $entry['credit'],
                    $entry['running_balance'],
                    $entry['narration'],
                    $entry['user_name'],
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Print-friendly view
     */
    public function print(Request $request)
    {
        if (!auth()->user()->can('accounting.view_daybook')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        
        // Get date range
        $start_date = $request->input('start_date') ? 
            $this->transactionUtil->uf_date($request->input('start_date')) : 
            \Carbon\Carbon::now()->format('Y-m-d');
        $end_date = $request->input('end_date') ? 
            $this->transactionUtil->uf_date($request->input('end_date')) : 
            \Carbon\Carbon::now()->format('Y-m-d');

        $permitted_locations = auth()->user()->permitted_locations();

        $filters = [
            'location_id' => $request->input('location_id'),
            'user_id' => $request->input('user_id'),
            'account_id' => $request->input('account_id'),
            'module_filter' => $request->input('module_filter'),
            'scope' => $request->input('scope', 'cash_bank'),
            'party_id' => $request->input('party_id'),
            'payment_method' => $request->input('payment_method'),
            'currency' => $request->input('currency'),
            'use_created_date' => $request->input('use_created_date', false),
            'permitted_locations' => $permitted_locations,
        ];

        $daybook_data = $this->daybookQuery->getDaybookEntries(
            $business_id,
            $start_date,
            $end_date,
            $filters
        );

        return view('accounting-reports::day-book.print', [
            'daybook_data' => $daybook_data,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'filters' => $filters,
            'business' => auth()->user()->business,
        ]);
    }
}


