<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Account;
use App\AccountTransaction;
use App\BusinessLocation;
use App\Utils\Util;
use Modules\AccountingReports\Entities\ChequeBookEntry;
use Yajra\DataTables\Facades\DataTables;
use DB;
use Carbon\Carbon;

class ChequeBookEntryController extends Controller
{
    public function index()
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        
        // Get bank accounts only
        $accounts = Account::where('business_id', $businessId)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('name', 'id')
            ->prepend(__('messages.all'), '');

        return view('accounting-reports::cheque-book.index', compact('accounts'));
    }

    public function create(Request $request)
    {
        if (!auth()->user()->can('accounting.add_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        
        // Get bank accounts from BankAccount module if bank_account_id is provided
        $selectedAccountId = null;
        $selectedChequeNo = null;
        $selectedChequeBookId = null;
        
        if ($request->has('bank_account_id') && !empty($request->bank_account_id)) {
            $bankAccount = \Modules\AccountingReports\Entities\BankAccount::where('business_id', $businessId)
                ->find($request->bank_account_id);
            if ($bankAccount && $bankAccount->linked_account_id) {
                $selectedAccountId = $bankAccount->linked_account_id;
            }
        }
        
        // If cheque_id is provided, pre-fill the form (for Issue button)
        if ($request->has('cheque_id') && !empty($request->cheque_id)) {
            $chequeEntry = ChequeBookEntry::where('business_id', $businessId)
                ->with(['account', 'chequeBook'])
                ->find($request->cheque_id);
            
            if ($chequeEntry) {
                $selectedAccountId = $chequeEntry->account_id;
                $selectedChequeNo = $chequeEntry->cheque_no;
                $selectedChequeBookId = $chequeEntry->cheque_book_id;
            }
        }
        
        $accounts = Account::where('business_id', $businessId)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('name', 'id');

        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::cheque-book.create', compact('accounts', 'locations', 'selectedAccountId', 'selectedChequeNo', 'selectedChequeBookId'));
    }

    /**
     * Get available/pending cheque numbers for an account
     */
    public function getAvailableCheques(Request $request)
    {
        if (!auth()->user()->can('accounting.add_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $accountId = $request->input('account_id');

        if (!$accountId) {
            return response()->json(['available_cheques' => []]);
        }

        // Get all active cheque books for this account
        $chequeBooks = \Modules\AccountingReports\Entities\ChequeBook::where('business_id', $businessId)
            ->where('account_id', $accountId)
            ->where('status', 'active')
            ->get();

        $availableCheques = [];

        if ($chequeBooks->isEmpty()) {
            return response()->json([
                'available_cheques' => [],
                'message' => 'No active cheque books found for this account'
            ]);
        }

        // Get all used cheque numbers for this account (already issued)
        // Include both with and without cheque_book_id to catch all issued cheques
        $usedCheques = \Modules\AccountingReports\Entities\ChequeBookEntry::where('business_id', $businessId)
            ->where('account_id', $accountId)
            ->where('type', 'issued')
            ->whereNull('deleted_at')
            ->pluck('cheque_no')
            ->map(function($no) {
                return (string) $no; // Ensure string comparison
            })
            ->unique()
            ->values()
            ->toArray();

        foreach ($chequeBooks as $chequeBook) {
            // Get all cheque numbers in the book
            $allCheques = $chequeBook->getChequeNumberSequence();
            
            // Get available cheques (not yet issued) from this book
            foreach ($allCheques as $chequeNo) {
                $chequeNoStr = (string) $chequeNo; // Ensure string comparison
                if (!in_array($chequeNoStr, $usedCheques, true)) {
                    $availableCheques[] = [
                        'cheque_no' => $chequeNoStr,
                        'cheque_book' => $chequeBook->book_name,
                        'display' => $chequeNoStr . ' (' . $chequeBook->book_name . ')'
                    ];
                }
            }
        }

        // Sort by cheque number (numeric sort for better ordering)
        usort($availableCheques, function($a, $b) {
            // Try numeric comparison first, fallback to string
            $aNum = (int) $a['cheque_no'];
            $bNum = (int) $b['cheque_no'];
            if ($aNum != $bNum) {
                return $aNum <=> $bNum;
            }
            return strcmp($a['cheque_no'], $b['cheque_no']);
        });

        return response()->json([
            'available_cheques' => $availableCheques,
            'count' => count($availableCheques)
        ]);
    }

    public function store(Request $request)
    {
        if (!auth()->user()->can('accounting.add_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $businessId = auth()->user()->business_id;
            DB::beginTransaction();

            // Validate cheque number uniqueness per account for issued cheques
            if ($request->type === 'issued') {
                $existing = ChequeBookEntry::where('business_id', $businessId)
                    ->where('account_id', $request->account_id)
                    ->where('cheque_no', $request->cheque_no)
                    ->where('type', 'issued')
                    ->whereNull('deleted_at')
                    ->first();

                if ($existing) {
                    DB::rollBack();
                    if ($request->ajax()) {
                        return response()->json([
                            'success' => false,
                            'msg' => __('accounting-reports::lang.cheque_no_already_exists')
                        ], 422);
                    }
                    return redirect()->back()
                        ->withInput()
                        ->with('error', __('accounting-reports::lang.cheque_no_already_exists'));
                }

                // Validate cheque number is from an active cheque book
                $chequeBook = \Modules\AccountingReports\Entities\ChequeBook::where('business_id', $businessId)
                    ->where('account_id', $request->account_id)
                    ->where('status', 'active')
                    ->get()
                    ->first(function($book) use ($request) {
                        return $book->isChequeInRange($request->cheque_no);
                    });

                if (!$chequeBook) {
                    DB::rollBack();
                    if ($request->ajax()) {
                        return response()->json([
                            'success' => false,
                            'msg' => __('accounting-reports::lang.cheque_no_not_in_active_book')
                        ], 422);
                    }
                    return redirect()->back()
                        ->withInput()
                        ->with('error', __('accounting-reports::lang.cheque_no_not_in_active_book'));
                }
            }

            $cheque = new ChequeBookEntry();
            $cheque->business_id = $businessId;
            $cheque->account_id = $request->account_id;
            $cheque->cheque_no = $request->cheque_no;
            $cheque->cheque_date = $this->convert_date($request->cheque_date);
            $cheque->amount = $request->amount;
            $cheque->type = $request->type;
            $cheque->status = $request->status ?? 'pending';
            $cheque->payee_name = $request->payee_name;
            $cheque->narration = $request->narration;
            $cheque->reference_no = $request->reference_no;
            $cheque->location_id = $request->location_id;
            $cheque->created_by = auth()->id();
            
            // Link to cheque book if found
            if (isset($chequeBook)) {
                $cheque->cheque_book_id = $chequeBook->id;
            }
            
            if ($request->has('cleared_date') && !empty($request->cleared_date)) {
                $cheque->cleared_date = $this->convert_date($request->cleared_date);
            }
            
            if ($request->has('bank_statement_ref')) {
                $cheque->bank_statement_ref = $request->bank_statement_ref;
            }

            $cheque->save();

            // If cheque is issued, create AccountTransaction for bank ledger
            if ($request->type === 'issued' && !empty($request->amount)) {
                $this->createChequeTransaction($cheque, $request);
            }

            DB::commit();

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'msg' => __('accounting-reports::lang.cheque_entry_added_success')
                ]);
            }

            return redirect()->route('accounting-reports.cheque-book.index')
                ->with('success', __('accounting-reports::lang.cheque_entry_added_success'));

        } catch (\Exception $e) {
            \Log::error('Cheque Entry Store Error: ' . $e->getMessage());
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'msg' => __('messages.something_went_wrong') . ': ' . $e->getMessage()
                ], 422);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', __('messages.something_went_wrong'));
        }
    }

    public function edit($id)
    {
        if (!auth()->user()->can('accounting.edit_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        $cheque = ChequeBookEntry::findOrFail($id);
        $businessId = auth()->user()->business_id;

        if ($cheque->business_id != $businessId) {
            abort(403, 'Unauthorized action.');
        }

        $accounts = Account::where('business_id', $businessId)
            ->where(function($q) {
                $q->whereRaw('LOWER(name) LIKE ?', ['%bank%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%cheque%'])
                  ->orWhereRaw('LOWER(name) LIKE ?', ['%transfer%']);
            })
            ->where('is_closed', 0)
            ->pluck('name', 'id');

        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::cheque-book.edit', compact('cheque', 'accounts', 'locations'));
    }

    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.edit_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $cheque = ChequeBookEntry::findOrFail($id);
            $businessId = auth()->user()->business_id;

            if ($cheque->business_id != $businessId) {
                abort(403, 'Unauthorized action.');
            }

            // Validate cheque number uniqueness (excluding current entry)
            if ($cheque->cheque_no != $request->cheque_no || $cheque->account_id != $request->account_id) {
                $existing = ChequeBookEntry::where('business_id', $businessId)
                    ->where('account_id', $request->account_id)
                    ->where('cheque_no', $request->cheque_no)
                    ->where('id', '!=', $id)
                    ->whereNull('deleted_at')
                    ->first();

                if ($existing) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', __('accounting-reports::lang.cheque_no_already_exists'));
                }
            }

            $cheque->account_id = $request->account_id;
            $cheque->cheque_no = $request->cheque_no;
            $cheque->cheque_date = $this->convert_date($request->cheque_date);
            $cheque->amount = $request->amount;
            $cheque->type = $request->type;
            $cheque->status = $request->status;
            $cheque->payee_name = $request->payee_name;
            $cheque->narration = $request->narration;
            $cheque->reference_no = $request->reference_no;
            $cheque->location_id = $request->location_id;
            
            if ($request->has('cleared_date') && !empty($request->cleared_date)) {
                $cheque->cleared_date = $this->convert_date($request->cleared_date);
            } else {
                $cheque->cleared_date = null;
            }
            
            if ($request->has('bounced_date') && !empty($request->bounced_date)) {
                $cheque->bounced_date = $this->convert_date($request->bounced_date);
            }
            
            if ($request->has('bounce_reason')) {
                $cheque->bounce_reason = $request->bounce_reason;
            }
            
            if ($request->has('bank_statement_ref')) {
                $cheque->bank_statement_ref = $request->bank_statement_ref;
            }
            
            if ($request->has('bank_statement_date') && !empty($request->bank_statement_date)) {
                $cheque->bank_statement_date = $this->convert_date($request->bank_statement_date);
            }

            $cheque->save();

            // Update AccountTransaction if cheque is issued and amount changed
            if ($cheque->type === 'issued' && !empty($cheque->amount)) {
                // Find existing transaction
                $existingTransaction = AccountTransaction::where('account_id', $cheque->account_id)
                    ->where('sub_type', 'cheque_issued')
                    ->where('reff_no', 'CHEQUE-' . $cheque->cheque_no)
                    ->whereNull('deleted_at')
                    ->first();

                if ($existingTransaction) {
                    // Update existing transaction
                    $commonUtil = app(Util::class);
                    $existingTransaction->amount = $commonUtil->num_uf($request->amount);
                    $existingTransaction->operation_date = $cheque->cheque_date;
                    $existingTransaction->note = !empty($cheque->narration) 
                        ? $cheque->narration 
                        : 'Cheque Issued - ' . $cheque->cheque_no . ($cheque->payee_name ? ' to ' . $cheque->payee_name : '');
                    $existingTransaction->save();
                } else {
                    // Create new transaction if doesn't exist
                    $this->createChequeTransaction($cheque, $request);
                }
            }

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'msg' => __('accounting-reports::lang.cheque_entry_updated_success')
                ]);
            }

            return redirect()->route('accounting-reports.cheque-book.index')
                ->with('success', __('accounting-reports::lang.cheque_entry_updated_success'));

        } catch (\Exception $e) {
            \Log::error('Cheque Entry Update Error: ' . $e->getMessage());
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'msg' => __('messages.something_went_wrong') . ': ' . $e->getMessage()
                ], 422);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', __('messages.something_went_wrong'));
        }
    }

    public function destroy($id)
    {
        if (!auth()->user()->can('accounting.delete_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $cheque = ChequeBookEntry::findOrFail($id);
            $businessId = auth()->user()->business_id;

            if ($cheque->business_id != $businessId) {
                abort(403, 'Unauthorized action.');
            }

            $cheque->delete();

            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'msg' => __('accounting-reports::lang.cheque_entry_deleted_success')
                ]);
            }

            return redirect()->route('accounting-reports.cheque-book.index')
                ->with('success', __('accounting-reports::lang.cheque_entry_deleted_success'));

        } catch (\Exception $e) {
            \Log::error('Cheque Entry Delete Error: ' . $e->getMessage());
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'msg' => __('messages.something_went_wrong') . ': ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->with('error', __('messages.something_went_wrong'));
        }
    }

    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;

        $query = ChequeBookEntry::where('business_id', $businessId)
            ->with(['account', 'location', 'createdBy']);

        if ($request->has('account_id') && !empty($request->account_id)) {
            $query->where('account_id', $request->account_id);
        }

        if ($request->has('type') && !empty($request->type)) {
            $query->where('type', $request->type);
        }

        if ($request->has('status') && !empty($request->status)) {
            $query->where('status', $request->status);
        }

        if ($request->has('start_date') && !empty($request->start_date)) {
            $query->whereDate('cheque_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && !empty($request->end_date)) {
            $query->whereDate('cheque_date', '<=', $request->end_date);
        }

        return DataTables::of($query)
            ->addColumn('account_name', function ($row) {
                return $row->account->name ?? '-';
            })
            ->addColumn('location_name', function ($row) {
                return $row->location->name ?? '-';
            })
            ->addColumn('created_by_name', function ($row) {
                return $row->createdBy ? $row->createdBy->user_full_name : '-';
            })
            ->editColumn('cheque_date', function ($row) {
                return $row->cheque_date ? \Carbon\Carbon::parse($row->cheque_date)->format('d/m/Y') : '-';
            })
            ->editColumn('amount', function ($row) {
                return number_format($row->amount, 2);
            })
            ->editColumn('type', function ($row) {
                return ucfirst($row->type);
            })
            ->editColumn('status', function ($row) {
                $colors = [
                    'pending' => 'warning',
                    'cleared' => 'success',
                    'bounced' => 'danger',
                    'cancelled' => 'default',
                    'deposited' => 'info',
                ];
                $color = $colors[$row->status] ?? 'default';
                return '<span class="label label-' . $color . '">' . ucfirst($row->status) . '</span>';
            })
            ->addColumn('action', function ($row) {
                $html = '';
                
                // Issue button - only show for pending cheques that haven't been issued yet
                if (auth()->user()->can('accounting.add_cheque_entry') && $row->status === 'pending' && $row->type === 'issued' && empty($row->amount)) {
                    $html .= '<button type="button" class="btn btn-xs btn-success issue_cheque_btn" 
                        data-cheque-id="' . $row->id . '"
                        data-account-id="' . $row->account_id . '"
                        data-cheque-no="' . htmlspecialchars($row->cheque_no, ENT_QUOTES) . '"
                        data-cheque-book-id="' . ($row->cheque_book_id ?? '') . '"
                        title="' . __('accounting-reports::lang.issue_cheque') . '">
                        <i class="fa fa-check"></i> ' . __('accounting-reports::lang.issue') . '
                    </button> ';
                }
                
                if (auth()->user()->can('accounting.edit_cheque_entry')) {
                    $html .= '<a href="' . route('accounting-reports.cheque-book.edit', [$row->id]) . '" class="btn btn-xs btn-primary">
                        <i class="fa fa-edit"></i> ' . __('messages.edit') . '
                    </a> ';
                }
                
                if (auth()->user()->can('accounting.delete_cheque_entry')) {
                    $html .= '<button type="button" class="btn btn-xs btn-danger delete_cheque_entry" data-href="' . route('accounting-reports.cheque-book.destroy', [$row->id]) . '">
                        <i class="fa fa-trash"></i> ' . __('messages.delete') . '
                    </button>';
                }
                
                return $html;
            })
            ->rawColumns(['status', 'action'])
            ->make(true);
    }

    /**
     * Create AccountTransaction for issued cheque
     */
    private function createChequeTransaction($cheque, $request)
    {
        $commonUtil = app(Util::class);
        
        $transactionData = [
            'amount' => $commonUtil->num_uf($request->amount),
            'account_id' => $cheque->account_id,
            'type' => 'credit', // Issued cheque reduces bank balance (credit)
            'sub_type' => 'cheque_issued',
            'operation_date' => $cheque->cheque_date,
            'created_by' => auth()->id(),
            'note' => !empty($cheque->narration) 
                ? $cheque->narration 
                : 'Cheque Issued - ' . $cheque->cheque_no . ($cheque->payee_name ? ' to ' . $cheque->payee_name : ''),
            'reff_no' => 'CHEQUE-' . $cheque->cheque_no,
        ];

        $accountTransaction = AccountTransaction::createAccountTransaction($transactionData);
        
        return $accountTransaction;
    }

    /**
     * Get cheque issue history
     */
    public function getHistory(Request $request)
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $accountId = $request->input('account_id');

        $query = ChequeBookEntry::where('business_id', $businessId)
            ->where('type', 'issued')
            ->with(['account', 'location', 'createdBy', 'chequeBook']);

        if ($accountId) {
            $query->where('account_id', $accountId);
        }

        if ($request->has('start_date') && !empty($request->start_date)) {
            $query->whereDate('cheque_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && !empty($request->end_date)) {
            $query->whereDate('cheque_date', '<=', $request->end_date);
        }

        return DataTables::of($query)
            ->addColumn('account_name', function ($row) {
                return $row->account->name ?? '-';
            })
            ->addColumn('cheque_book_name', function ($row) {
                return $row->chequeBook->book_name ?? '-';
            })
            ->editColumn('cheque_date', function ($row) {
                return $row->cheque_date ? \Carbon\Carbon::parse($row->cheque_date)->format('d/m/Y') : '-';
            })
            ->editColumn('amount', function ($row) {
                return number_format($row->amount, 2);
            })
            ->editColumn('status', function ($row) {
                $colors = [
                    'pending' => 'warning',
                    'cleared' => 'success',
                    'bounced' => 'danger',
                    'cancelled' => 'default',
                    'deposited' => 'info',
                ];
                $color = $colors[$row->status] ?? 'default';
                return '<span class="label label-' . $color . '">' . ucfirst($row->status) . '</span>';
            })
            ->editColumn('cleared_date', function ($row) {
                return $row->cleared_date ? \Carbon\Carbon::parse($row->cleared_date)->format('d/m/Y') : '-';
            })
            ->rawColumns(['status'])
            ->make(true);
    }

    private function convert_date($date)
    {
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            return $date;
        }
        
        try {
            $dt = Carbon::createFromFormat('d/m/Y', $date);
            return $dt->format('Y-m-d');
        } catch (\Exception $e) {
            try {
                $dt = Carbon::createFromFormat('Y-m-d', $date);
                return $dt->format('Y-m-d');
            } catch (\Exception $e2) {
                return $date;
            }
        }
    }
}

