<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Account;
use App\AccountTransaction;
use App\BusinessLocation;
use App\Utils\Util;
use DataTables;
use DB;

class CashFlowController extends Controller
{
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    public function index()
    {
        if (!auth()->user()->can('accounting.view_cashflow')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $accounts = Account::forDropdown($businessId, false);
        $business_locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::cash-flow.index', compact('accounts', 'business_locations'));
    }

    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_cashflow')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;

        $accounts = AccountTransaction::join(
            'accounts as A',
            'account_transactions.account_id',
            '=',
            'A.id'
        )
        ->leftjoin(
            'transaction_payments as TP',
            'account_transactions.transaction_payment_id',
            '=',
            'TP.id'
        )
        ->leftjoin(
            'transaction_payments as child_payments',
            'TP.id',
            '=',
            'child_payments.parent_id'
        )
        ->leftjoin(
            'transactions as child_sells',
            'child_sells.id',
            '=',
            'child_payments.transaction_id'
        )
        ->leftJoin('users AS u', 'account_transactions.created_by', '=', 'u.id')
        ->leftJoin('contacts AS c', 'TP.payment_for', '=', 'c.id')
        ->where('A.business_id', $business_id)
        ->with(['transaction', 'transaction.contact', 'transfer_transaction', 'transaction.transaction_for'])
        ->select([
            'account_transactions.type',
            'account_transactions.amount',
            'operation_date',
            'account_transactions.sub_type',
            'transfer_transaction_id',
            'account_transactions.transaction_id',
            'account_transactions.id',
            'A.name as account_name',
            'TP.payment_ref_no as payment_ref_no',
            'TP.is_return',
            'TP.is_advance',
            'TP.method',
            'TP.transaction_no',
            'TP.card_transaction_number',
            'TP.card_number',
            'TP.card_type',
            'TP.card_holder_name',
            'TP.card_month',
            'TP.card_year',
            'TP.card_security',
            'TP.cheque_number',
            'TP.bank_account_number',
            'account_transactions.account_id',
            DB::raw("CONCAT(COALESCE(u.surname, ''),' ',COALESCE(u.first_name, ''),' ',COALESCE(u.last_name,'')) as added_by"),
            'c.name as payment_for_contact',
            'c.type as payment_for_type',
            'c.supplier_business_name as payment_for_business_name',
            DB::raw('SUM(child_payments.amount) total_recovered'),
            DB::raw("GROUP_CONCAT(child_sells.invoice_no SEPARATOR ', ') as child_sells"),
        ])
        ->groupBy('account_transactions.id')
        ->orderBy('account_transactions.operation_date', 'asc');

        if (!empty($request->input('type'))) {
            $accounts->where('account_transactions.type', $request->input('type'));
        }

        $permitted_locations = auth()->user()->permitted_locations();
        $account_ids = [];
        if ($permitted_locations != 'all') {
            $locations = BusinessLocation::where('business_id', $business_id)
                            ->whereIn('id', $permitted_locations)
                            ->get();

            foreach ($locations as $location) {
                if (!empty($location->default_payment_accounts)) {
                    $default_payment_accounts = json_decode($location->default_payment_accounts, true);
                    foreach ($default_payment_accounts as $key => $account) {
                        if (!empty($account['is_enabled']) && !empty($account['account'])) {
                            $account_ids[] = $account['account'];
                        }
                    }
                }
            }

            $account_ids = array_unique($account_ids);
        }

        if ($permitted_locations != 'all') {
            $accounts->whereIn('A.id', $account_ids);
        }

        $location_id = $request->input('location_id');
        if (!empty($location_id)) {
            $location = BusinessLocation::find($location_id);
            if (!empty($location->default_payment_accounts)) {
                $default_payment_accounts = json_decode($location->default_payment_accounts, true);
                $account_ids = [];
                foreach ($default_payment_accounts as $key => $account) {
                    if (!empty($account['is_enabled']) && !empty($account['account'])) {
                        $account_ids[] = $account['account'];
                    }
                }

                $accounts->whereIn('A.id', $account_ids);
            }
        }

        if (!empty($request->input('account_id'))) {
            $accounts->where('A.id', $request->input('account_id'));
        }

        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');

        if (!empty($start_date) && !empty($end_date)) {
            $accounts->whereBetween(DB::raw('date(operation_date)'), [$start_date, $end_date]);
        }

        if ($request->has('only_payment_recovered')) {
            $accounts->leftJoin('transactions AS t', 'TP.transaction_id', '=', 't.id')
                ->whereDate('operation_date', '=', \Carbon\Carbon::now()->format('Y-m-d'))
                ->where(function ($q) {
                    $q->whereDate('t.transaction_date', '<', \Carbon\Carbon::now()->format('Y-m-d'))
                    ->orWhere('TP.is_advance', 1);
                });
        }

        $payment_types = $this->commonUtil->payment_types(null, true, $business_id);

        return DataTables::of($accounts)
            ->editColumn('method', function ($row) use ($payment_types) {
                if (!empty($row->method) && isset($payment_types[$row->method])) {
                    return $payment_types[$row->method];
                } else {
                    return '';
                }
            })
            ->addColumn('payment_details', function ($row) {
                $arr = [];
                if (!empty($row->transaction_no)) {
                    $arr[] = '<b>'.__('lang_v1.transaction_no').'</b>: '.$row->transaction_no;
                }

                if ($row->method == 'card' && !empty($row->card_transaction_number)) {
                    $arr[] = '<b>'.__('lang_v1.card_transaction_no').'</b>: '.$row->card_transaction_number;
                }

                if ($row->method == 'card' && !empty($row->card_number)) {
                    $arr[] = '<b>'.__('lang_v1.card_no').'</b>: '.$row->card_number;
                }
                if ($row->method == 'card' && !empty($row->card_type)) {
                    $arr[] = '<b>'.__('lang_v1.card_type').'</b>: '.$row->card_type;
                }
                if ($row->method == 'card' && !empty($row->card_holder_name)) {
                    $arr[] = '<b>'.__('lang_v1.card_holder_name').'</b>: '.$row->card_holder_name;
                }
                if ($row->method == 'card' && !empty($row->card_month)) {
                    $arr[] = '<b>'.__('lang_v1.month').'</b>: '.$row->card_month;
                }
                if ($row->method == 'card' && !empty($row->card_year)) {
                    $arr[] = '<b>'.__('lang_v1.year').'</b>: '.$row->card_year;
                }
                if ($row->method == 'card' && !empty($row->card_security)) {
                    $arr[] = '<b>'.__('lang_v1.security_code').'</b>: '.$row->card_security;
                }
                if (!empty($row->cheque_number)) {
                    $arr[] = '<b>'.__('lang_v1.cheque_no').'</b>: '.$row->cheque_number;
                }
                if (!empty($row->bank_account_number)) {
                    $arr[] = '<b>'.__('lang_v1.card_no').'</b>: '.$row->bank_account_number;
                }

                return implode(', ', $arr);
            })
            ->addColumn('debit', '@if($type == "debit")<span class="debit" data-orig-value="{{$amount}}">@format_currency($amount)</span>@endif')
            ->addColumn('credit', '@if($type == "credit")<span class="credit" data-orig-value="{{$amount}}">@format_currency($amount)</span>@endif')
            ->addColumn('balance', function ($row) {
                $balance_result = AccountTransaction::where('account_id', $row->account_id)
                                ->where('operation_date', '<=', $row->operation_date)
                                ->whereNull('deleted_at')
                                ->select(DB::raw("SUM(IF(type='credit', amount, -1 * amount)) as balance"))
                                ->first();
                
                $balance = $balance_result ? ($balance_result->balance ?? 0) : 0;

                return '<span class="balance" data-orig-value="'.$balance.'">'.$this->commonUtil->num_f($balance, true).'</span>';
            })
            ->addColumn('total_balance', function ($row) use ($business_id, $account_ids, $permitted_locations, $request) {
                $query = AccountTransaction::join(
                    'accounts as A',
                    'account_transactions.account_id',
                    '=',
                    'A.id'
                )
                ->where('A.business_id', $business_id)
                ->where('operation_date', '<=', $row->operation_date)
                ->whereNull('account_transactions.deleted_at')
                ->select(DB::raw("SUM(IF(type='credit', amount, -1 * amount)) as balance"));

                if (!empty($request->input('type'))) {
                    $query->where('type', $request->input('type'));
                }
                
                // Build account_ids for total balance calculation
                $total_account_ids = $account_ids;
                if ($permitted_locations != 'all' || !empty($request->input('location_id'))) {
                    if (empty($total_account_ids)) {
                        // Rebuild account_ids if needed
                        $total_account_ids = [];
                        $location_id = $request->input('location_id');
                        if (!empty($location_id)) {
                            $location = BusinessLocation::find($location_id);
                            if (!empty($location->default_payment_accounts)) {
                                $default_payment_accounts = json_decode($location->default_payment_accounts, true);
                                foreach ($default_payment_accounts as $key => $account) {
                                    if (!empty($account['is_enabled']) && !empty($account['account'])) {
                                        $total_account_ids[] = $account['account'];
                                    }
                                }
                            }
                        }
                    }
                    
                    if (!empty($total_account_ids)) {
                        $query->whereIn('A.id', $total_account_ids);
                    }
                }

                if (!empty($request->input('account_id'))) {
                    $query->where('A.id', $request->input('account_id'));
                }

                $balance_result = $query->first();
                $balance = $balance_result ? $balance_result->balance : 0;

                return '<span class="total_balance" data-orig-value="'.$balance.'">'.$this->commonUtil->num_f($balance, true).'</span>';
            })
            ->editColumn('operation_date', function ($row) {
                return $this->commonUtil->format_date($row->operation_date, true);
            })
            ->editColumn('sub_type', function ($row) {
                return $this->__getPaymentDetails($row);
            })
            ->removeColumn('id')
            ->rawColumns(['credit', 'debit', 'balance', 'sub_type', 'total_balance', 'payment_details'])
            ->make(true);
    }

    public function __getPaymentDetails($row)
    {
        $details = '';
        if (!empty($row->sub_type)) {
            $details = __('account.'.$row->sub_type);
            if (in_array($row->sub_type, ['fund_transfer', 'deposit']) && !empty($row->transfer_transaction)) {
                if ($row->type == 'credit') {
                    $details .= ' ( '.__('account.from').': '.$row->transfer_transaction->account->name.')';
                } else {
                    $details .= ' ( '.__('account.to').': '.$row->transfer_transaction->account->name.')';
                }
            }
        } else {
            if (!empty($row->transaction)) {
                if ($row->transaction->type == 'purchase') {
                    $details = __('purchase.purchase');
                } elseif ($row->transaction->type == 'sell') {
                    $details = __('sale.sale');
                } elseif ($row->transaction->type == 'expense') {
                    $details = __('lang_v1.expense');
                }
                if (!empty($row->transaction->contact)) {
                    $details .= ' ('.$row->transaction->contact->name.')';
                }
            }
        }

        return $details;
    }

    public function export(Request $request)
    {
        // TODO: Implement export
        return redirect()->back();
    }
}


