<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\BusinessLocation;
use App\Account;
use App\User;
use App\Utils\TransactionUtil;
use App\Utils\Util;
use Modules\AccountingReports\Services\CashbookQuery;
use Yajra\DataTables\Facades\DataTables;
use DB;
use Carbon\Carbon;

class CashBookController extends Controller
{
    protected $transactionUtil;
    protected $commonUtil;
    protected $cashbookQuery;

    public function __construct(TransactionUtil $transactionUtil, Util $commonUtil, CashbookQuery $cashbookQuery)
    {
        $this->transactionUtil = $transactionUtil;
        $this->commonUtil = $commonUtil;
        $this->cashbookQuery = $cashbookQuery;
    }

    public function index()
    {
        if (!auth()->user()->can('accounting.view_cashbook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $locations = BusinessLocation::forDropdown($businessId, true);
        $accounts = Account::forDropdown($businessId, true, false, false);
        $users = User::forDropdown($businessId, false, true, false);

        return view('accounting-reports::cash-book.index', compact(
            'locations',
            'accounts',
            'users'
        ));
    }

    /**
     * Get Cash Book data in Tally-style format
     * Shows only cash account transactions with receipts (debit) and payments (credit)
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_cashbook')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;

            // Handle date range input - convert from user format to Y-m-d if needed
            $start_date = null;
            $end_date = null;
            
            if ($request->has('start_date') && !empty($request->input('start_date'))) {
                $date_input = trim($request->input('start_date'));
                
                // Check if date is already in Y-m-d format (datepicker sends this)
                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_input)) {
                    $start_date = $date_input;
                } else {
                    // Try to parse using uf_date (business format)
                    try {
                        $start_date = $this->transactionUtil->uf_date($date_input);
                        if (!$start_date) {
                            $start_date = Carbon::now()->format('Y-m-d');
                        }
                    } catch (\Exception $e) {
                        // If parsing fails, default to today
                        \Log::warning('Cash Book start_date parsing error: ' . $e->getMessage() . ' - Input: ' . $date_input);
                        $start_date = Carbon::now()->format('Y-m-d');
                    }
                }
            } else {
                $start_date = Carbon::now()->format('Y-m-d');
            }

            if ($request->has('end_date') && !empty($request->input('end_date'))) {
                $date_input = trim($request->input('end_date'));
                
                // Check if date is already in Y-m-d format (datepicker sends this)
                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_input)) {
                    $end_date = $date_input;
                } else {
                    // Try to parse using uf_date (business format)
                    try {
                        $end_date = $this->transactionUtil->uf_date($date_input);
                        if (!$end_date) {
                            $end_date = Carbon::now()->format('Y-m-d');
                        }
                    } catch (\Exception $e) {
                        // If parsing fails, default to today
                        \Log::warning('Cash Book end_date parsing error: ' . $e->getMessage() . ' - Input: ' . $date_input);
                        $end_date = Carbon::now()->format('Y-m-d');
                    }
                }
            } else {
                $end_date = Carbon::now()->format('Y-m-d');
            }

            // Get filters
            $filters = [
                'location_id' => $request->input('location_id'),
                'user_id' => $request->input('user_id'),
                'account_id' => $request->input('account_id'),
                'permitted_locations' => auth()->user()->permitted_locations(),
            ];

            // Get cashbook data
            $cashbook_data = $this->cashbookQuery->getCashbookEntries(
                $business_id,
                $start_date,
                $end_date,
                $filters
            );

            $entries = $cashbook_data['entries'];
            $opening_balance = $cashbook_data['summary']['opening_balance'];
            
            // Calculate current period totals
            $current_period_debit = 0;
            $current_period_credit = 0;
            
            foreach ($entries as $entry) {
                $type = $entry['type'] ?? '';
                $amount = floatval($entry['amount'] ?? 0);
                
                if ($type == 'debit') {
                    $current_period_debit += $amount;
                } else {
                    $current_period_credit += $amount;
                }
            }
            
            // Current Total = Current Period + Opening Balance (only if opening balance exists)
            if (abs($opening_balance) > 0.01 && $start_date) {
                // Opening balance always shows in Debit side for cash book
                $current_period_debit += abs($opening_balance);
            }
            
            // Get closing balance (last running balance from entries)
            $closing_balance = 0;
            if (!empty($entries)) {
                $last_entry = end($entries);
                $closing_balance = $last_entry['running_balance'] ?? $opening_balance;
            } else {
                $closing_balance = $opening_balance;
            }

            return DataTables::of($entries)
                ->with([
                    'opening_balance' => $opening_balance,
                    'current_period_debit' => $current_period_debit,
                    'current_period_credit' => $current_period_credit,
                    'closing_balance' => $closing_balance
                ])
                ->editColumn('datetime', function ($row) {
                    $datetime = $row['datetime'] ?? null;
                    return $datetime ? $this->commonUtil->format_date($datetime, true) : '-';
                })
                ->editColumn('voucher_no', function ($row) {
                    $voucher_no = $row['voucher_no'] ?? '';
                    $transaction_id = $row['transaction_id'] ?? null;
                    $transaction_payment_id = $row['transaction_payment_id'] ?? null;
                    $account_transaction_id = $row['account_transaction_id'] ?? null;
                    $module = $row['module'] ?? '';
                    
                    if ($transaction_id || $transaction_payment_id || $account_transaction_id) {
                        $data_attr = '';
                        if ($transaction_id) {
                            $data_attr = 'data-transaction-id="' . $transaction_id . '" data-module="' . htmlspecialchars($module, ENT_QUOTES) . '"';
                        } elseif ($transaction_payment_id) {
                            $data_attr = 'data-payment-id="' . $transaction_payment_id . '" data-module="' . htmlspecialchars($module, ENT_QUOTES) . '"';
                        } elseif ($account_transaction_id) {
                            $data_attr = 'data-account-transaction-id="' . $account_transaction_id . '" data-module="' . htmlspecialchars($module, ENT_QUOTES) . '"';
                        }
                        return '<a href="#" class="voucher-details-link" ' . $data_attr . ' style="cursor: pointer; color: #0066cc;">' . $voucher_no . '</a>';
                    }
                    return $voucher_no;
                })
                ->editColumn('module', function ($row) {
                    $module = $row['module'] ?? '';
                    return ucfirst(str_replace('_', ' ', $module));
                })
                ->editColumn('party', function ($row) {
                    return $row['party'] ?? '-';
                })
                ->editColumn('location', function ($row) {
                    return $row['location'] ?? '-';
                })
                ->editColumn('account', function ($row) {
                    return $row['account'] ?? '-';
                })
                ->editColumn('debit', function ($row) {
                    $type = $row['type'] ?? '';
                    $amount = floatval($row['amount'] ?? 0);
                    return $type == 'debit' ? $this->commonUtil->num_f($amount) : '-';
                })
                ->editColumn('credit', function ($row) {
                    $type = $row['type'] ?? '';
                    $amount = floatval($row['amount'] ?? 0);
                    return $type == 'credit' ? $this->commonUtil->num_f($amount) : '-';
                })
                ->editColumn('narration', function ($row) {
                    return $row['narration'] ?? '-';
                })
                ->editColumn('user', function ($row) {
                    return $row['user'] ?? '-';
                })
                ->rawColumns(['voucher_no', 'module', 'debit', 'credit'])
                ->make(true);

        } catch (\Exception $e) {
            \Log::error('Cash Book Error: ' . $e->getMessage());
            \Log::error('Cash Book Trace: ' . $e->getTraceAsString());
            \Log::error('Cash Book File: ' . $e->getFile() . ' Line: ' . $e->getLine());
            \Log::error('Cash Book Request: ' . json_encode($request->all()));
            
            // Return proper DataTables JSON format even on error
            $draw = intval($request->get('draw', 1));
            return response()->json([
                'draw' => $draw,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error loading cash book data: ' . $e->getMessage() . ' (Check logs for details)'
            ], 200); // Return 200 to prevent DataTables error popup
        }
    }

    public function export(Request $request)
    {
        // TODO: Implement export
        return redirect()->back();
    }
}


