<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Entities\CapitalAccount;
use App\Utils\Util;
use DB;
use Yajra\DataTables\Facades\DataTables;

class CapitalAccountController extends Controller
{
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of capital accounts
     */
    public function index()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting-reports::capital-accounts.index');
    }

    /**
     * Get capital accounts data for DataTables
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;

        $query = CapitalAccount::with(['creator'])
            ->where('business_id', $business_id);

        // Filter by active status if provided
        if ($request->has('is_active') && $request->is_active !== '') {
            $query->where('is_active', $request->is_active);
        }

        // Filter by account type if provided
        if ($request->has('account_type') && $request->account_type !== '') {
            $query->where('account_type', $request->account_type);
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $html = '<div class="btn-group">';
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-info btn-xs view_account" data-href="' . 
                        action([self::class, 'show'], [$row->id]) . '"><i class="fa fa-eye"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-primary btn-xs edit_account" data-href="' . 
                        action([self::class, 'edit'], [$row->id]) . '"><i class="fa fa-edit"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-danger btn-xs delete_account" data-href="' . 
                        action([self::class, 'destroy'], [$row->id]) . '"><i class="fa fa-trash"></i></button>';
                }
                
                $html .= '</div>';
                return $html;
            })
            ->editColumn('account_type', function ($row) {
                return $row->account_type_label;
            })
            ->editColumn('opening_balance', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true">' . $row->opening_balance . '</span>';
            })
            ->editColumn('current_balance', function ($row) {
                $class = $row->current_balance >= 0 ? 'text-success' : 'text-danger';
                return '<span class="display_currency ' . $class . '" data-currency_symbol="true">' . $row->current_balance . '</span>';
            })
            ->editColumn('opening_date', function ($row) {
                return !empty($row->opening_date) ? $this->commonUtil->format_date($row->opening_date, false) : '-';
            })
            ->editColumn('is_active', function ($row) {
                if ($row->is_active) {
                    return '<span class="label label-success">' . __('accounting-reports::lang.active') . '</span>';
                } else {
                    return '<span class="label label-danger">' . __('accounting-reports::lang.inactive') . '</span>';
                }
            })
            ->editColumn('description', function ($row) {
                return !empty($row->description) ? $row->description : '-';
            })
            ->editColumn('contact_person', function ($row) {
                return !empty($row->contact_person) ? $row->contact_person : '-';
            })
            ->editColumn('created_at', function ($row) {
                return $this->commonUtil->format_date($row->created_at, true);
            })
            ->rawColumns(['action', 'is_active', 'opening_balance', 'current_balance'])
            ->toJson();
    }

    /**
     * Show the form for creating a new capital account
     */
    public function create()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $accountTypes = [
            'owner' => __('accounting-reports::lang.owner'),
            'partner' => __('accounting-reports::lang.partner'),
            'shareholder' => __('accounting-reports::lang.shareholder'),
            'investor' => __('accounting-reports::lang.investor'),
            'other' => __('accounting-reports::lang.other'),
        ];

        return view('accounting-reports::capital-accounts.create', compact('accountTypes'));
    }

    /**
     * Store a newly created capital account
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $user_id = auth()->user()->id;

            $request->validate([
                'account_name' => [
                    'required',
                    'string',
                    'max:255',
                    'unique:ar_capital_accounts,account_name,NULL,id,business_id,' . $business_id . ',deleted_at,NULL'
                ],
                'account_type' => 'required|in:owner,partner,shareholder,investor,other',
                'description' => 'nullable|string',
                'opening_balance' => 'nullable|numeric',
                'opening_date' => 'nullable|date',
                'contact_person' => 'nullable|string|max:255',
                'contact_email' => 'nullable|email|max:255',
                'contact_phone' => 'nullable|string|max:50',
                'address' => 'nullable|string',
            ]);

            $data = $request->only([
                'account_name', 
                'account_type', 
                'description', 
                'opening_balance', 
                'opening_date',
                'contact_person',
                'contact_email',
                'contact_phone',
                'address'
            ]);
            
            $data['business_id'] = $business_id;
            $data['created_by'] = $user_id;
            $data['is_active'] = $request->has('is_active') ? 1 : 1; // Default to active
            
            // Set current_balance = opening_balance initially
            $data['current_balance'] = $data['opening_balance'] ?? 0;
            
            // Format opening_date if provided
            if (!empty($data['opening_date'])) {
                $data['opening_date'] = $this->commonUtil->uf_date($data['opening_date'], false);
            }

            $capital_account = CapitalAccount::create($data);

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.capital_account_added_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Display the specified capital account
     */
    public function show($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $capital_account = CapitalAccount::where('business_id', $business_id)
            ->with(['creator'])
            ->findOrFail($id);

        return view('accounting-reports::capital-accounts.show', compact('capital_account'));
    }

    /**
     * Show the form for editing the specified capital account
     */
    public function edit($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $capital_account = CapitalAccount::where('business_id', $business_id)->findOrFail($id);

        $accountTypes = [
            'owner' => __('accounting-reports::lang.owner'),
            'partner' => __('accounting-reports::lang.partner'),
            'shareholder' => __('accounting-reports::lang.shareholder'),
            'investor' => __('accounting-reports::lang.investor'),
            'other' => __('accounting-reports::lang.other'),
        ];

        return view('accounting-reports::capital-accounts.edit', compact('capital_account', 'accountTypes'));
    }

    /**
     * Update the specified capital account
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $capital_account = CapitalAccount::where('business_id', $business_id)->findOrFail($id);

            $request->validate([
                'account_name' => [
                    'required',
                    'string',
                    'max:255',
                    'unique:ar_capital_accounts,account_name,' . $id . ',id,business_id,' . $business_id . ',deleted_at,NULL'
                ],
                'account_type' => 'required|in:owner,partner,shareholder,investor,other',
                'description' => 'nullable|string',
                'opening_balance' => 'nullable|numeric',
                'opening_date' => 'nullable|date',
                'contact_person' => 'nullable|string|max:255',
                'contact_email' => 'nullable|email|max:255',
                'contact_phone' => 'nullable|string|max:50',
                'address' => 'nullable|string',
            ]);

            $data = $request->only([
                'account_name', 
                'account_type', 
                'description', 
                'opening_balance', 
                'opening_date',
                'contact_person',
                'contact_email',
                'contact_phone',
                'address'
            ]);
            
            $data['is_active'] = $request->has('is_active') ? 1 : 0;
            
            // Format opening_date if provided
            if (!empty($data['opening_date'])) {
                $data['opening_date'] = $this->commonUtil->uf_date($data['opening_date'], false);
            }

            $capital_account->update($data);

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.capital_account_updated_success')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Remove the specified capital account
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $capital_account = CapitalAccount::where('business_id', $business_id)->findOrFail($id);

            $capital_account->delete();

            $output = [
                'success' => true,
                'msg' => __('accounting-reports::lang.capital_account_deleted_success')
            ];

            return response()->json($output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            return response()->json($output);
        }
    }
}

