<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Services\BankAccountService;
use Modules\AccountingReports\Services\ChequeEntryService;
use Modules\AccountingReports\Services\BankReconciliationService;
use Modules\AccountingReports\Http\Requests\StoreBankAccountRequest;
use Modules\AccountingReports\Http\Requests\UpdateBankAccountRequest;
use Modules\AccountingReports\Http\Requests\StoreChequeEntryRequest;
use Modules\AccountingReports\Http\Requests\ReconcileTransactionRequest;
use Modules\AccountingReports\Entities\BankAccount;
use Modules\AccountingReports\Entities\ChequeBookEntry;
use App\Utils\Util;
use App\BusinessLocation;
use Carbon\Carbon;
use Yajra\DataTables\Facades\DataTables;

class BankManagementController extends Controller
{
    protected $bankAccountService;
    protected $chequeEntryService;
    protected $reconciliationService;
    protected $commonUtil;

    public function __construct(
        BankAccountService $bankAccountService,
        ChequeEntryService $chequeEntryService,
        BankReconciliationService $reconciliationService,
        Util $commonUtil
    ) {
        $this->bankAccountService = $bankAccountService;
        $this->chequeEntryService = $chequeEntryService;
        $this->reconciliationService = $reconciliationService;
        $this->commonUtil = $commonUtil;
    }

    // ==================== BANK ACCOUNTS ====================

    /**
     * Display bank accounts listing
     */
    public function index()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting-reports::bank-management.index');
    }

    /**
     * Get bank accounts data for DataTables
     */
    public function getBankAccounts(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;

        $query = BankAccount::with(['creator', 'linkedAccount'])
            ->where('business_id', $businessId);

        // Apply filters
        if ($request->has('account_type') && $request->account_type !== '') {
            $query->where('account_type', $request->account_type);
        }

        if ($request->has('is_active') && $request->is_active !== '') {
            $query->where('is_active', $request->is_active);
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $html = '<div class="btn-group">';
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-info btn-xs view_bank_account" data-href="' . 
                        route('accounting-reports.bank-management.show', $row->id) . '"><i class="fa fa-eye"></i></button>';
                    
                    $html .= '<button type="button" class="btn btn-primary btn-xs edit_bank_account" data-href="' . 
                        route('accounting-reports.bank-management.edit', $row->id) . '"><i class="fa fa-edit"></i></button>';
                    
                    $html .= '<button type="button" class="btn btn-danger btn-xs delete_bank_account" data-href="' . 
                        route('accounting-reports.bank-management.destroy', $row->id) . '"><i class="fa fa-trash"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all') && $row->linked_account_id) {
                    $html .= '<button type="button" class="btn btn-warning btn-xs view_statement" data-href="' . 
                        route('accounting-reports.bank-management.statement', $row->id) . '" title="View Statement"><i class="fa fa-book"></i></button>';
                    
                    $html .= '<button type="button" class="btn btn-success btn-xs manage_bank" data-href="' . 
                        route('accounting-reports.bank-management.manage', $row->id) . '" title="Manage Bank"><i class="fa fa-cog"></i></button>';
                }
                
                $html .= '</div>';
                return $html;
            })
            ->editColumn('account_type', function ($row) {
                $badge = $row->account_type === 'liability' ? 'warning' : 'info';
                return '<span class="label label-' . $badge . '">' . ucfirst($row->account_type) . '</span>';
            })
            ->editColumn('is_active', function ($row) {
                return $row->is_active 
                    ? '<span class="label label-success">Active</span>'
                    : '<span class="label label-danger">Inactive</span>';
            })
            ->editColumn('balance', function ($row) {
                try {
                    $balance = $this->bankAccountService->getBalance($row);
                    return '<span class="display_currency" data-currency_symbol="true">' . $balance . '</span>';
                } catch (\Exception $e) {
                    return '<span class="text-danger">Error</span>';
                }
            })
            ->editColumn('opening_balance', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true">' . $row->opening_balance . '</span>';
            })
            ->rawColumns(['action', 'account_type', 'is_active', 'balance', 'opening_balance'])
            ->toJson();
    }

    /**
     * Show the form for creating a new bank account
     */
    public function create()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $availableAccounts = $this->bankAccountService->getAvailableAccounts($businessId);
        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::bank-management.create', compact('availableAccounts', 'locations'));
    }

    /**
     * Store a newly created bank account
     */
    public function store(StoreBankAccountRequest $request)
    {
        try {
            $businessId = auth()->user()->business_id;
            $userId = auth()->user()->id;

            $data = $request->validated();
            
            // Format opening balance
            if (isset($data['opening_balance'])) {
                $data['opening_balance'] = $this->commonUtil->num_uf($data['opening_balance']);
            }

            $bankAccount = $this->bankAccountService->createBankAccount($data, $businessId, $userId);

            return response()->json([
                'success' => true,
                'msg' => __('accounting-reports::lang.bank_account_added_success'),
                'data' => $bankAccount
            ]);

        } catch (\Exception $e) {
            \Log::error('BankManagementController::store - Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'msg' => __('messages.something_went_wrong') . ': ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified bank account
     */
    public function show($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $bankAccount = BankAccount::where('business_id', $businessId)->findOrFail($id);
        
        $balance = $this->bankAccountService->getBalance($bankAccount);
        $summary = $this->bankAccountService->getSummary($businessId);

        return view('accounting-reports::bank-management.show', compact('bankAccount', 'balance', 'summary'));
    }

    /**
     * Show the form for editing the specified bank account
     */
    public function edit($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $bankAccount = BankAccount::where('business_id', $businessId)->findOrFail($id);
        
        $availableAccounts = $this->bankAccountService->getAvailableAccounts($businessId);
        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::bank-management.edit', compact('bankAccount', 'availableAccounts', 'locations'));
    }

    /**
     * Update the specified bank account
     */
    public function update(UpdateBankAccountRequest $request, $id)
    {
        try {
            $businessId = auth()->user()->business_id;
            $bankAccount = BankAccount::where('business_id', $businessId)->findOrFail($id);

            $data = $request->validated();
            
            // Format opening balance
            if (isset($data['opening_balance'])) {
                $data['opening_balance'] = $this->commonUtil->num_uf($data['opening_balance']);
            }

            $bankAccount = $this->bankAccountService->updateBankAccount($bankAccount, $data, $businessId);

            return response()->json([
                'success' => true,
                'msg' => __('accounting-reports::lang.bank_account_updated_success'),
                'data' => $bankAccount
            ]);

        } catch (\Exception $e) {
            \Log::error('BankManagementController::update - Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'msg' => __('messages.something_went_wrong') . ': ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified bank account
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $businessId = auth()->user()->business_id;
            $bankAccount = BankAccount::where('business_id', $businessId)->findOrFail($id);

            $this->bankAccountService->deleteBankAccount($bankAccount);

            return response()->json([
                'success' => true,
                'msg' => __('accounting-reports::lang.bank_account_deleted_success')
            ]);

        } catch (\Exception $e) {
            \Log::error('BankManagementController::destroy - Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'msg' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get bank account statement
     */
    public function statement(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $bankAccount = BankAccount::where('business_id', $businessId)->findOrFail($id);

        $startDate = $request->input('start_date') 
            ? Carbon::parse($request->input('start_date'))
            : Carbon::now()->startOfMonth();
        
        $endDate = $request->input('end_date')
            ? Carbon::parse($request->input('end_date'))
            : Carbon::now();
        
        $locationId = $request->input('location_id');

        $statement = $this->bankAccountService->getStatement($bankAccount, $startDate, $endDate, $locationId);

        return response()->json([
            'success' => true,
            'data' => $statement
        ]);
    }

    /**
     * Bank account management dashboard
     */
    public function manage($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $bankAccount = BankAccount::where('business_id', $businessId)->findOrFail($id);

        if (!$bankAccount->linked_account_id) {
            return redirect()->back()
                ->with('error', 'Bank account must be linked to an account for management.');
        }

        // Get reconciliation summary
        $reconciliationSummary = $this->reconciliationService->getReconciliationSummary($bankAccount);
        
        // Get cheque summary
        $chequeSummary = $this->chequeEntryService->getChequeSummary($bankAccount->id);
        
        // Get current balance
        $balance = $this->bankAccountService->getBalance($bankAccount);

        return view('accounting-reports::bank-management.manage', compact(
            'bankAccount',
            'reconciliationSummary',
            'chequeSummary',
            'balance'
        ));
    }

    // ==================== CHEQUE ENTRIES ====================

    /**
     * Get cheque entries for a bank account
     */
    public function getChequeEntries(Request $request, $bankAccountId)
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $bankAccount = BankAccount::where('business_id', $businessId)->findOrFail($bankAccountId);

        $filters = [
            'type' => $request->input('type'),
            'status' => $request->input('status'),
            'start_date' => $request->input('start_date'),
            'end_date' => $request->input('end_date'),
        ];

        $entries = $this->chequeEntryService->getChequeEntries($bankAccount->id, $filters);

        return DataTables::of($entries)
            ->addColumn('action', function ($row) {
                $html = '<div class="btn-group">';
                
                if (auth()->user()->can('accounting.edit_cheque_entry') && $row->status === 'pending') {
                    $html .= '<button type="button" class="btn btn-primary btn-xs edit_cheque" data-href="' . 
                        route('accounting-reports.bank-management.cheque.edit', $row->id) . '"><i class="fa fa-edit"></i></button>';
                }
                
                if (auth()->user()->can('accounting.delete_cheque_entry') && $row->status === 'pending') {
                    $html .= '<button type="button" class="btn btn-danger btn-xs delete_cheque" data-href="' . 
                        route('accounting-reports.bank-management.cheque.destroy', $row->id) . '"><i class="fa fa-trash"></i></button>';
                }
                
                $html .= '</div>';
                return $html;
            })
            ->editColumn('type', function ($row) {
                $badge = $row->type === 'issued' ? 'warning' : 'info';
                return '<span class="label label-' . $badge . '">' . ucfirst($row->type) . '</span>';
            })
            ->editColumn('status', function ($row) {
                $badges = [
                    'pending' => 'default',
                    'cleared' => 'success',
                    'bounced' => 'danger',
                    'cancelled' => 'warning',
                    'deposited' => 'info',
                ];
                $badge = $badges[$row->status] ?? 'default';
                return '<span class="label label-' . $badge . '">' . ucfirst($row->status) . '</span>';
            })
            ->editColumn('amount', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true">' . $row->amount . '</span>';
            })
            ->rawColumns(['action', 'type', 'status', 'amount'])
            ->toJson();
    }

    /**
     * Store a new cheque entry
     */
    public function storeChequeEntry(StoreChequeEntryRequest $request)
    {
        try {
            $businessId = auth()->user()->business_id;
            $userId = auth()->user()->id;

            $data = $request->validated();
            
            // Format amount
            if (isset($data['amount'])) {
                $data['amount'] = $this->commonUtil->num_uf($data['amount']);
            }

            $chequeEntry = $this->chequeEntryService->createChequeEntry($data, $businessId, $userId);

            return response()->json([
                'success' => true,
                'msg' => __('accounting-reports::lang.cheque_entry_added_success'),
                'data' => $chequeEntry
            ]);

        } catch (\Exception $e) {
            \Log::error('BankManagementController::storeChequeEntry - Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'msg' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mark cheque as cleared
     */
    public function markChequeCleared(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.edit_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $chequeEntry = ChequeBookEntry::findOrFail($id);
            $clearedDate = $request->input('cleared_date') 
                ? Carbon::parse($request->input('cleared_date'))
                : Carbon::now();

            $chequeEntry = $this->chequeEntryService->markAsCleared($chequeEntry, $clearedDate);

            return response()->json([
                'success' => true,
                'msg' => 'Cheque marked as cleared successfully.',
                'data' => $chequeEntry
            ]);

        } catch (\Exception $e) {
            \Log::error('BankManagementController::markChequeCleared - Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'msg' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mark cheque as bounced
     */
    public function markChequeBounced(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.edit_cheque_entry')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $chequeEntry = ChequeBookEntry::findOrFail($id);
            
            $request->validate([
                'bounce_reason' => 'required|string|max:500',
                'bounced_date' => 'nullable|date',
            ]);

            $bouncedDate = $request->input('bounced_date')
                ? Carbon::parse($request->input('bounced_date'))
                : Carbon::now();

            $chequeEntry = $this->chequeEntryService->markAsBounced(
                $chequeEntry,
                $request->input('bounce_reason'),
                $bouncedDate
            );

            return response()->json([
                'success' => true,
                'msg' => 'Cheque marked as bounced successfully.',
                'data' => $chequeEntry
            ]);

        } catch (\Exception $e) {
            \Log::error('BankManagementController::markChequeBounced - Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'msg' => $e->getMessage()
            ], 500);
        }
    }

    // ==================== BANK RECONCILIATION ====================

    /**
     * Get unreconciled transactions
     */
    public function getUnreconciledTransactions(Request $request, $bankAccountId)
    {
        if (!auth()->user()->can('accounting.reconcile_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $bankAccount = BankAccount::where('business_id', $businessId)->findOrFail($bankAccountId);

        $asOfDate = $request->input('as_of_date')
            ? Carbon::parse($request->input('as_of_date'))
            : Carbon::now();

        $transactions = $this->reconciliationService->getUnreconciledTransactions($bankAccount, $asOfDate);

        return response()->json([
            'success' => true,
            'data' => $transactions
        ]);
    }

    /**
     * Reconcile transactions
     */
    public function reconcileTransactions(ReconcileTransactionRequest $request, $bankAccountId)
    {
        try {
            $businessId = auth()->user()->business_id;
            $bankAccount = BankAccount::where('business_id', $businessId)->findOrFail($bankAccountId);

            $clearedDate = Carbon::parse($request->input('cleared_date'));
            $bankStatementRef = $request->input('bank_statement_ref');

            $reconciled = $this->reconciliationService->reconcileTransactions(
                $bankAccount,
                $request->input('transaction_ids'),
                $clearedDate,
                $bankStatementRef
            );

            return response()->json([
                'success' => true,
                'msg' => count($reconciled) . ' transaction(s) reconciled successfully.',
                'data' => $reconciled
            ]);

        } catch (\Exception $e) {
            \Log::error('BankManagementController::reconcileTransactions - Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'msg' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get reconciliation statement
     */
    public function getReconciliationStatement(Request $request, $bankAccountId)
    {
        if (!auth()->user()->can('accounting.reconcile_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $bankAccount = BankAccount::where('business_id', $businessId)->findOrFail($bankAccountId);

        $startDate = $request->input('start_date')
            ? Carbon::parse($request->input('start_date'))
            : Carbon::now()->startOfMonth();
        
        $endDate = $request->input('end_date')
            ? Carbon::parse($request->input('end_date'))
            : Carbon::now();

        $statement = $this->reconciliationService->getReconciliationStatement($bankAccount, $startDate, $endDate);

        return response()->json([
            'success' => true,
            'data' => $statement
        ]);
    }

    /**
     * Get bank accounts summary
     */
    public function getSummary(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $asOfDate = $request->input('as_of_date')
            ? Carbon::parse($request->input('as_of_date'))
            : Carbon::now();

        $summary = $this->bankAccountService->getSummary($businessId, $asOfDate);

        return response()->json([
            'success' => true,
            'data' => $summary
        ]);
    }
}

