<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Entities\BankAccount;
use App\Account;
use App\AccountTransaction;
use App\Utils\Util;
use DB;
use Carbon\Carbon;
use Yajra\DataTables\Facades\DataTables;

class BankAccountController extends Controller
{
    protected $commonUtil;

    public function __construct(Util $commonUtil)
    {
        $this->commonUtil = $commonUtil;
    }

    /**
     * Display a listing of bank accounts
     */
    public function index()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting-reports::bank-accounts.index');
    }

    /**
     * Get bank accounts data for DataTables
     */
    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;

        $query = BankAccount::with(['creator', 'linkedAccount'])
            ->where('business_id', $business_id);

        // Filter by account type if provided
        if ($request->has('account_type') && $request->account_type !== '') {
            $query->where('account_type', $request->account_type);
        }

        // Filter by active status if provided
        if ($request->has('is_active') && $request->is_active !== '') {
            $query->where('is_active', $request->is_active);
        }

        return DataTables::of($query)
            ->addColumn('action', function ($row) {
                $html = '<div class="btn-group">';
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-info btn-xs view_bank_account" data-href="' . 
                        action([self::class, 'show'], [$row->id]) . '"><i class="fa fa-eye"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-primary btn-xs edit_bank_account" data-href="' . 
                        action([self::class, 'edit'], [$row->id]) . '"><i class="fa fa-edit"></i></button>';
                }
                
                if (auth()->user()->can('accounting.view_all')) {
                    $html .= '<button type="button" class="btn btn-danger btn-xs delete_bank_account" data-href="' . 
                        action([self::class, 'destroy'], [$row->id]) . '"><i class="fa fa-trash"></i></button>';
                }
                
                // Link to ledger
                if (auth()->user()->can('accounting.view_all') && $row->linked_account_id) {
                    $html .= '<button type="button" class="btn btn-warning btn-xs view_ledger" data-href="' . 
                        action([self::class, 'ledger'], [$row->id]) . '" title="' . __('accounting-reports::lang.view_ledger') . '"><i class="fa fa-book"></i></button>';
                }
                
                // Link to bank management (Reconciliation, Cheque Books, Cheque Entries)
                if (auth()->user()->can('accounting.view_all') && $row->linked_account_id) {
                    $html .= '<button type="button" class="btn btn-info btn-xs manage_bank" data-href="' . 
                        action([self::class, 'manage'], [$row->id]) . '" title="' . __('accounting-reports::lang.manage_bank') . '"><i class="fa fa-cog"></i></button>';
                }
                
                $html .= '</div>';
                return $html;
            })
            ->editColumn('account_type', function ($row) {
                $badge = $row->account_type === 'liability' ? 'warning' : 'info';
                return '<span class="label label-' . $badge . '">' . $row->account_type_label . '</span>';
            })
            ->editColumn('is_active', function ($row) {
                if ($row->is_active) {
                    return '<span class="label label-success">' . __('accounting-reports::lang.active') . '</span>';
                } else {
                    return '<span class="label label-danger">' . __('accounting-reports::lang.inactive') . '</span>';
                }
            })
            ->editColumn('description', function ($row) {
                return !empty($row->description) ? $row->description : '-';
            })
            ->editColumn('linked_account_id', function ($row) {
                if ($row->linkedAccount) {
                    $accountUrl = url('/account/account/' . $row->linked_account_id);
                    return '<a href="' . $accountUrl . '" target="_blank" class="text-primary">' . $row->linkedAccount->name . ' <i class="fa fa-external-link"></i></a>';
                }
                return '<span class="text-danger">' . __('accounting-reports::lang.not_linked') . '</span>';
            })
            ->editColumn('opening_balance', function ($row) {
                return '<span class="display_currency" data-currency_symbol="true">' . $row->opening_balance . '</span>';
            })
            ->editColumn('opening_date', function ($row) {
                return !empty($row->opening_date) ? $this->commonUtil->format_date($row->opening_date, false) : '-';
            })
            ->editColumn('created_at', function ($row) {
                return $this->commonUtil->format_date($row->created_at, true);
            })
            ->rawColumns(['action', 'account_type', 'is_active', 'linked_account_id', 'opening_balance'])
            ->toJson();
    }

    /**
     * Show the form for creating a new bank account
     */
    public function create()
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        
        // Get all available accounts for linking (from Chart of Accounts)
        // Use direct query to get all accounts
        $accounts = Account::where('business_id', $business_id)
            ->where('is_closed', 0)
            ->orderBy('name', 'asc')
            ->get();
        
        $availableAccounts = ['' => __('messages.please_select')];
        foreach ($accounts as $account) {
            $availableAccounts[$account->id] = $account->name;
        }

        return view('accounting-reports::bank-accounts.create', compact('availableAccounts'));
    }

    /**
     * Store a newly created bank account
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $user_id = auth()->user()->id;

            $request->validate([
                'account_name' => [
                    'required',
                    'string',
                    'max:255',
                    'unique:ar_bank_accounts,account_name,NULL,id,business_id,' . $business_id . ',deleted_at,NULL'
                ],
                'account_type' => 'required|in:liability,asset',
                'bank_name' => 'nullable|string|max:255',
                'account_number' => 'nullable|string|max:255',
                'ifsc_code' => 'nullable|string|max:50',
                'branch_name' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'linked_account_id' => 'nullable|exists:accounts,id',
                'opening_balance' => 'nullable|numeric',
                'opening_date' => 'nullable|date',
            ]);

            $data = $request->only([
                'account_name', 'account_type', 'bank_name', 'account_number', 
                'ifsc_code', 'branch_name', 'description', 'linked_account_id',
                'opening_balance', 'opening_date'
            ]);
            $data['business_id'] = $business_id;
            $data['created_by'] = $user_id;
            $data['is_active'] = $request->has('is_active') ? 1 : 1; // Default to active
            
            // Format opening_balance - ensure it's numeric
            if (isset($data['opening_balance'])) {
                $data['opening_balance'] = $this->commonUtil->num_uf($data['opening_balance']);
            } else {
                $data['opening_balance'] = 0;
            }
            
            // Format opening_date if provided
            if (!empty($data['opening_date'])) {
                $data['opening_date'] = $this->commonUtil->uf_date($data['opening_date'], false);
            }
            
            // Handle empty linked_account_id
            if (empty($data['linked_account_id'])) {
                $data['linked_account_id'] = null;
            }

            DB::beginTransaction();
            try {
                $bankAccount = BankAccount::create($data);

                // Create account transaction for opening balance if linked account and opening balance are provided
                if (!empty($data['linked_account_id']) && !empty($data['opening_balance']) && abs($data['opening_balance']) > 0.01) {
                    // Pass the original opening_date from request, not the formatted one
                    $openingDate = $request->input('opening_date');
                    $this->createOrUpdateOpeningBalanceTransaction($bankAccount, $data['opening_balance'], $openingDate);
                }

                DB::commit();

                $output = [
                    'success' => true,
                    'msg' => __('accounting-reports::lang.bank_account_added_success')
                ];
            } catch (\Exception $e) {
                DB::rollBack();
                throw $e;
            }

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Display the specified bank account
     */
    public function show($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $bankAccount = BankAccount::where('business_id', $business_id)
            ->with(['creator', 'linkedAccount'])
            ->findOrFail($id);

        return view('accounting-reports::bank-accounts.show', compact('bankAccount'));
    }

    /**
     * Show the form for editing the specified bank account
     */
    public function edit($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $bankAccount = BankAccount::where('business_id', $business_id)->findOrFail($id);

        // Get all available accounts for linking (from Chart of Accounts)
        // Use direct query to get all accounts
        $accounts = Account::where('business_id', $business_id)
            ->where('is_closed', 0)
            ->orderBy('name', 'asc')
            ->get();
        
        $availableAccounts = ['' => __('messages.please_select')];
        foreach ($accounts as $account) {
            $availableAccounts[$account->id] = $account->name;
        }

        return view('accounting-reports::bank-accounts.edit', compact('bankAccount', 'availableAccounts'));
    }

    /**
     * Update the specified bank account
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $bankAccount = BankAccount::where('business_id', $business_id)->findOrFail($id);

            $request->validate([
                'account_name' => [
                    'required',
                    'string',
                    'max:255',
                    'unique:ar_bank_accounts,account_name,' . $id . ',id,business_id,' . $business_id . ',deleted_at,NULL'
                ],
                'account_type' => 'required|in:liability,asset',
                'bank_name' => 'nullable|string|max:255',
                'account_number' => 'nullable|string|max:255',
                'ifsc_code' => 'nullable|string|max:50',
                'branch_name' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'linked_account_id' => 'nullable|exists:accounts,id',
                'opening_balance' => 'nullable|numeric',
                'opening_date' => 'nullable|date',
            ]);

            $data = $request->only([
                'account_name', 'account_type', 'bank_name', 'account_number', 
                'ifsc_code', 'branch_name', 'description', 'linked_account_id',
                'opening_balance', 'opening_date'
            ]);
            $data['is_active'] = $request->has('is_active') ? 1 : 0;
            
            // Format opening_balance - ensure it's numeric
            if (isset($data['opening_balance'])) {
                $data['opening_balance'] = $this->commonUtil->num_uf($data['opening_balance']);
            } else {
                $data['opening_balance'] = 0;
            }
            
            // Format opening_date if provided
            if (!empty($data['opening_date'])) {
                $data['opening_date'] = $this->commonUtil->uf_date($data['opening_date'], false);
            }
            
            // Handle empty linked_account_id
            if (empty($data['linked_account_id'])) {
                $data['linked_account_id'] = null;
            }

            DB::beginTransaction();
            try {
                $oldOpeningBalance = (float) ($bankAccount->opening_balance ?? 0);
                $oldLinkedAccountId = $bankAccount->linked_account_id;
                $oldTransactionId = $bankAccount->opening_balance_transaction_id;
                $newOpeningBalance = (float) ($data['opening_balance'] ?? 0);
                
                $bankAccount->update($data);
                
                // Refresh model to get updated values
                $bankAccount->refresh();

                // Handle opening balance transaction
                if (!empty($data['linked_account_id'])) {
                    // If opening balance changed or linked account changed, update transaction
                    if (abs($oldOpeningBalance - $newOpeningBalance) > 0.01 || $oldLinkedAccountId != $data['linked_account_id']) {
                        // Delete old transaction if exists
                        if ($oldTransactionId) {
                            $oldTransaction = AccountTransaction::find($oldTransactionId);
                            if ($oldTransaction) {
                                $oldTransaction->delete();
                            }
                        }
                        
                        // Create new transaction if opening balance is provided
                        if (!empty($data['opening_balance']) && abs($newOpeningBalance) > 0.01) {
                            // Pass the original opening_date from request, not the formatted one
                            $openingDate = $request->input('opening_date');
                            $this->createOrUpdateOpeningBalanceTransaction($bankAccount, $data['opening_balance'], $openingDate);
                        } else {
                            $bankAccount->opening_balance_transaction_id = null;
                            $bankAccount->save();
                        }
                    }
                } else {
                    // If linked account is removed, delete the transaction
                    if ($oldTransactionId) {
                        $oldTransaction = AccountTransaction::find($oldTransactionId);
                        if ($oldTransaction) {
                            $oldTransaction->delete();
                        }
                        $bankAccount->opening_balance_transaction_id = null;
                        $bankAccount->save();
                    }
                }

                DB::commit();

                $output = [
                    'success' => true,
                    'msg' => __('accounting-reports::lang.bank_account_updated_success')
                ];
            } catch (\Exception $e) {
                DB::rollBack();
                \Log::emergency('Bank Account Update Error: ' . $e->getMessage());
                \Log::emergency('Stack trace: ' . $e->getTraceAsString());
                throw $e;
            }

            if ($request->ajax()) {
                return response()->json($output);
            }

            return redirect()->action([self::class, 'index'])->with('status', $output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            if ($request->ajax()) {
                return response()->json($output);
            }

            return back()->with('error', $output)->withInput();
        }
    }

    /**
     * Remove the specified bank account
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $bankAccount = BankAccount::where('business_id', $business_id)->findOrFail($id);

            DB::beginTransaction();
            try {
                // Delete opening balance transaction if exists
                if ($bankAccount->opening_balance_transaction_id) {
                    $transaction = AccountTransaction::find($bankAccount->opening_balance_transaction_id);
                    if ($transaction) {
                        $transaction->delete();
                    }
                }

                $bankAccount->delete();

                DB::commit();

                $output = [
                    'success' => true,
                    'msg' => __('accounting-reports::lang.bank_account_deleted_success')
                ];
            } catch (\Exception $e) {
                DB::rollBack();
                throw $e;
            }

            return response()->json($output);

        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ];

            return response()->json($output);
        }
    }

    /**
     * Create or update opening balance transaction for linked account
     */
    protected function createOrUpdateOpeningBalanceTransaction($bankAccount, $openingBalance, $openingDate = null)
    {
        if (empty($bankAccount->linked_account_id)) {
            return null;
        }

        // Format amount - ensure it's numeric
        $amount = $this->commonUtil->num_uf($openingBalance);
        
        if (empty($amount) || abs($amount) < 0.01) {
            return null;
        }

        // Determine transaction type based on account type
        // Asset accounts: opening balance is debit (money coming in)
        // Liability accounts: opening balance is credit (money owed)
        $transactionType = $bankAccount->account_type === 'asset' ? 'debit' : 'credit';

        // Use opening date or current date
        if ($openingDate) {
            // If it's already a Carbon instance, use it
            if ($openingDate instanceof \Carbon\Carbon) {
                $operationDate = $openingDate;
            } elseif (is_string($openingDate)) {
                try {
                    $operationDate = $this->commonUtil->uf_date($openingDate, true);
                    if (!$operationDate) {
                        $operationDate = \Carbon::parse($openingDate);
                    }
                } catch (\Exception $e) {
                    $operationDate = \Carbon::parse($openingDate);
                }
            } else {
                $operationDate = \Carbon::parse($openingDate);
            }
        } else {
            $operationDate = \Carbon::now();
        }
        
        // Ensure it's a Carbon instance
        if (!$operationDate instanceof \Carbon\Carbon) {
            $operationDate = \Carbon::parse($operationDate);
        }
        
        // For opening balance, set the date to be at least 1 day before today
        // This ensures it's always included in opening balance calculations
        // If the opening date is in the future or today, set it to yesterday
        $today = \Carbon::now()->startOfDay();
        if ($operationDate->startOfDay()->greaterThanOrEqualTo($today)) {
            $operationDate = $today->copy()->subDay();
        }

        // Create account transaction
        $transactionData = [
            'amount' => $amount,
            'account_id' => $bankAccount->linked_account_id,
            'type' => $transactionType,
            'sub_type' => 'opening_balance',
            'operation_date' => $operationDate,
            'created_by' => auth()->user()->id,
            'note' => 'Opening balance for Bank Account: ' . $bankAccount->account_name,
        ];

        $accountTransaction = AccountTransaction::createAccountTransaction($transactionData);

        // Update bank account with transaction ID
        $bankAccount->opening_balance_transaction_id = $accountTransaction->id;
        $bankAccount->save();

        return $accountTransaction;
    }

    /**
     * Show ledger for a bank account
     */
    public function ledger($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $bankAccount = BankAccount::where('business_id', $business_id)
            ->with(['linkedAccount'])
            ->findOrFail($id);

        if (!$bankAccount->linked_account_id) {
            return response()->json([
                'success' => false,
                'msg' => __('accounting-reports::lang.linked_account_required_for_ledger')
            ], 400);
        }

        return view('accounting-reports::bank-accounts.ledger', compact('bankAccount'));
    }

    /**
     * Get ledger data for DataTables
     */
    public function getLedgerData(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $bankAccount = BankAccount::where('business_id', $business_id)
                ->findOrFail($id);

            if (!$bankAccount->linked_account_id) {
                return response()->json([
                    'draw' => intval($request->get('draw', 1)),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'error' => 'Linked account required'
                ], 200);
            }

            // Handle date range input
            $start_date = null;
            $end_date = null;
            
            if ($request->has('start_date') && !empty($request->input('start_date'))) {
                $date_input = trim($request->input('start_date'));
                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_input)) {
                    $start_date = $date_input;
                } else {
                    try {
                        $start_date = $this->commonUtil->uf_date($date_input);
                        if (!$start_date) {
                            $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                        }
                    } catch (\Exception $e) {
                        $start_date = \Carbon\Carbon::now()->format('Y-m-d');
                    }
                }
            } else {
                $start_date = \Carbon\Carbon::now()->format('Y-m-d');
            }

            if ($request->has('end_date') && !empty($request->input('end_date'))) {
                $date_input = trim($request->input('end_date'));
                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_input)) {
                    $end_date = $date_input;
                } else {
                    try {
                        $end_date = $this->commonUtil->uf_date($date_input);
                        if (!$end_date) {
                            $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                        }
                    } catch (\Exception $e) {
                        $end_date = \Carbon\Carbon::now()->format('Y-m-d');
                    }
                }
            } else {
                $end_date = \Carbon\Carbon::now()->format('Y-m-d');
            }

            // Calculate opening balance (same as account detail page)
            $opening_balance = (float) DB::table('account_transactions as AT')
                ->join('accounts as A', 'AT.account_id', '=', 'A.id')
                ->where('A.business_id', $business_id)
                ->where('A.id', $bankAccount->linked_account_id)
                ->where('AT.operation_date', '<', $start_date)
                ->whereNull('AT.deleted_at')
                ->select(DB::raw('SUM(IF(AT.type="credit", AT.amount, -1 * AT.amount)) as balance'))
                ->value('balance') ?? 0;

            // Get opening balance transaction to display as a row
            $opening_balance_transaction = AccountTransaction::join('accounts as A', 'account_transactions.account_id', '=', 'A.id')
                ->leftJoin('users AS u', 'account_transactions.created_by', '=', 'u.id')
                ->where('A.business_id', $business_id)
                ->where('A.id', $bankAccount->linked_account_id)
                ->where('account_transactions.sub_type', 'opening_balance')
                ->whereNull('account_transactions.deleted_at')
                ->select([
                    'account_transactions.id',
                    'account_transactions.operation_date',
                    'account_transactions.type',
                    'account_transactions.amount',
                    'account_transactions.sub_type',
                    'account_transactions.note',
                    'account_transactions.reff_no',
                    DB::raw("COALESCE(account_transactions.reff_no, CONCAT('TXN-', account_transactions.id)) as voucher_no"),
                    DB::raw("'Opening Balance' as module"),
                    DB::raw("'-' as party"),
                    DB::raw("'-' as location"),
                    DB::raw("A.name as account"),
                    DB::raw("COALESCE(CONCAT(u.first_name, ' ', u.last_name), '-') as user"),
                    DB::raw("'' as cheque_no"),
                ])
                ->first();

            // Get transactions for the date range
            $query = AccountTransaction::join('accounts as A', 'account_transactions.account_id', '=', 'A.id')
                ->leftJoin('transaction_payments AS tp', 'account_transactions.transaction_payment_id', '=', 'tp.id')
                ->leftJoin('contacts AS c', 'tp.payment_for', '=', 'c.id')
                ->leftJoin('users AS u', 'account_transactions.created_by', '=', 'u.id')
                ->leftJoin('transactions AS t', 'account_transactions.transaction_id', '=', 't.id')
                ->leftJoin('business_locations AS bl', 't.location_id', '=', 'bl.id')
                ->where('A.business_id', $business_id)
                ->where('A.id', $bankAccount->linked_account_id)
                ->whereNull('account_transactions.deleted_at')
                ->select([
                    'account_transactions.id',
                    'account_transactions.operation_date',
                    'account_transactions.type',
                    'account_transactions.amount',
                    'account_transactions.sub_type',
                    'account_transactions.note',
                    'account_transactions.reff_no',
                    DB::raw("COALESCE(t.invoice_no, t.ref_no, account_transactions.reff_no, CONCAT('TXN-', account_transactions.id)) as voucher_no"),
                    DB::raw("CASE 
                        WHEN account_transactions.sub_type = 'opening_balance' THEN 'Opening Balance'
                        WHEN t.type = 'sell' THEN 'Sale'
                        WHEN t.type = 'purchase' THEN 'Purchase'
                        WHEN t.type = 'expense' THEN 'Expense'
                        WHEN account_transactions.sub_type = 'fund_transfer' THEN 'Fund Transfer'
                        WHEN account_transactions.sub_type = 'deposit' THEN 'Deposit'
                        ELSE COALESCE(account_transactions.sub_type, 'Payment')
                    END as module"),
                    DB::raw("COALESCE(c.name, '-') as party"),
                    DB::raw("COALESCE(bl.name, '-') as location"),
                    DB::raw("A.name as account"),
                    DB::raw("COALESCE(CONCAT(u.first_name, ' ', u.last_name), '-') as user"),
                    DB::raw("COALESCE(tp.cheque_number, '') as cheque_no"),
                ]);

            if ($start_date && $end_date) {
                // Always include opening balance transactions regardless of date range
                $query->where(function($q) use ($start_date, $end_date) {
                    $q->whereBetween(DB::raw('DATE(account_transactions.operation_date)'), [$start_date, $end_date])
                      ->orWhere('account_transactions.sub_type', 'opening_balance');
                });
            }

            $transactions = $query->orderBy('account_transactions.operation_date', 'asc')
                ->orderBy('account_transactions.id', 'asc')
                ->get();

            // Format entries with running balance
            $entries = [];
            $running_balance = $opening_balance;
            
            // Add opening balance transaction as first row if it exists
            if ($opening_balance_transaction && $opening_balance != 0) {
                $ob_amount = (float) $opening_balance_transaction->amount;
                $ob_type = $opening_balance_transaction->type;
                
                // Format opening balance narration
                $ob_narration = !empty($opening_balance_transaction->note) 
                    ? $opening_balance_transaction->note 
                    : 'Opening balance for Bank Account: ' . $bankAccount->account_name;
                
                $entries[] = [
                    'datetime' => $opening_balance_transaction->operation_date ?? $start_date,
                    'voucher_no' => $opening_balance_transaction->voucher_no ?? 'TXN-' . $opening_balance_transaction->id,
                    'module' => 'Opening Balance',
                    'party' => '-',
                    'location' => '-',
                    'account' => $opening_balance_transaction->account ?? $bankAccount->linkedAccount->name ?? '-',
                    'type' => $ob_type,
                    'amount' => $ob_amount,
                    'debit' => $ob_type === 'debit' ? $ob_amount : 0,
                    'credit' => $ob_type === 'credit' ? $ob_amount : 0,
                    'cheque_no' => '-',
                    'narration' => $ob_narration,
                    'user' => $opening_balance_transaction->user ?? '-',
                    'running_balance' => $opening_balance,
                ];
            }
            
            // Add other transactions (excluding opening balance if already added)
            foreach ($transactions as $transaction) {
                // Skip opening balance transaction if we already added it
                if ($transaction->sub_type === 'opening_balance' && $opening_balance_transaction) {
                    continue;
                }
                
                $amount = (float) $transaction->amount;
                
                if ($transaction->type === 'debit') {
                    $running_balance += $amount;
                } else {
                    $running_balance -= $amount;
                }
                
                $entries[] = [
                    'datetime' => $transaction->operation_date,
                    'voucher_no' => $transaction->voucher_no ?? '-',
                    'module' => $transaction->module ?? '-',
                    'party' => $transaction->party ?? '-',
                    'location' => $transaction->location ?? '-',
                    'account' => $transaction->account ?? '-',
                    'type' => $transaction->type,
                    'amount' => $amount,
                    'debit' => $transaction->type === 'debit' ? $amount : 0,
                    'credit' => $transaction->type === 'credit' ? $amount : 0,
                    'cheque_no' => $transaction->cheque_no ?? '-',
                    'narration' => $transaction->note ?? '-',
                    'user' => $transaction->user ?? '-',
                    'running_balance' => $running_balance,
                ];
            }

            // Calculate totals
            $total_debit = array_sum(array_column($entries, 'debit'));
            $total_credit = array_sum(array_column($entries, 'credit'));
            $closing_balance = $running_balance;

            return DataTables::of($entries)
                ->with([
                    'opening_balance' => $opening_balance,
                    'total_debit' => $total_debit,
                    'total_credit' => $total_credit,
                    'closing_balance' => $closing_balance,
                    'current_period_debit' => $total_debit,
                    'current_period_credit' => $total_credit,
                ])
                ->editColumn('datetime', function ($row) {
                    return $this->commonUtil->format_date($row['datetime'], true);
                })
                ->editColumn('debit', function ($row) {
                    return $row['debit'] > 0 ? $this->commonUtil->num_f($row['debit']) : '-';
                })
                ->editColumn('credit', function ($row) {
                    return $row['credit'] > 0 ? $this->commonUtil->num_f($row['credit']) : '-';
                })
                ->rawColumns(['debit', 'credit'])
                ->make(true);

        } catch (\Exception $e) {
            \Log::error('Bank Account Ledger Error: ' . $e->getMessage());
            \Log::error('Bank Account Ledger Trace: ' . $e->getTraceAsString());
            
            $draw = intval($request->get('draw', 1));
            return response()->json([
                'draw' => $draw,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error loading ledger data: ' . $e->getMessage()
            ], 200);
        }
    }

    /**
     * Bank Management page - Shows Reconciliation, Cheque Books, and Cheque Entries
     */
    public function manage($id)
    {
        if (!auth()->user()->can('accounting.view_all')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = auth()->user()->business_id;
        $bankAccount = BankAccount::where('business_id', $business_id)
            ->with(['linkedAccount'])
            ->findOrFail($id);

        if (!$bankAccount->linked_account_id) {
            return redirect()->back()
                ->with('error', __('accounting-reports::lang.linked_account_required_for_management'));
        }

        return view('accounting-reports::bank-accounts.manage', compact('bankAccount'));
    }

    /**
     * Get reconciliation data for bank account
     */
    public function getReconciliationData(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $bankAccount = BankAccount::where('business_id', $business_id)->findOrFail($id);
            
            if (!$bankAccount->linked_account_id) {
                return response()->json([
                    'success' => false,
                    'msg' => __('accounting-reports::lang.linked_account_required_for_management')
                ], 422);
            }
            
            $accountId = $bankAccount->linked_account_id;
            $end_date = $request->input('end_date', Carbon::now()->format('Y-m-d'));
            $start_date = $request->input('start_date', Carbon::now()->subMonths(3)->format('Y-m-d'));
            
            $filters = [
                'location_id' => $request->input('location_id'),
                'user_id' => $request->input('user_id'),
                'account_id' => $accountId,
                'permitted_locations' => auth()->user()->permitted_locations(),
            ];

            $bankbookQuery = app(\Modules\AccountingReports\Services\BankbookQuery::class);
            $bankbook_data = $bankbookQuery->getBankbookEntries(
                $business_id,
                $start_date,
                $end_date,
                $filters
            );

            $entries = $bankbook_data['entries'];
            
            // Filter reconciled/unreconciled
            $unreconciled = array_filter($entries, function($entry) {
                return !($entry['is_reconciled'] ?? false);
            });

            return response()->json([
                'success' => true,
                'unreconciled' => array_values($unreconciled),
                'all_entries' => $entries,
            ]);

        } catch (\Exception $e) {
            \Log::error('Bank Reconciliation Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'msg' => 'Error loading reconciliation data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mark transaction as reconciled
     */
    public function markReconciled(Request $request)
    {
        if (!auth()->user()->can('accounting.reconcile_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $transaction_payment_id = $request->input('transaction_payment_id');
            $cheque_entry_id = $request->input('cheque_entry_id');
            $cleared_date = $request->input('cleared_date', Carbon::now()->format('Y-m-d'));
            $bank_statement_ref = $request->input('bank_statement_ref');

            if ($cheque_entry_id) {
                // Update cheque entry
                $cheque = \Modules\AccountingReports\Entities\ChequeBookEntry::findOrFail($cheque_entry_id);
                $cheque->status = 'cleared';
                $cheque->cleared_date = $cleared_date;
                if ($bank_statement_ref) {
                    $cheque->bank_statement_ref = $bank_statement_ref;
                }
                $cheque->save();
            }

            return response()->json([
                'success' => true,
                'msg' => __('accounting-reports::lang.reconciliation_marked_success')
            ]);

        } catch (\Exception $e) {
            \Log::error('Mark Reconciled Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'msg' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all cheque numbers for a bank account
     */
    public function getChequeNumbersData(Request $request, $id)
    {
        if (!auth()->user()->can('accounting.view_bankbook')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = auth()->user()->business_id;
            $bankAccount = BankAccount::where('business_id', $business_id)->findOrFail($id);
            
            if (!$bankAccount->linked_account_id) {
                return response()->json([
                    'draw' => intval($request->get('draw', 1)),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => []
                ]);
            }
            
            $accountId = $bankAccount->linked_account_id;
            
            // Get all cheque books for this account (include all statuses, not just active)
            $chequeBooks = \Modules\AccountingReports\Entities\ChequeBook::where('business_id', $business_id)
                ->where('account_id', $accountId)
                ->get();
            
            // Log for debugging
            \Log::info('Bank Account Cheque Numbers Query', [
                'bank_account_id' => $id,
                'linked_account_id' => $accountId,
                'cheque_books_found' => $chequeBooks->count(),
                'cheque_book_ids' => $chequeBooks->pluck('id')->toArray()
            ]);
            
            $allChequeNumbers = [];
            
            foreach ($chequeBooks as $chequeBook) {
                // Get all cheque numbers in sequence
                $chequeSequence = $chequeBook->getChequeNumberSequence();
                
                // Get used cheque entries - key by string cheque_no for proper matching
                $usedEntries = $chequeBook->chequeEntries()
                    ->with(['chequeBook'])
                    ->get()
                    ->keyBy(function($entry) {
                        return (string) $entry->cheque_no; // Ensure string key
                    });
                
                foreach ($chequeSequence as $chequeNo) {
                    $chequeNoStr = (string) $chequeNo; // Ensure string type
                    $entry = $usedEntries->get($chequeNoStr);
                    
                    $allChequeNumbers[] = [
                        'cheque_no' => $chequeNoStr,
                        'cheque_book_name' => $chequeBook->book_name,
                        'cheque_book_id' => $chequeBook->id,
                        'status' => $entry ? $entry->status : 'pending',
                        'type' => $entry ? $entry->type : '-',
                        'amount' => $entry ? $entry->amount : 0,
                        'payee_name' => $entry ? $entry->payee_name : '-',
                        'cheque_date' => $entry ? $entry->cheque_date : $chequeBook->issue_date,
                        'presented_date' => $entry ? $entry->presented_date : null,
                        'cleared_date' => $entry ? $entry->cleared_date : null,
                        'entry_id' => $entry ? $entry->id : null,
                    ];
                }
            }
            
            // Log for debugging
            \Log::info('All Cheque Numbers Collected', [
                'total_cheque_numbers' => count($allChequeNumbers),
                'cheque_books_processed' => $chequeBooks->count(),
                'cheque_book_names' => $chequeBooks->pluck('book_name')->toArray()
            ]);
            
            // Apply filters
            $statusFilter = $request->input('status');
            if ($statusFilter && $statusFilter !== 'all') {
                $allChequeNumbers = array_filter($allChequeNumbers, function($item) use ($statusFilter) {
                    return $item['status'] === $statusFilter;
                });
            }
            
            // Apply search
            $search = $request->input('search.value');
            if ($search) {
                $allChequeNumbers = array_filter($allChequeNumbers, function($item) use ($search) {
                    return stripos($item['cheque_no'], $search) !== false ||
                           stripos($item['cheque_book_name'], $search) !== false ||
                           ($item['payee_name'] !== '-' && stripos($item['payee_name'], $search) !== false);
                });
            }
            
            // Sort
            $orderColumn = $request->input('order.0.column', 0);
            $orderDir = $request->input('order.0.dir', 'asc');
            $columns = ['cheque_no', 'cheque_book_name', 'status', 'type', 'amount', 'payee_name', 'cheque_date'];
            $sortColumn = $columns[$orderColumn] ?? 'cheque_no';
            
            usort($allChequeNumbers, function($a, $b) use ($sortColumn, $orderDir) {
                if ($sortColumn === 'amount') {
                    $result = $a[$sortColumn] <=> $b[$sortColumn];
                } elseif ($sortColumn === 'cheque_date') {
                    $aDate = $a[$sortColumn] ? strtotime($a[$sortColumn]) : 0;
                    $bDate = $b[$sortColumn] ? strtotime($b[$sortColumn]) : 0;
                    $result = $aDate <=> $bDate;
                } else {
                    $result = strcmp($a[$sortColumn] ?? '', $b[$sortColumn] ?? '');
                }
                return $orderDir === 'desc' ? -$result : $result;
            });
            
            // Paginate
            $start = intval($request->input('start', 0));
            $length = intval($request->input('length', 25));
            $totalRecords = count($allChequeNumbers);
            $paginated = array_slice($allChequeNumbers, $start, $length);
            
            // Format for DataTables
            $data = array_map(function($item) {
                $statusColors = [
                    'pending' => 'warning',
                    'issued' => 'info',
                    'presented' => 'primary',
                    'cleared' => 'success',
                    'bounced' => 'danger',
                    'cancelled' => 'default',
                ];
                $statusColor = $statusColors[$item['status']] ?? 'default';
                
                $actionHtml = '';
                // Show issue button for pending cheques (even if no entry exists yet)
                if ($item['status'] === 'pending') {
                    if (auth()->user()->can('accounting.add_cheque_entry')) {
                        $url = route('accounting-reports.cheque-book.create');
                        if ($item['entry_id']) {
                            $url .= '?cheque_id=' . $item['entry_id'];
                        } else {
                            // Create new entry for this cheque number
                            $url .= '?account_id=' . $accountId . '&cheque_no=' . urlencode($item['cheque_no']) . '&cheque_book_id=' . $item['cheque_book_id'];
                        }
                        $actionHtml .= '<button type="button" class="btn btn-xs btn-success issue_cheque_btn" 
                            data-href="' . $url . '"
                            title="' . __('accounting-reports::lang.issue_cheque') . '">
                            <i class="fa fa-check"></i> ' . __('accounting-reports::lang.issue') . '
                        </button> ';
                    }
                }
                
                if ($item['entry_id'] && auth()->user()->can('accounting.edit_cheque_entry')) {
                    $actionHtml .= '<a href="' . route('accounting-reports.cheque-book.edit', [$item['entry_id']]) . '" class="btn btn-xs btn-primary">
                        <i class="fa fa-edit"></i> ' . __('messages.edit') . '
                    </a>';
                }
                
                return [
                    'cheque_no' => $item['cheque_no'],
                    'cheque_book_name' => $item['cheque_book_name'],
                    'status' => '<span class="label label-' . $statusColor . '">' . ucfirst($item['status']) . '</span>',
                    'type' => $item['type'],
                    'amount' => $item['amount'] > 0 ? number_format($item['amount'], 2) : '-',
                    'payee_name' => $item['payee_name'],
                    'cheque_date' => $item['cheque_date'] ? \Carbon\Carbon::parse($item['cheque_date'])->format('d/m/Y') : '-',
                    'presented_date' => $item['presented_date'] ? \Carbon\Carbon::parse($item['presented_date'])->format('d/m/Y') : '-',
                    'cleared_date' => $item['cleared_date'] ? \Carbon\Carbon::parse($item['cleared_date'])->format('d/m/Y') : '-',
                    'action' => $actionHtml,
                ];
            }, $paginated);
            
            return response()->json([
                'draw' => intval($request->get('draw', 1)),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $totalRecords,
                'data' => $data
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Get Cheque Numbers Error: ' . $e->getMessage());
            \Log::error('Get Cheque Numbers Trace: ' . $e->getTraceAsString());
            return response()->json([
                'draw' => intval($request->get('draw', 1)),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error loading cheque numbers: ' . $e->getMessage()
            ], 200);
        }
    }
}

