<?php

namespace Modules\AccountingReports\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\AccountingReports\Services\BalanceSheetService;
use App\Utils\TransactionUtil;
use App\BusinessLocation;

class BalanceSheetController extends Controller
{
    protected $balanceSheetService;
    protected $transactionUtil;

    public function __construct(BalanceSheetService $balanceSheetService, TransactionUtil $transactionUtil)
    {
        $this->balanceSheetService = $balanceSheetService;
        $this->transactionUtil = $transactionUtil;
    }

    public function index()
    {
        if (!auth()->user()->can('accounting.view_balance_sheet')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = auth()->user()->business_id;
        $locations = BusinessLocation::forDropdown($businessId, true);

        return view('accounting-reports::balance-sheet.index', compact('locations'));
    }

    public function getData(Request $request)
    {
        if (!auth()->user()->can('accounting.view_balance_sheet')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $businessId = auth()->user()->business_id;
            
            // Parse date using UltimatePOS date utility
            $asOfDate = !empty($request->input('end_date')) 
                ? $this->transactionUtil->uf_date($request->input('end_date'), false) 
                : date('Y-m-d');

            // Ensure date is in Y-m-d format
            if ($asOfDate instanceof \Carbon\Carbon) {
                $asOfDate = $asOfDate->format('Y-m-d');
            }

            $locationId = !empty($request->input('location_id')) ? $request->input('location_id') : null;

            $data = $this->balanceSheetService->getBalanceSheet($businessId, $asOfDate, $locationId);

            // Get business name
            $business = \App\Business::find($businessId);
            $data['business_name'] = $business->name ?? '';

            return response()->json($data);
        } catch (\Exception $e) {
            \Log::error('BalanceSheet getData Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'error' => 'Error generating Balance Sheet: ' . $e->getMessage(),
                'liabilities' => [],
                'assets' => [],
                'total_liabilities' => 0,
                'total_assets' => 0
            ], 500);
        }
    }

    public function export(Request $request)
    {
        if (!auth()->user()->can('accounting.export_reports')) {
            abort(403, 'Unauthorized action.');
        }

        // TODO: Implement export to PDF/Excel
        return redirect()->back();
    }
}


