<?php

namespace Modules\AccountingReports\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Loan extends Model
{
    use SoftDeletes;

    protected $table = 'ar_loans';

    protected $fillable = [
        'business_id',
        'loan_number',
        'loan_name',
        'description',
        'loan_amount',
        'interest_rate',
        'interest_type',
        'loan_date',
        'maturity_date',
        'lender_name',
        'lender_contact',
        'lender_address',
        'liability_account_id',
        'payment_account_id',
        'location_id',
        'status',
        'principal_paid',
        'interest_paid',
        'outstanding_principal',
        'outstanding_interest',
        'document',
        'reference_no',
        'created_by',
    ];

    protected $casts = [
        'loan_amount' => 'decimal:2',
        'interest_rate' => 'decimal:2',
        'principal_paid' => 'decimal:2',
        'interest_paid' => 'decimal:2',
        'outstanding_principal' => 'decimal:2',
        'outstanding_interest' => 'decimal:2',
        'loan_date' => 'date',
        'maturity_date' => 'date',
    ];

    /**
     * Get the business that owns the loan
     */
    public function business()
    {
        return $this->belongsTo(\App\Business::class, 'business_id');
    }

    /**
     * Get the location
     */
    public function location()
    {
        return $this->belongsTo(\App\BusinessLocation::class, 'location_id');
    }

    /**
     * Get the liability account
     */
    public function liabilityAccount()
    {
        return $this->belongsTo(\App\Account::class, 'liability_account_id');
    }

    /**
     * Get the payment account
     */
    public function paymentAccount()
    {
        return $this->belongsTo(\App\Account::class, 'payment_account_id');
    }

    /**
     * Get the user who created this loan
     */
    public function creator()
    {
        return $this->belongsTo(\App\User::class, 'created_by');
    }

    /**
     * Get all loan transactions
     */
    public function transactions()
    {
        return $this->hasMany(LoanTransaction::class, 'loan_id');
    }

    /**
     * Get total outstanding amount
     */
    public function getTotalOutstandingAttribute()
    {
        return $this->outstanding_principal + $this->outstanding_interest;
    }

    /**
     * Check if loan is fully paid
     */
    public function isFullyPaid()
    {
        return abs($this->outstanding_principal) < 0.01 && abs($this->outstanding_interest) < 0.01;
    }
}







