<?php

namespace Modules\AccountingReports\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class FixedAsset extends Model
{
    use SoftDeletes;

    protected $table = 'ar_fixed_assets';

    protected $fillable = [
        'business_id',
        'name',
        'description',
        'purchase_date',
        'amount',
        'current_book_value',
        'accumulated_depreciation',
        'depreciation_percentage',
        'is_active',
        'created_by',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'purchase_date' => 'date',
        'amount' => 'decimal:2',
        'current_book_value' => 'decimal:2',
        'accumulated_depreciation' => 'decimal:2',
        'depreciation_percentage' => 'decimal:2',
    ];

    /**
     * Get all revaluations for this asset
     */
    public function revaluations()
    {
        return $this->hasMany(FixedAssetRevaluation::class, 'fixed_asset_id')->orderBy('revaluation_date', 'desc');
    }

    /**
     * Get net book value (current book value - accumulated depreciation)
     */
    public function getNetBookValueAttribute()
    {
        $bookValue = $this->current_book_value ?? $this->amount;
        return $bookValue - ($this->accumulated_depreciation ?? 0);
    }

    /**
     * Calculate annual depreciation amount based on current book value
     */
    public function getAnnualDepreciationAttribute()
    {
        if (empty($this->depreciation_percentage) || $this->depreciation_percentage <= 0) {
            return 0;
        }
        
        $bookValue = $this->current_book_value ?? $this->amount;
        return ($bookValue * $this->depreciation_percentage) / 100;
    }

    /**
     * Get the business that owns the fixed asset
     */
    public function business()
    {
        return $this->belongsTo(\App\Business::class, 'business_id');
    }

    /**
     * Get the user who created this fixed asset
     */
    public function creator()
    {
        return $this->belongsTo(\App\User::class, 'created_by');
    }
}

