<?php

namespace Modules\AccountingReports\Entities;

use Illuminate\Database\Eloquent\Model;

class FifoLayer extends Model
{
    protected $table = 'ar_fifo_layers';

    protected $guarded = ['id'];

    protected $casts = [
        'quantity' => 'decimal:4',
        'unit_cost' => 'decimal:4',
        'total_cost' => 'decimal:4',
        'allocated_quantity' => 'decimal:4',
        'layer_date' => 'date',
    ];

    /**
     * Relationships
     */
    public function business()
    {
        return $this->belongsTo(\App\Business::class, 'business_id');
    }

    public function product()
    {
        return $this->belongsTo(\App\Product::class, 'product_id');
    }

    public function variation()
    {
        return $this->belongsTo(\App\Variation::class, 'variation_id');
    }

    public function location()
    {
        return $this->belongsTo(\App\BusinessLocation::class, 'location_id');
    }

    public function transaction()
    {
        return $this->belongsTo(\App\Transaction::class, 'transaction_id');
    }

    public function journalEntryLines()
    {
        return $this->hasMany(JournalEntryLine::class, 'fifo_layer_id');
    }

    /**
     * Get available quantity (quantity - allocated_quantity)
     */
    public function getAvailableQuantityAttribute()
    {
        return $this->quantity - $this->allocated_quantity;
    }

    /**
     * Check if layer is fully allocated
     */
    public function isFullyAllocated()
    {
        return $this->available_quantity <= 0.0001;
    }

    /**
     * Scopes
     */
    public function scopeAvailable($query)
    {
        return $query->where('status', 'available')
            ->whereRaw('quantity > allocated_quantity');
    }

    public function scopeByProduct($query, $productId, $variationId = null, $locationId = null)
    {
        $query->where('product_id', $productId);
        
        if ($variationId) {
            $query->where('variation_id', $variationId);
        }
        
        if ($locationId) {
            $query->where('location_id', $locationId);
        }
        
        return $query;
    }
}


