<?php

namespace Modules\AccountingReports\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class ChequeBook extends Model
{
    use SoftDeletes;

    protected $table = 'ar_cheque_books';

    protected $guarded = ['id'];

    protected $casts = [
        'issue_date' => 'date',
        'expiry_date' => 'date',
        'total_cheques' => 'integer',
    ];

    /**
     * Relationships
     */
    public function business()
    {
        return $this->belongsTo(\App\Business::class, 'business_id');
    }

    public function account()
    {
        return $this->belongsTo(\App\Account::class, 'account_id');
    }

    public function location()
    {
        return $this->belongsTo(\App\BusinessLocation::class, 'location_id');
    }

    public function createdBy()
    {
        return $this->belongsTo(\App\User::class, 'created_by');
    }

    public function chequeEntries()
    {
        return $this->hasMany(ChequeBookEntry::class, 'cheque_book_id');
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeByAccount($query, $account_id)
    {
        return $query->where('account_id', $account_id);
    }

    /**
     * Get used cheque count
     */
    public function getUsedChequesAttribute()
    {
        // Count distinct cheque numbers (in case of duplicates)
        // Use groupBy to ensure we count unique cheque numbers
        return $this->chequeEntries()
            ->select('cheque_no')
            ->groupBy('cheque_no')
            ->get()
            ->count();
    }

    /**
     * Get remaining cheque count
     */
    public function getRemainingChequesAttribute()
    {
        return $this->total_cheques - $this->used_cheques;
    }

    /**
     * Check if cheque number is in range
     */
    public function isChequeInRange($cheque_no)
    {
        $start = (int) $this->start_cheque_no;
        $end = (int) $this->end_cheque_no;
        $check = (int) $cheque_no;
        
        return $check >= $start && $check <= $end;
    }

    /**
     * Generate cheque number sequence
     */
    public function getChequeNumberSequence()
    {
        $start = (int) $this->start_cheque_no;
        $end = (int) $this->end_cheque_no;
        $sequence = [];
        
        // Check if numbers are padded (e.g., 0001-0030)
        $padding = strlen($this->start_cheque_no);
        
        for ($i = $start; $i <= $end; $i++) {
            $sequence[] = str_pad($i, $padding, '0', STR_PAD_LEFT);
        }
        
        return $sequence;
    }
}










