<?php

namespace Modules\AccountingReports\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class BankAccount extends Model
{
    use SoftDeletes;

    protected $table = 'ar_bank_accounts';

    protected $fillable = [
        'business_id',
        'account_name',
        'account_type',
        'bank_name',
        'account_number',
        'ifsc_code',
        'branch_name',
        'description',
        'linked_account_id',
        'opening_balance',
        'opening_date',
        'opening_balance_transaction_id',
        'is_active',
        'created_by',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'opening_date' => 'date',
        'opening_balance' => 'decimal:2',
    ];

    /**
     * Get the business that owns this bank account
     */
    public function business()
    {
        return $this->belongsTo(\App\Business::class, 'business_id');
    }

    /**
     * Get the user who created this bank account
     */
    public function creator()
    {
        return $this->belongsTo(\App\User::class, 'created_by');
    }

    /**
     * Get the linked account (for bank-book integration)
     */
    public function linkedAccount()
    {
        return $this->belongsTo(\App\Account::class, 'linked_account_id');
    }

    /**
     * Get the opening balance transaction
     */
    public function openingBalanceTransaction()
    {
        return $this->belongsTo(\App\AccountTransaction::class, 'opening_balance_transaction_id');
    }

    /**
     * Get cheque entries for this bank account
     */
    public function chequeEntries()
    {
        return $this->hasMany(ChequeBookEntry::class, 'bank_account_id');
    }

    /**
     * Get bank reconciliations for this bank account
     */
    public function reconciliations()
    {
        return $this->hasMany(BankReconciliation::class, 'account_id', 'linked_account_id');
    }

    /**
     * Get account type label
     */
    public function getAccountTypeLabelAttribute()
    {
        return $this->account_type === 'liability' ? 'Liability' : 'Asset';
    }
}

