<?php

namespace Modules\AccountingReports\Database\Seeders;

use Illuminate\Database\Seeder;
use Modules\AccountingReports\Entities\ChartOfAccount;
use App\User;
use App\Business;

class AccountingReportsDatabaseSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        // Create default Chart of Accounts for all businesses
        $businesses = Business::all();
        
        foreach ($businesses as $business) {
            $this->seedChartOfAccounts($business);
        }
    }

    /**
     * Seed Chart of Accounts for a business
     */
    protected function seedChartOfAccounts(Business $business)
    {
        // Check if already seeded
        if (ChartOfAccount::where('business_id', $business->id)->exists()) {
            return;
        }

        $adminUser = User::where('business_id', $business->id)
            ->where('user_type', 'user')
            ->first();

        if (!$adminUser) {
            return;
        }

        $accounts = [
            // Current Assets
            ['code' => '1000', 'name' => 'Current Assets', 'account_group' => 'assets_current', 'account_type' => 'asset', 'parent_id' => null],
            ['code' => '1100', 'name' => 'Cash in Hand', 'account_group' => 'assets_current', 'account_type' => 'asset', 'parent_id' => null, 'is_control_account' => true, 'control_type' => 'cash'],
            ['code' => '1200', 'name' => 'Bank Accounts', 'account_group' => 'assets_current', 'account_type' => 'asset', 'parent_id' => null, 'is_control_account' => true, 'control_type' => 'bank'],
            ['code' => '1300', 'name' => 'Accounts Receivable', 'account_group' => 'assets_current', 'account_type' => 'asset', 'parent_id' => null, 'is_control_account' => true, 'control_type' => 'receivables', 'requires_contact' => true],
            ['code' => '1400', 'name' => 'Inventory', 'account_group' => 'assets_current', 'account_type' => 'asset', 'parent_id' => null, 'is_control_account' => true, 'control_type' => 'inventory'],
            ['code' => '1500', 'name' => 'Prepaid Expenses', 'account_group' => 'assets_current', 'account_type' => 'asset', 'parent_id' => null],
            
            // Non-Current Assets
            ['code' => '2000', 'name' => 'Non-Current Assets', 'account_group' => 'assets_non_current', 'account_type' => 'asset', 'parent_id' => null],
            ['code' => '2100', 'name' => 'Fixed Assets', 'account_group' => 'assets_non_current', 'account_type' => 'asset', 'parent_id' => null],
            ['code' => '2200', 'name' => 'Accumulated Depreciation', 'account_group' => 'assets_non_current', 'account_type' => 'asset', 'parent_id' => null],
            
            // Current Liabilities
            ['code' => '3000', 'name' => 'Current Liabilities', 'account_group' => 'liabilities_current', 'account_type' => 'liability', 'parent_id' => null],
            ['code' => '3100', 'name' => 'Accounts Payable', 'account_group' => 'liabilities_current', 'account_type' => 'liability', 'parent_id' => null, 'is_control_account' => true, 'control_type' => 'payables', 'requires_contact' => true],
            ['code' => '3200', 'name' => 'Short-term Loans', 'account_group' => 'liabilities_current', 'account_type' => 'liability', 'parent_id' => null],
            ['code' => '3300', 'name' => 'Accrued Expenses', 'account_group' => 'liabilities_current', 'account_type' => 'liability', 'parent_id' => null],
            
            // Non-Current Liabilities
            ['code' => '4000', 'name' => 'Non-Current Liabilities', 'account_group' => 'liabilities_non_current', 'account_type' => 'liability', 'parent_id' => null],
            ['code' => '4100', 'name' => 'Long-term Loans', 'account_group' => 'liabilities_non_current', 'account_type' => 'liability', 'parent_id' => null],
            
            // Equity
            ['code' => '5000', 'name' => 'Equity', 'account_group' => 'equity', 'account_type' => 'equity', 'parent_id' => null],
            ['code' => '5100', 'name' => 'Capital', 'account_group' => 'equity', 'account_type' => 'equity', 'parent_id' => null],
            ['code' => '5200', 'name' => 'Retained Earnings', 'account_group' => 'equity', 'account_type' => 'equity', 'parent_id' => null],
            ['code' => '5300', 'name' => 'Drawings', 'account_group' => 'equity', 'account_type' => 'equity', 'parent_id' => null],
            
            // Income
            ['code' => '6000', 'name' => 'Income', 'account_group' => 'income', 'account_type' => 'income', 'parent_id' => null],
            ['code' => '6100', 'name' => 'Sales Revenue', 'account_group' => 'income', 'account_type' => 'income', 'parent_id' => null],
            ['code' => '6200', 'name' => 'Other Income', 'account_group' => 'other_income', 'account_type' => 'income', 'parent_id' => null],
            
            // COGS
            ['code' => '7000', 'name' => 'Cost of Goods Sold', 'account_group' => 'cogs', 'account_type' => 'expense', 'parent_id' => null],
            ['code' => '7100', 'name' => 'COGS - Direct Materials', 'account_group' => 'cogs', 'account_type' => 'expense', 'parent_id' => null],
            
            // Expenses
            ['code' => '8000', 'name' => 'Operating Expenses', 'account_group' => 'expenses', 'account_type' => 'expense', 'parent_id' => null],
            ['code' => '8100', 'name' => 'Salaries & Wages', 'account_group' => 'expenses', 'account_type' => 'expense', 'parent_id' => null],
            ['code' => '8200', 'name' => 'Rent', 'account_group' => 'expenses', 'account_type' => 'expense', 'parent_id' => null],
            ['code' => '8300', 'name' => 'Utilities', 'account_group' => 'expenses', 'account_type' => 'expense', 'parent_id' => null],
            ['code' => '8400', 'name' => 'Office Supplies', 'account_group' => 'expenses', 'account_type' => 'expense', 'parent_id' => null],
            ['code' => '8500', 'name' => 'Depreciation', 'account_group' => 'expenses', 'account_type' => 'expense', 'parent_id' => null],
            
            // Control Accounts - Taxes, Discounts, Rounding
            ['code' => '9000', 'name' => 'Tax Control', 'account_group' => 'liabilities_current', 'account_type' => 'liability', 'parent_id' => null, 'is_control_account' => true, 'control_type' => 'tax'],
            ['code' => '9100', 'name' => 'Discount Control', 'account_group' => 'expenses', 'account_type' => 'expense', 'parent_id' => null, 'is_control_account' => true, 'control_type' => 'discount'],
            ['code' => '9200', 'name' => 'Rounding Account', 'account_group' => 'expenses', 'account_type' => 'expense', 'parent_id' => null, 'is_control_account' => true, 'control_type' => 'rounding'],
        ];

        $accountMap = [];
        foreach ($accounts as $accountData) {
            $parentId = null;
            if (isset($accountData['parent_id']) && $accountData['parent_id'] !== null) {
                // Find parent by code
                foreach ($accountMap as $code => $id) {
                    if (strpos($accountData['code'], $code) === 0 && strlen($accountData['code']) > strlen($code)) {
                        $parentId = $id;
                        break;
                    }
                }
            }

            $account = ChartOfAccount::create([
                'business_id' => $business->id,
                'code' => $accountData['code'],
                'name' => $accountData['name'],
                'account_group' => $accountData['account_group'],
                'account_type' => $accountData['account_type'],
                'parent_id' => $parentId,
                'is_control_account' => $accountData['is_control_account'] ?? false,
                'control_type' => $accountData['control_type'] ?? 'none',
                'requires_contact' => $accountData['requires_contact'] ?? false,
                'is_active' => true,
                'created_by' => $adminUser->id,
            ]);

            $accountMap[$accountData['code']] = $account->id;
        }
    }
}


