<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        Schema::create('ar_cheque_books', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('business_id')->unsigned();
            $table->foreign('business_id')->references('id')->on('business')->onDelete('cascade');
            
            // Bank account reference
            $table->integer('account_id')->unsigned();
            $table->foreign('account_id')->references('id')->on('accounts')->onDelete('cascade');
            
            // Cheque book details
            $table->string('book_name', 255)->comment('Cheque book name/identifier');
            $table->string('book_number', 100)->nullable()->comment('Cheque book number from bank');
            $table->string('start_cheque_no', 100)->comment('Starting cheque number in the book');
            $table->string('end_cheque_no', 100)->comment('Ending cheque number in the book');
            $table->integer('total_cheques')->unsigned()->comment('Total number of cheques in the book');
            
            // Status
            $table->enum('status', ['active', 'completed', 'cancelled'])->default('active');
            $table->date('issue_date')->nullable()->comment('Date when cheque book was issued/received');
            $table->date('expiry_date')->nullable()->comment('Expiry date if applicable');
            
            // Location
            $table->integer('location_id')->unsigned()->nullable();
            $table->foreign('location_id')->references('id')->on('business_locations')->onDelete('set null');
            
            $table->text('notes')->nullable();
            
            $table->integer('created_by')->unsigned();
            $table->foreign('created_by')->references('id')->on('users')->onDelete('cascade');
            
            $table->timestamps();
            $table->softDeletes();
            
            $table->index(['business_id', 'account_id', 'status'], 'cheque_books_business_account_status_idx');
            $table->index(['business_id', 'account_id', 'start_cheque_no'], 'cheque_books_range_idx');
        });
        
        // Add cheque_book_id to cheque entries
        if (Schema::hasTable('ar_cheque_book_entries')) {
            Schema::table('ar_cheque_book_entries', function (Blueprint $table) {
                $table->integer('cheque_book_id')->unsigned()->nullable()->after('account_id');
                $table->foreign('cheque_book_id')->references('id')->on('ar_cheque_books')->onDelete('set null');
                $table->index(['cheque_book_id', 'cheque_no']);
            });
        }
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        if (Schema::hasTable('ar_cheque_book_entries')) {
            Schema::table('ar_cheque_book_entries', function (Blueprint $table) {
                if (Schema::hasColumn('ar_cheque_book_entries', 'cheque_book_id')) {
                    $table->dropForeign(['cheque_book_id']);
                    $table->dropColumn('cheque_book_id');
                }
            });
        }
        
        Schema::dropIfExists('ar_cheque_books');
    }
};

