<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        Schema::create('ar_receivables_payables', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('business_id')->unsigned();
            $table->foreign('business_id')->references('id')->on('business')->onDelete('cascade');
            
            // Contact (Customer/Supplier)
            $table->integer('contact_id')->unsigned();
            $table->foreign('contact_id')->references('id')->on('contacts')->onDelete('cascade');
            
            // Type (AR or AP)
            $table->enum('type', ['receivable', 'payable'])->default('receivable');
            
            // Transaction reference
            $table->integer('transaction_id')->unsigned();
            $table->foreign('transaction_id')->references('id')->on('transactions')->onDelete('cascade');
            $table->integer('journal_entry_id')->unsigned()->nullable();
            $table->foreign('journal_entry_id')->references('id')->on('ar_journal_entry_headers')->onDelete('set null');
            
            // Document details
            $table->string('document_no', 100)->nullable();
            $table->date('document_date');
            $table->date('due_date')->nullable();
            
            // Amounts
            $table->decimal('invoice_amount', 22, 4)->default(0)->comment('Original invoice/bill amount');
            $table->decimal('allocated_amount', 22, 4)->default(0)->comment('Total payments/receipts applied');
            $table->decimal('balance', 22, 4)->default(0)->comment('invoice_amount - allocated_amount');
            $table->decimal('discount_amount', 22, 4)->default(0);
            $table->decimal('tax_amount', 22, 4)->default(0);
            
            // Status
            $table->enum('status', ['open', 'partially_paid', 'paid', 'overpaid', 'written_off', 'cancelled'])->default('open');
            
            // Aging
            $table->integer('days_past_due')->default(0);
            $table->enum('aging_bucket', ['current', 'b0_7', 'b8_30', 'b31_60', 'b61_90', 'b90_plus'])->nullable();
            
            // Currency
            $table->string('currency', 3)->default('PKR');
            $table->decimal('exchange_rate', 15, 6)->default(1.0);
            
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes - using custom names to avoid MySQL 64-character limit
            $table->index(['business_id', 'contact_id', 'type', 'status'], 'ar_arp_contact_status_idx');
            $table->index(['business_id', 'type', 'document_date'], 'ar_arp_doc_date_idx');
            $table->index(['business_id', 'type', 'days_past_due'], 'ar_arp_aging_idx');
            $table->index(['business_id', 'transaction_id'], 'ar_arp_trans_idx');
        });
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        Schema::dropIfExists('ar_receivables_payables');
    }
};


