<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        Schema::create('ar_journal_entry_headers', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('business_id')->unsigned();
            $table->foreign('business_id')->references('id')->on('business')->onDelete('cascade');
            
            // Voucher identification
            $table->string('voucher_no', 100)->comment('Unique voucher number (JE-001, SAL-001, etc.)');
            $table->date('voucher_date');
            $table->dateTime('transaction_date')->comment('Actual transaction timestamp');
            
            // Source tracking
            $table->enum('source_module', [
                'manual', 'sale', 'purchase', 'expense', 'payment', 
                'receipt', 'stock_adjustment', 'opening_balance', 
                'return', 'transfer', 'reconciliation', 'revaluation'
            ])->default('manual');
            $table->integer('source_transaction_id')->unsigned()->nullable()->comment('Reference to transactions.id');
            $table->foreign('source_transaction_id')->references('id')->on('transactions')->onDelete('set null');
            
            // Location & currency
            $table->integer('location_id')->unsigned()->nullable();
            $table->foreign('location_id')->references('id')->on('business_locations')->onDelete('set null');
            $table->string('currency', 3)->default('PKR');
            $table->decimal('exchange_rate', 15, 6)->default(1.0)->comment('Base currency conversion rate');
            
            // Narration/description
            $table->text('narration')->nullable();
            $table->text('reference')->nullable()->comment('External reference number');
            
            // Balance validation
            $table->decimal('total_debit', 22, 4)->default(0);
            $table->decimal('total_credit', 22, 4)->default(0);
            $table->boolean('is_balanced')->default(false)->comment('Dr == Cr validation flag');
            $table->decimal('balance_diff', 22, 4)->default(0)->comment('Dr - Cr (should be 0)');
            
            // Status
            $table->enum('status', ['draft', 'posted', 'cancelled', 'reversed'])->default('draft');
            $table->boolean('is_reversed')->default(false);
            $table->integer('reversed_by_journal_id')->unsigned()->nullable();
            
            // Audit
            $table->integer('created_by')->unsigned();
            $table->foreign('created_by')->references('id')->on('users')->onDelete('cascade');
            $table->integer('posted_by')->unsigned()->nullable();
            $table->foreign('posted_by')->references('id')->on('users')->onDelete('set null');
            $table->dateTime('posted_at')->nullable();
            
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes - using custom names to avoid MySQL 64-character limit
            $table->unique(['business_id', 'voucher_no'], 'ar_jeh_voucher_unique');
            $table->index(['business_id', 'voucher_date'], 'ar_jeh_date_idx');
            $table->index(['business_id', 'source_module', 'source_transaction_id'], 'ar_jeh_source_idx');
            $table->index(['business_id', 'location_id', 'voucher_date'], 'ar_jeh_loc_date_idx');
            $table->index('status', 'ar_jeh_status_idx');
        });
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        Schema::dropIfExists('ar_journal_entry_headers');
    }
};


