<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        Schema::create('ar_chart_of_accounts', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('business_id')->unsigned();
            $table->foreign('business_id')->references('id')->on('business')->onDelete('cascade');
            
            // Account identification
            $table->string('code', 50)->nullable()->comment('Account code (e.g., 1000, 2000)');
            $table->string('name');
            $table->text('description')->nullable();
            
            // Account classification
            $table->string('account_group', 50)->comment('assets_current, assets_non_current, liabilities_current, liabilities_non_current, equity, income, cogs, expenses, other_income, other_expenses');
            $table->enum('account_type', ['asset', 'liability', 'equity', 'income', 'expense', 'cogs', 'other'])->default('asset');
            
            // Hierarchy
            $table->integer('parent_id')->unsigned()->nullable();
            $table->foreign('parent_id')->references('id')->on('ar_chart_of_accounts')->onDelete('cascade');
            $table->integer('level')->default(0)->comment('Depth in hierarchy (0=root)');
            $table->string('path', 500)->nullable()->comment('Full path for hierarchy queries');
            
            // Control accounts (for AR/AP/Inventory reconciliation)
            $table->boolean('is_control_account')->default(false);
            $table->enum('control_type', ['receivables', 'payables', 'inventory', 'cash', 'bank', 'tax', 'discount', 'rounding', 'none'])->default('none');
            
            // Behavior flags
            $table->boolean('is_active')->default(true);
            $table->boolean('is_closed')->default(false);
            $table->boolean('allow_negative_balance')->default(false);
            $table->boolean('requires_contact')->default(false)->comment('For AR/AP accounts');
            
            // Mapping to existing UltimatePOS accounts (optional)
            $table->integer('account_id')->unsigned()->nullable();
            $table->foreign('account_id')->references('id')->on('accounts')->onDelete('set null');
            
            // Metadata
            $table->integer('created_by')->unsigned();
            $table->foreign('created_by')->references('id')->on('users')->onDelete('cascade');
            $table->integer('updated_by')->unsigned()->nullable();
            $table->foreign('updated_by')->references('id')->on('users')->onDelete('set null');
            
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes
            $table->index(['business_id', 'account_group']);
            $table->index(['business_id', 'account_type']);
            $table->index(['business_id', 'code']);
            $table->index('parent_id');
            $table->unique(['business_id', 'code']);
        });
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        Schema::dropIfExists('ar_chart_of_accounts');
    }
};


