<?php
namespace Modules\Accounting\Http\Controllers;

use App\Utils\ModuleUtil;
use App\Utils\Util;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Modules\Accounting\Entities\AccountingAccount;
use Modules\Accounting\Entities\AccountingAccountsTransaction;
use Modules\Accounting\Entities\AccountingAccTransMapping;
use Modules\Accounting\Utils\AccountingUtil;
use PDO;

class JournalEntryController extends Controller
{
    /**
     * All Utils instance.
     */
    protected $util;

    /**
     * Constructor
     *
     * @param  ProductUtils  $product
     * @return void
     */
    public function __construct(Util $util, ModuleUtil $moduleUtil, AccountingUtil $accountingUtil)
    {
        $this->util           = $util;
        $this->moduleUtil     = $moduleUtil;
        $this->accountingUtil = $accountingUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */

    /*-----------------MAIN INDEX-----------*/

    //  public function index()
    //  {
    //      $business_id = request()->session()->get('user.business_id');

    //     if (! (auth()->user()->can('superadmin') ||
    //         $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
    //         ! (auth()->user()->can('accounting.view_journal'))) {
    //         abort(403, 'Unauthorized action.');
    //     }

    //     if (request()->ajax()) {
    //         $journal = AccountingAccTransMapping::where('accounting_acc_trans_mappings.business_id', $business_id)
    //                     ->join('users as u', 'accounting_acc_trans_mappings.created_by', 'u.id')
    //                     ->where('type', 'journal_entry')
    //                     ->select(['accounting_acc_trans_mappings.id', 'ref_no', 'operation_date', 'note',
    //                         DB::raw("CONCAT(COALESCE(u.surname, ''),' ',COALESCE(u.first_name, ''),' ',COALESCE(u.last_name,'')) as added_by"),
    //                     ]);

    //         if (! empty(request()->start_date) && ! empty(request()->end_date)) {
    //             $start = request()->start_date;
    //             $end = request()->end_date;
    //             $journal->whereDate('accounting_acc_trans_mappings.operation_date', '>=', $start)
    //                         ->whereDate('accounting_acc_trans_mappings.operation_date', '<=', $end);
    //         }

    //         return Datatables::of($journal)
    //             ->addColumn(
    //                 'action', function ($row) {
    //                     $html = '<div class="btn-group">
    //                             <button type="button" class="tw-dw-btn tw-dw-btn-xs tw-dw-btn-outline tw-dw-btn-info tw-w-max dropdown-toggle"
    //                                 data-toggle="dropdown" aria-expanded="false">'.
    //                                 __('messages.actions').
    //                                 '<span class="caret"></span><span class="sr-only">Toggle Dropdown
    //                                 </span>
    //                             </button>
    //                             <ul class="dropdown-menu dropdown-menu-right" role="menu">';
    //                     if (auth()->user()->can('accounting.view_journal')) {
    //                         // $html .= '<li>
    //                         //         <a href="#" data-href="'.action([\Modules\Accounting\Http\Controllers\JournalEntryController::class, 'show'], [$row->id]).'">
    //                         //             <i class="fas fa-eye" aria-hidden="true"></i>'.__("messages.view").'
    //                         //         </a>
    //                         //         </li>';
    //                     }

    //                     if (auth()->user()->can('accounting.edit_journal')) {
    //                         $html .= '<li>
    //                                 <a href="'.action([\Modules\Accounting\Http\Controllers\JournalEntryController::class, 'edit'], [$row->id]).'">
    //                                     <i class="fas fa-edit"></i>'.__('messages.edit').'
    //                                 </a>
    //                             </li>';
    //                     }

    //                     if (auth()->user()->can('accounting.delete_journal')) {
    //                         $html .= '<li>
    //                                 <a href="#" data-href="'.action([\Modules\Accounting\Http\Controllers\JournalEntryController::class, 'destroy'], [$row->id]).'" class="delete_journal_button">
    //                                     <i class="fas fa-trash" aria-hidden="true"></i>'.__('messages.delete').'
    //                                 </a>
    //                                 </li>';
    //                     }

    //                     $html .= '</ul></div>';

    //                     return $html;
    //                 })
    //             ->rawColumns(['action'])
    //             ->make(true);
    //     }

    //     return view('accounting::journal_entry.index');
    // }

    /*-----------------MAIN INDEX END-----------*/

    /*-----------------MODIFY MAIN INDEX-----------*/
    public function index()
    {
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.view_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $journalEntries = AccountingAccTransMapping::where('accounting_acc_trans_mappings.business_id', $business_id)
                ->join('users as u', 'accounting_acc_trans_mappings.created_by', '=', 'u.id')
            // ->where('accounting_acc_trans_mappings.type', 'journal_entry')
                ->select([
                    'accounting_acc_trans_mappings.id',
                    'accounting_acc_trans_mappings.ref_no',
                    'accounting_acc_trans_mappings.operation_date',
                    'accounting_acc_trans_mappings.note',
                    'accounting_acc_trans_mappings.type',
                    DB::raw("CONCAT(COALESCE(u.surname, ''),' ',COALESCE(u.first_name, ''),' ',COALESCE(u.last_name,'')) as added_by"),
                ])->orderBy('accounting_acc_trans_mappings.operation_date', 'desc')
                ->paginate(5); // Paginate results (5 journal entries per page)

            // Add transactions along with accounting account info
            $journalEntries->getCollection()->transform(function ($journal) {
                // Fetch all transactions related to this journal entry and join `accounting_accounts`
                $transactions = DB::table('accounting_accounts_transactions as aat')
                    ->join('accounting_accounts as aa', 'aat.accounting_account_id', '=', 'aa.id') // Join accounting accounts
                    ->leftJoin('accounts as a', 'aa.account_id', '=', 'a.id')
                    ->where('aat.acc_trans_mapping_id', $journal->id)
                    ->select([
                        'aat.id as transaction_id',
                        'aat.accounting_account_id',
                        'aat.amount',
                        'aat.type',
                        'aat.sub_type',
                        'aat.map_type',
                        'aat.operation_date',
                        'aat.note',
                        'aa.name as account_name',
                        'a.account_number',
                        'aa.gl_code'
                    ])
                    ->get();

                // Convert transactions to an array
                $journal->transactions = $transactions;

                return $journal;
            });

            return response()->json($journalEntries);
        }

        return view('accounting::journal_entry.index');
    }

    /*-----------------MODIFY MAIN INDEX END-----------*/

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.add_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting::journal_entry.create');
    }

    //CREATE RECEIPT
    public function createReceipt()
    {
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.add_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting::journal_entry.create_receipt');
    }

    //CREATE EXPENSE
    public function createExpense()
    {
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.add_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting::journal_entry.create_expense');
    }

    //OPENING BALANCE 1
    public function openingBalance1()
    {
        return 'opening blance 1 test';
    }

    //OPENING BALANCE 2
    public function openingBalance2()
    {
        return 'opening blance 2 test';
    }

    //Print Journal
    public function journalPrint($id)
    {
        $business_id = request()->session()->get('user.business_id');
        $journal = AccountingAccTransMapping::where('accounting_acc_trans_mappings.business_id', $business_id)
            ->join('users as u', 'accounting_acc_trans_mappings.created_by', '=', 'u.id')
            ->leftJoin('business', 'accounting_acc_trans_mappings.business_id', '=', 'business.id')
            // ->where('accounting_acc_trans_mappings.type', 'journal_entry')
            ->select([
                'accounting_acc_trans_mappings.id',
                'accounting_acc_trans_mappings.ref_no',
                'accounting_acc_trans_mappings.operation_date',
                'accounting_acc_trans_mappings.note',
                'accounting_acc_trans_mappings.type',
                'business.name as business_name',
                DB::raw("CONCAT(COALESCE(u.surname, ''),' ',COALESCE(u.first_name, ''),' ',COALESCE(u.last_name,'')) as added_by"),
            ])
            ->find($id);
        $transactions = DB::table('accounting_accounts_transactions as aat')
            ->join('accounting_accounts as aa', 'aat.accounting_account_id', '=', 'aa.id') // Join accounting accounts
            ->leftJoin('accounts as a', 'aa.account_id', '=', 'a.id')
            ->where('aat.acc_trans_mapping_id', $journal->id)
            ->select([
                'aat.id as transaction_id',
                'aat.accounting_account_id',
                'aat.amount',
                'aat.type',
                'aat.sub_type',
                'aat.map_type',
                'aat.operation_date',
                'aat.note',
                'aa.name as account_name',
                'a.account_number',

            ])
            ->get();

        // Convert transactions to an array
        $journal->transactions = $transactions;

        // dd($journal);

        return view('accounting::journal_entry.printview', compact('journal'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  Request  $request
     * @return Response
     */
    public function store(Request $request)
    {
        // dd("request data", $request->all());
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.add_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $user_id = request()->session()->get('user.id');

            $account_ids  = $request->get('account_id');
            $credits      = $request->get('credit');
            $debits       = $request->get('debit');
            $journal_date = $request->get('journal_date');

            $accounting_settings = $this->accountingUtil->getAccountingSettings($business_id);

            $ref_no    = $request->get('ref_no');
            $ref_count = $this->util->setAndGetReferenceCount('journal_entry');
            if (empty($ref_no)) {
                $prefix = ! empty($accounting_settings['journal_entry_prefix']) ?
                $accounting_settings['journal_entry_prefix'] : '';

                //Generate reference number
                $ref_no = $this->util->generateReferenceNumber('journal_entry', $ref_count, $business_id, $prefix);
            }

            $acc_trans_mapping                 = new AccountingAccTransMapping();
            $acc_trans_mapping->business_id    = $business_id;
            $acc_trans_mapping->ref_no         = $ref_no;
            $acc_trans_mapping->note           = $request->get('note');
            $acc_trans_mapping->type           = 'journal_entry';
            $acc_trans_mapping->created_by     = $user_id;
            $acc_trans_mapping->operation_date = $this->util->uf_date($journal_date, true);
            $acc_trans_mapping->save();

            //save details in account trnsactions table
            foreach ($account_ids as $index => $account_id) {
                if (! empty($account_id)) {
                    $transaction_row                          = [];
                    $transaction_row['accounting_account_id'] = $account_id;

                    if (! empty($credits[$index])) {
                        $transaction_row['amount'] = $credits[$index];
                        $transaction_row['type']   = 'credit';
                    }

                    if (! empty($debits[$index])) {
                        $transaction_row['amount'] = $debits[$index];
                        $transaction_row['type']   = 'debit';
                    }

                    $transaction_row['created_by']           = $user_id;
                    $transaction_row['operation_date']       = $this->util->uf_date($journal_date, true);
                    $transaction_row['sub_type']             = 'journal_entry';
                    $transaction_row['acc_trans_mapping_id'] = $acc_trans_mapping->id;

                    $accounts_transactions = new AccountingAccountsTransaction();
                    $accounts_transactions->fill($transaction_row);
                    $accounts_transactions->save();
                }
            }

            DB::commit();

            $output = [
                'success' => 1,
                'msg'     => __('lang_v1.added_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => 0,
                'msg'     => __('messages.something_went_wrong'),
            ];
        }

        return redirect()->route('journal-entry.index')->with('status', $output);
    }

    //STORE RECEIPT
    public function storeReceipt(Request $request)
    {
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.add_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $user_id = request()->session()->get('user.id');

            $account_id     = $request->get('account_id');
            $payment_method = $request->get('payment_method');
            $credits        = $request->get('amount');
            $debits         = $request->get('amount');
            $journal_date   = $request->get('journal_date');
            // dd("request store receipt", $request->all(), $user_id,$account_id,$payment_method, $credits, $debits, $journal_date );

            $accounting_settings = $this->accountingUtil->getAccountingSettings($business_id);

            $ref_no    = $request->get('ref_no');
            $ref_count = $this->util->setAndGetReferenceCount('journal_entry');
            if (empty($ref_no)) {
                $prefix = ! empty($accounting_settings['journal_entry_prefix']) ?
                $accounting_settings['journal_entry_prefix'] : '';

                //Generate reference number
                $ref_no = $this->util->generateReferenceNumber('journal_entry', $ref_count, $business_id, $prefix);
            }

            $acc_trans_mapping                 = new AccountingAccTransMapping();
            $acc_trans_mapping->business_id    = $business_id;
            $acc_trans_mapping->ref_no         = $ref_no;
            $acc_trans_mapping->note           = $request->get('note');
            $acc_trans_mapping->type           = 'receipt';
            $acc_trans_mapping->created_by     = $user_id;
            $acc_trans_mapping->operation_date = $this->util->uf_date($journal_date, true);
            $acc_trans_mapping->save();

            //save details in account trnsactions table

            $fromAccount = [
                "accounting_account_id" => $account_id,
                "amount"                => $credits,
                "type"                  => "credit",
                "created_by"            => $user_id,
                "operation_date"        => $this->util->uf_date($journal_date, true),
                'sub_type'              => "receipt",
                "acc_trans_mapping_id"  => $acc_trans_mapping->id,
            ];
            AccountingAccountsTransaction::createTransaction($fromAccount);

            $toAccount = [
                "accounting_account_id" => $payment_method,
                "amount"                => $debits,
                "type"                  => "debit",
                "created_by"            => $user_id,
                "operation_date"        => $this->util->uf_date($journal_date, true),
                'sub_type'              => "receipt",
                "acc_trans_mapping_id"  => $acc_trans_mapping->id,
            ];
            AccountingAccountsTransaction::createTransaction($toAccount);

            DB::commit();

            $output = [
                'success' => 1,
                'msg'     => __('lang_v1.added_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => 0,
                'msg'     => __('messages.something_went_wrong'),
            ];
        }

        return redirect()->route('journal-entry.index')->with('status', $output);
    }

    //UPDATE RECEIPT
    public function updateReceipt(Request $request, $id)
    {
        // dd($request->all());
        $business_id = request()->session()->get('user.business_id');

        // Authorization check
        if (
            ! (auth()->user()->can('superadmin') ||
                ! $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module') ||
                ! auth()->user()->can('accounting.edit_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $user_id = request()->session()->get('user.id');

            // Fetch the existing receipt
            $acc_trans_mapping = AccountingAccTransMapping::where('business_id', $business_id)
                ->findOrFail($id);

            // Update receipt details
            $acc_trans_mapping->note           = $request->get('note');
            $acc_trans_mapping->operation_date = $this->util->uf_date($request->get('journal_date'), true);
            $acc_trans_mapping->save();

            // Fetch the existing transactions
            $existing_transactions = AccountingAccountsTransaction::where('acc_trans_mapping_id', $id)->get();

            // Update the first transaction (user account)
            $user_account_transaction = $existing_transactions->first();
            $user_account_transaction->update([
                "accounting_account_id" => $request->get('user_account_id'),
                "amount"                => $request->get('amount'),
                "type"                  => "credit",
                "operation_date"        => $this->util->uf_date($request->get('journal_date'), true),
            ]);

            // Update the second transaction (payment method)
            $payment_method_transaction = $existing_transactions->skip(1)->first();
            $payment_method_transaction->update([
                "accounting_account_id" => $request->get('payment_method_id'),
                "amount"                => $request->get('amount'),
                "type"                  => "debit",
                "operation_date"        => $this->util->uf_date($request->get('journal_date'), true),
            ]);

            DB::commit();

            $output = [
                'success' => 1,
                'msg'     => __('lang_v1.updated_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => 0,
                'msg'     => __('messages.something_went_wrong'),
            ];
        }

        return redirect()->route('journal-entry.index')->with('status', $output);
    }

    //STORE OPENING BALANCE DEBIT

    public function storeOpeningBalanceDebit(Request $request)
    {
        // dd($request->all());
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.add_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $user_id = request()->session()->get('user.id');

            $account_id     = $request->get('opening_balance_account');
            $payment_method = $request->get('user_accounts');
            $credits        = $request->get('opening_balance');
            $debits         = $request->get('opening_balance');
            $journal_date   = $request->get('date');
            // dd("request store receipt", $request->all(), $user_id,$account_id,$payment_method, $credits, $debits, $journal_date );

            $accounting_settings = $this->accountingUtil->getAccountingSettings($business_id);

            $ref_no    = $request->get('ref_no');
            $ref_count = $this->util->setAndGetReferenceCount('journal_entry');
            if (empty($ref_no)) {
                $prefix = ! empty($accounting_settings['journal_entry_prefix']) ?
                $accounting_settings['journal_entry_prefix'] : '';

                //Generate reference number
                $ref_no = $this->util->generateReferenceNumber('journal_entry', $ref_count, $business_id, $prefix);
            }

            $acc_trans_mapping                 = new AccountingAccTransMapping();
            $acc_trans_mapping->business_id    = $business_id;
            $acc_trans_mapping->ref_no         = $ref_no;
            $acc_trans_mapping->note           = $request->get('note');
            $acc_trans_mapping->type           = 'opening_balance_debit';
            $acc_trans_mapping->created_by     = $user_id;
            $acc_trans_mapping->operation_date = $this->util->uf_date($journal_date, true);
            $acc_trans_mapping->save();

            //save details in account trnsactions table

            $fromAccount = [
                "accounting_account_id" => $account_id,
                "amount"                => $credits,
                "type"                  => "credit",
                "created_by"            => $user_id,
                "operation_date"        => $this->util->uf_date($journal_date, true),
                'sub_type'              => "opening_balance_debit",
                "acc_trans_mapping_id"  => $acc_trans_mapping->id,
            ];
            AccountingAccountsTransaction::createTransaction($fromAccount);

            $toAccount = [
                "accounting_account_id" => $payment_method,
                "amount"                => $debits,
                "type"                  => "debit",
                "created_by"            => $user_id,
                "operation_date"        => $this->util->uf_date($journal_date, true),
                'sub_type'              => "opening_balance_debit",
                "acc_trans_mapping_id"  => $acc_trans_mapping->id,
            ];
            AccountingAccountsTransaction::createTransaction($toAccount);

            DB::commit();

            $output = [
                'success' => 1,
                'msg'     => __('lang_v1.added_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => 0,
                'msg'     => __('messages.something_went_wrong'),
            ];
        }

        return redirect()->route('journal-entry.index')->with('status', $output);
    }

        //UPDATE OPENING BALANCE DEBIT 
        public function updateOpeningBalanceDebit(Request $request, $id)
        {
            // dd($request->all());
            $business_id = request()->session()->get('user.business_id');
    
            if (
                ! (auth()->user()->can('superadmin') ||
                    $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
                ! (auth()->user()->can('accounting.edit_journal'))
            ) {
                abort(403, 'Unauthorized action.');
            }
    
            try {
                DB::beginTransaction();
    
                $user_id = request()->session()->get('user.id');
    
                $account_id     = $request->get('opening_balance_account');
                $payment_method = $request->get('user_accounts');
                $credits        = $request->get('opening_balance');
                $debits         = $request->get('opening_balance');
                $journal_date   = $request->get('date');
    
                $acc_trans_mapping = AccountingAccTransMapping::where('business_id', $business_id)
                    ->where('type', 'opening_balance_debit')
                    ->findOrFail($id);
                // Update mapping info
                $acc_trans_mapping->note           = $request->get('note');
                $acc_trans_mapping->operation_date = $this->util->uf_date($journal_date, true);
                $acc_trans_mapping->created_by     = $user_id;
                $acc_trans_mapping->save();
    
                // Delete previous transactions related to this mapping
                AccountingAccountsTransaction::where('acc_trans_mapping_id', $acc_trans_mapping->id)->delete();
    
    
                // Re-create updated transactions
                $fromAccount = [
                    "accounting_account_id" => $account_id,
                    "amount"                => $credits,
                    "type"                  => "credit",
                    "created_by"            => $user_id,
                    "operation_date"        => $this->util->uf_date($journal_date, true),
                    'sub_type'              => "opening_balance_debit",
                    "acc_trans_mapping_id"  => $acc_trans_mapping->id,
                ];
                AccountingAccountsTransaction::createTransaction($fromAccount);
                // dd($acc_trans_mapping, $user_id, $fromAccount); 
    
                $toAccount = [
                    "accounting_account_id" => $payment_method,
                    "amount"                => $debits,
                    "type"                  => "debit",
                    "created_by"            => $user_id,
                    "operation_date"        => $this->util->uf_date($journal_date, true),
                    'sub_type'              => "opening_balance_debit",
                    "acc_trans_mapping_id"  => $acc_trans_mapping->id
                ];
                AccountingAccountsTransaction::createTransaction($toAccount);
    
                DB::commit();
    
                $output = [
                    'success' => 1,
                    'msg'     => __('lang_v1.updated_success'),
                ];
            } catch (\Exception $e) {
                DB::rollBack();
                \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());
    
                $output = [
                    'success' => 0,
                    'msg'     => __('messages.something_went_wrong'),
                ];
            }
    
            return redirect()->route('journal-entry.index')->with('status', $output);
        }

    //STORE OPENING BALANCE CREDIT

    public function storeOpeningBalanceCredit(Request $request)
    {
        // dd($request->all());
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.add_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $user_id = request()->session()->get('user.id');

            $account_id     = $request->get('user_accounts');
            $payment_method = $request->get('opening_balance_account');
            $credits        = $request->get('opening_balance');
            $debits         = $request->get('opening_balance');
            $journal_date   = $request->get('date');
            // dd("request store receipt", $request->all(), $user_id,$account_id,$payment_method, $credits, $debits, $journal_date );

            $accounting_settings = $this->accountingUtil->getAccountingSettings($business_id);

            $ref_no    = $request->get('ref_no');
            $ref_count = $this->util->setAndGetReferenceCount('journal_entry');
            if (empty($ref_no)) {
                $prefix = ! empty($accounting_settings['journal_entry_prefix']) ?
                $accounting_settings['journal_entry_prefix'] : '';

                //Generate reference number
                $ref_no = $this->util->generateReferenceNumber('journal_entry', $ref_count, $business_id, $prefix);
            }

            $acc_trans_mapping                 = new AccountingAccTransMapping();
            $acc_trans_mapping->business_id    = $business_id;
            $acc_trans_mapping->ref_no         = $ref_no;
            $acc_trans_mapping->note           = $request->get('note');
            $acc_trans_mapping->type           = 'opening_balance_credit';
            $acc_trans_mapping->created_by     = $user_id;
            $acc_trans_mapping->operation_date = $this->util->uf_date($journal_date, true);
            $acc_trans_mapping->save();

            //save details in account trnsactions table

            $fromAccount = [
                "accounting_account_id" => $account_id,
                "amount"                => $credits,
                "type"                  => "credit",
                "created_by"            => $user_id,
                "operation_date"        => $this->util->uf_date($journal_date, true),
                'sub_type'              => "opening_balance_credit",
                "acc_trans_mapping_id"  => $acc_trans_mapping->id,
            ];
            AccountingAccountsTransaction::createTransaction($fromAccount);

            $toAccount = [
                "accounting_account_id" => $payment_method,
                "amount"                => $debits,
                "type"                  => "debit",
                "created_by"            => $user_id,
                "operation_date"        => $this->util->uf_date($journal_date, true),
                'sub_type'              => "opening_balance_credit",
                "acc_trans_mapping_id"  => $acc_trans_mapping->id,
            ];
            AccountingAccountsTransaction::createTransaction($toAccount);

            DB::commit();

            $output = [
                'success' => 1,
                'msg'     => __('lang_v1.added_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => 0,
                'msg'     => __('messages.something_went_wrong'),
            ];
        }

        return redirect()->route('journal-entry.index')->with('status', $output);
    }

    

    // UPDATE OPENING BALANCE CREDIT 
    public function updateOpeningBalanceCredit(Request $request)
    {
        // dd($request->all());
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.edit_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $user_id = request()->session()->get('user.id');

            $entry_id       = $request->get('id'); // ID of the entry to update
            $account_id     = $request->get('user_accounts'); // credit account
            $payment_method = $request->get('opening_balance_account'); // debit account
            $amount         = $request->get('opening_balance'); // both debit and credit
            $journal_date   = $request->get('date');
            $note           = $request->get('note');

            // Find the original journal entry
            $acc_trans_mapping = AccountingAccTransMapping::where('business_id', $business_id)
                ->where('type', 'opening_balance_credit')
                ->findOrFail($entry_id);

            // Update journal mapping
            $acc_trans_mapping->note = $note;
            $acc_trans_mapping->operation_date = $this->util->uf_date($journal_date, true);
            $acc_trans_mapping->created_by = $user_id;
            $acc_trans_mapping->save();

            // Delete old debit & credit transactions
            AccountingAccountsTransaction::where('acc_trans_mapping_id', $acc_trans_mapping->id)->delete();

            // Re-create credit transaction (from user_accounts)
            $fromAccount = [
                "accounting_account_id" => $account_id,
                "amount"                => $amount,
                "type"                  => "credit",
                "created_by"            => $user_id,
                "operation_date"        =>$this->util->uf_date($journal_date, true),
                'sub_type'              => "opening_balance_credit",
                "acc_trans_mapping_id"  => $acc_trans_mapping->id,
            ];
            AccountingAccountsTransaction::createTransaction($fromAccount);

            // Re-create debit transaction (to opening_balance_account)
            $toAccount = [
                "accounting_account_id" => $payment_method,
                "amount"                => $amount,
                "type"                  => "debit",
                "created_by"            => $user_id,
                "operation_date"        => $this->util->uf_date($journal_date, true),
                'sub_type'              => "opening_balance_credit",
                "acc_trans_mapping_id"  => $acc_trans_mapping->id,
            ];
            AccountingAccountsTransaction::createTransaction($toAccount);

            DB::commit();

            $output = [
                'success' => 1,
                'msg'     => __('lang_v1.updated_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => 0,
                'msg'     => __('messages.something_went_wrong'),
            ];
        }

        return redirect()->route('journal-entry.index')->with('status', $output);
    }



    // STORE EXPENSE
    public function storeExpense(Request $request)
    {
        // dd($request->all());
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.add_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $user_id = request()->session()->get('user.id');

            $account_id     = $request->get('payment_method');
            $payment_method = $request->get('expense_account_for'); //eikhane dukbe
            $credits        = $request->get('amount');
            $debits         = $request->get('amount');
            $journal_date   = $request->get('journal_date');
            // dd("request store receipt", $request->all(), $user_id,$account_id,$payment_method, $credits, $debits, $journal_date );

            $accounting_settings = $this->accountingUtil->getAccountingSettings($business_id);

            $ref_no    = $request->get('ref_no');
            $ref_count = $this->util->setAndGetReferenceCount('journal_entry');
            if (empty($ref_no)) {
                $prefix = ! empty($accounting_settings['journal_entry_prefix']) ?
                $accounting_settings['journal_entry_prefix'] : '';

                //Generate reference number
                $ref_no = $this->util->generateReferenceNumber('journal_entry', $ref_count, $business_id, $prefix);
            }

            $acc_trans_mapping                 = new AccountingAccTransMapping();
            $acc_trans_mapping->business_id    = $business_id;
            $acc_trans_mapping->ref_no         = $ref_no;
            $acc_trans_mapping->note           = $request->get('note');
            $acc_trans_mapping->type           = 'expense';
            $acc_trans_mapping->created_by     = $user_id;
            $acc_trans_mapping->operation_date = $this->util->uf_date($journal_date, true);
            $acc_trans_mapping->save();

            //save details in account trnsactions table

            $fromAccount = [
                "accounting_account_id" => $account_id,
                "amount"                => $credits,
                "type"                  => "credit",
                "created_by"            => $user_id,
                "operation_date"        => $this->util->uf_date($journal_date, true),
                'sub_type'              => "expense",
                "acc_trans_mapping_id"  => $acc_trans_mapping->id,
            ];
            AccountingAccountsTransaction::createTransaction($fromAccount);

            $toAccount = [
                "accounting_account_id" => $payment_method,
                "amount"                => $debits,
                "type"                  => "debit",
                "created_by"            => $user_id,
                "operation_date"        => $this->util->uf_date($journal_date, true),
                'sub_type'              => "expense",
                "acc_trans_mapping_id"  => $acc_trans_mapping->id,
            ];
            AccountingAccountsTransaction::createTransaction($toAccount);

            DB::commit();

            $output = [
                'success' => 1,
                'msg'     => __('lang_v1.added_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => 0,
                'msg'     => __('messages.something_went_wrong'),
            ];
        }

        return redirect()->route('journal-entry.index')->with('status', $output);
    }


    
    // UPDATE EXPENSE 
    public function updateExpense(Request $request)
    {
        // dd($request->all());
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.edit_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $user_id = request()->session()->get('user.id');

            $entry_id       = $request->get('id');
            $credit_account = $request->get('payment_method');        // credit = payment method
            $debit_account  = $request->get('expense_account_for');   // debit = expense account
            $amount         = $request->get('amount');
            $journal_date   = $request->get('journal_date');
            $note           = $request->get('note');
            $ref_no         = $request->get('ref_no');

            // Find existing entry
            $acc_trans_mapping = AccountingAccTransMapping::where('business_id', $business_id)
                ->where('type', 'expense')
                ->findOrFail($entry_id);

            // Update reference number
            $acc_trans_mapping->ref_no         = $ref_no;
            $acc_trans_mapping->note           = $note;
            $acc_trans_mapping->operation_date =  $this->util->uf_date($journal_date, true);
            $acc_trans_mapping->created_by     = $user_id;
            $acc_trans_mapping->save();

            // Delete existing transactions
            AccountingAccountsTransaction::where('acc_trans_mapping_id', $acc_trans_mapping->id)->delete();

            // Re-create credit transaction (payment method)
            $credit_transaction = [
                "accounting_account_id" => $credit_account,
                "amount"                => $amount,
                "type"                  => "credit",
                "created_by"            => $user_id,
                "operation_date"        => $this->util->uf_date($journal_date, true),
                "sub_type"              => "expense",
                "acc_trans_mapping_id"  => $acc_trans_mapping->id,
            ];
            AccountingAccountsTransaction::createTransaction($credit_transaction);

            // Re-create debit transaction (expense account)
            $debit_transaction = [
                "accounting_account_id" => $debit_account,
                "amount"                => $amount,
                "type"                  => "debit",
                "created_by"            => $user_id,
                "operation_date"        => $this->util->uf_date($journal_date, true),
                "sub_type"              => "expense",
                "acc_trans_mapping_id"  => $acc_trans_mapping->id,
            ];
            AccountingAccountsTransaction::createTransaction($debit_transaction);

            DB::commit();

            $output = [
                'success' => 1,
                'msg'     => __('lang_v1.updated_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:' . $e->getFile() . ' Line:' . $e->getLine() . ' Message:' . $e->getMessage());

            $output = [
                'success' => 0,
                'msg'     => __('messages.something_went_wrong'),
            ];
        }

        return redirect()->route('journal-entry.index')->with('status', $output);
    }


    /**
     * Show the specified resource.
     *
     * @param  int  $id
     * @return Response
     */
    public function show($id)
    {
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.view_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        return view('accounting::journal_entry.show');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return Response
     */
    public function edit($id)
    {
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.edit_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        $journal = AccountingAccTransMapping::where('business_id', $business_id)
        // ->where('type', 'journal_entry')
            ->where('id', $id)
            ->firstOrFail();
        $accounts_transactions = AccountingAccountsTransaction::with('account')
            ->where('acc_trans_mapping_id', $id)
            ->get()->toArray();
        if ($journal->type == "receipt") {
            $accounts = AccountingAccount::where('business_id', $business_id)->pluck('name', 'id');

            // Fetch the list of payment methods for dropdowns
            $paymentMethods = AccountingAccount::where('business_id', $business_id)->pluck('name', 'id');

            $user_account_transaction = AccountingAccountsTransaction::with('account')
                ->where('acc_trans_mapping_id', $id)
                ->get()->first();
            $payment_method_transaction = AccountingAccountsTransaction::with('account')
                ->where('acc_trans_mapping_id', $id)
                ->get()->skip(1)->first();

            // Extract data
            $user_account   = $user_account_transaction->account ?? null;
            $payment_method = $payment_method_transaction->account ?? null;
            $amount         = $user_account_transaction->amount; // Assuming amount is stored in the journal
            $journal_date   = $journal->operation_date;          // Assuming journal_date is stored in the journal
            $note           = $journal->note;                    // Assuming note is stored in the journal

            // Debugging (optional)
            // dd($accounts_transactions, $user_account, $payment_method, $amount, $journal_date, $note);

            return view('accounting::journal_entry.edit_receipt', compact(
                'journal',
                'accounts_transactions',
                'user_account',
                'accounts',
                'paymentMethods',
                'payment_method',
                'amount',
                'journal_date',
                'note'
            ));

            // dd($accounts_transactions);
        } elseif ($journal->type == "journal_entry") {
            return view('accounting::journal_entry.edit')
                ->with(compact('journal', 'accounts_transactions'));
        } elseif ($journal->type == 'expense') {
            $business_id = request()->session()->get('user.business_id');

            if (
                ! (auth()->user()->can('superadmin') ||
                    $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
                ! (auth()->user()->can('accounting.edit_journal'))
            ) {
                abort(403, 'Unauthorized action.');
            }

            // Find the expense journal entry
            $journal_entry = AccountingAccTransMapping::with('transactions')
                ->where('business_id', $business_id)
                ->where('type', 'expense')
                ->find($id);
                $accounts = AccountingAccount::where('business_id', $business_id)->pluck('name', 'id');
            // dd($journal_entry);

            return view('accounting::journal_entry.edit_expense', compact('journal_entry','accounts'));
        }
      
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  Request  $request
     * @param  int  $id
     * @return Response
     */
    public function update(Request $request, $id)
    {
        $business_id = request()->session()->get('user.business_id');

        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.edit_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        try {
            DB::beginTransaction();

            $user_id = request()->session()->get('user.id');

            $account_ids              = $request->get('account_id');
            $accounts_transactions_id = $request->get('accounts_transactions_id');
            $credits                  = $request->get('credit');
            $debits                   = $request->get('debit');
            $journal_date             = $request->get('journal_date');

            $acc_trans_mapping = AccountingAccTransMapping::where('business_id', $business_id)
                ->where('type', 'journal_entry')
                ->where('id', $id)
                ->firstOrFail();
            $acc_trans_mapping->note           = $request->get('note');
            $acc_trans_mapping->operation_date = $this->util->uf_date($journal_date, true);
            $acc_trans_mapping->update();

            //save details in account trnsactions table
            foreach ($account_ids as $index => $account_id) {
                if (! empty($account_id)) {
                    $transaction_row                          = [];
                    $transaction_row['accounting_account_id'] = $account_id;

                    if (! empty($credits[$index])) {
                        $transaction_row['amount'] = $credits[$index];
                        $transaction_row['type']   = 'credit';
                    }

                    if (! empty($debits[$index])) {
                        $transaction_row['amount'] = $debits[$index];
                        $transaction_row['type']   = 'debit';
                    }

                    $transaction_row['created_by']           = $user_id;
                    $transaction_row['operation_date']       = $this->util->uf_date($journal_date, true);
                    $transaction_row['sub_type']             = 'journal_entry';
                    $transaction_row['acc_trans_mapping_id'] = $acc_trans_mapping->id;

                    if (! empty($accounts_transactions_id[$index])) {
                        $accounts_transactions = AccountingAccountsTransaction::find($accounts_transactions_id[$index]);
                        $accounts_transactions->fill($transaction_row);
                        $accounts_transactions->update();
                    } else {
                        $accounts_transactions = new AccountingAccountsTransaction();
                        $accounts_transactions->fill($transaction_row);
                        $accounts_transactions->save();
                    }
                } elseif (! empty($accounts_transactions_id[$index])) {
                    AccountingAccountsTransaction::delete($accounts_transactions_id[$index]);
                }
            }

            $output = [
                'success' => 1,
                'msg'     => __('lang_v1.updated_success'),
            ];

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            print_r($e->getMessage());
            exit;
            \Log::emergency('File:' . $e->getFile() . 'Line:' . $e->getLine() . 'Message:' . $e->getMessage());

            $output = [
                'success' => 0,
                'msg'     => __('messages.something_went_wrong'),
            ];
        }

        return redirect()->route('journal-entry.index')->with('status', $output);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return Response
     */
    public function destroy($id)
    {
        $business_id = request()->session()->get('user.business_id');
        if (
            ! (auth()->user()->can('superadmin') ||
                $this->moduleUtil->hasThePermissionInSubscription($business_id, 'accounting_module')) ||
            ! (auth()->user()->can('accounting.delete_journal'))
        ) {
            abort(403, 'Unauthorized action.');
        }

        $user_id = request()->session()->get('user.id');

        $acc_trans_mapping = AccountingAccTransMapping::where('id', $id)
            ->where('business_id', $business_id)->firstOrFail();

        if (! empty($acc_trans_mapping)) {
            $acc_trans_mapping->delete();
            AccountingAccountsTransaction::where('acc_trans_mapping_id', $id)->delete();
        }

        return [
            'success' => 1,
            'msg'     => __('lang_v1.deleted_success'),
        ];
    }
}
