# Fix: "Class 'PDO' not found" Error

## 🔴 Problem
Error on `https://dev.posifyme.com/api/license/validate`: **"Class 'PDO' not found"**

## ✅ Root Cause
The **PDO (PHP Data Objects)** extension is not installed or enabled on the server. PDO is required for Laravel to connect to databases.

## 🚀 Solution

### For Server Administrator (dev.posifyme.com)

#### Step 1: Check if PDO is Installed

**SSH into the server:**
```bash
ssh user@dev.posifyme.com
```

**Check PHP version and extensions:**
```bash
php -v
php -m | grep -i pdo
```

**Check for PDO MySQL:**
```bash
php -m | grep -i mysql
```

#### Step 2: Install PDO Extension

**For Ubuntu/Debian:**
```bash
sudo apt-get update
sudo apt-get install php8.2-pdo php8.2-mysql
# Or for your specific PHP version:
sudo apt-get install php-pdo php-mysql
```

**For CentOS/RHEL:**
```bash
sudo yum install php-pdo php-mysql
# Or for newer versions:
sudo dnf install php-pdo php-mysql
```

**For cPanel/WHM:**
1. Login to WHM
2. Go to: **Software** → **Module Installers** → **PHP Pecl**
3. Install: `pdo` and `pdo_mysql`

#### Step 3: Enable PDO in php.ini

**Find php.ini location:**
```bash
php --ini
```

**Edit php.ini:**
```bash
sudo nano /etc/php/8.2/apache2/php.ini
# Or
sudo nano /etc/php/8.2/fpm/php.ini
```

**Uncomment these lines (remove the semicolon):**
```ini
extension=pdo
extension=pdo_mysql
```

**Or add if they don't exist:**
```ini
extension=pdo.so
extension=pdo_mysql.so
```

#### Step 4: Restart Web Server

**For Apache:**
```bash
sudo systemctl restart apache2
# Or
sudo service apache2 restart
```

**For Nginx + PHP-FPM:**
```bash
sudo systemctl restart php8.2-fpm
sudo systemctl restart nginx
# Or
sudo service php-fpm restart
sudo service nginx restart
```

#### Step 5: Verify Installation

**Test PDO:**
```bash
php -r "var_dump(extension_loaded('pdo'));"
php -r "var_dump(extension_loaded('pdo_mysql'));"
```

**Both should return:** `bool(true)`

**Or create a test file:**
```php
<?php
// test_pdo.php
if (extension_loaded('pdo')) {
    echo "PDO: ✓ Installed\n";
} else {
    echo "PDO: ✗ Not installed\n";
}

if (extension_loaded('pdo_mysql')) {
    echo "PDO MySQL: ✓ Installed\n";
} else {
    echo "PDO MySQL: ✗ Not installed\n";
}

try {
    $pdo = new PDO('mysql:host=localhost', 'root', '');
    echo "PDO Connection: ✓ Working\n";
} catch (Exception $e) {
    echo "PDO Connection: ✗ Error - " . $e->getMessage() . "\n";
}
```

**Run:**
```bash
php test_pdo.php
```

## 🔧 Alternative: Quick Fix for Shared Hosting

If you don't have server access, contact your hosting provider and ask them to:
1. Enable PDO extension
2. Enable PDO MySQL extension
3. Restart PHP-FPM/Apache

## ✅ Verification

After fixing, test the API endpoint:

```bash
# Test GET
curl https://dev.posifyme.com/api/license/validate

# Test POST
curl -X POST https://dev.posifyme.com/api/license/validate \
  -H "Content-Type: application/x-www-form-urlencoded" \
  -d "license_code=TEST" \
  -d "username=test" \
  -d "product_id=25"
```

**Should return JSON response, not "Class 'PDO' not found" error.**

## 📝 Notes

- **PDO is required** for Laravel database connections
- **PDO MySQL** is required for MySQL/MariaDB databases
- **Must restart web server** after enabling extensions
- **Check both CLI and web PHP** - they may use different php.ini files

## 🎯 Expected Result

After fixing:
- ✅ API endpoint returns JSON responses
- ✅ No "Class 'PDO' not found" error
- ✅ Database connections work
- ✅ Module installation works

**This is a server configuration issue, not a code issue. The code is correct, but the server needs PDO extension enabled.**

