# Deploy License API Endpoint to dev.posifyme.com

## 🎯 Goal
Deploy the license validation API endpoint to `https://dev.posifyme.com/api/license/validate` so it works without errors.

## 📋 Files to Deploy

### 1. **API Controller** (Main File)
**File:** `Modules/BusinessManagement/Http/Controllers/Api/LicenseValidationController.php`

**Location on server:** 
```
/var/www/html/Modules/BusinessManagement/Http/Controllers/Api/LicenseValidationController.php
```

**What it does:**
- Handles POST requests to `/api/license/validate`
- Validates licenses from `module_licenses` table
- Auto-creates table if missing
- Returns JSON responses

### 2. **API Routes**
**File:** `routes/api.php` (Main application routes)

**Location on server:**
```
/var/www/html/routes/api.php
```

**What to add:**
```php
// Public License Validation API (No authentication required)
Route::post('/license/validate', [\Modules\BusinessManagement\Http\Controllers\Api\LicenseValidationController::class, 'validate'])
    ->name('api.license.validate')
    ->middleware('api'); // Only throttle, no auth required

// API endpoint info (GET request for testing/documentation)
Route::get('/license/validate', [\Modules\BusinessManagement\Http\Controllers\Api\LicenseValidationController::class, 'info'])
    ->name('api.license.info')
    ->middleware('api');
```

### 3. **Module Config**
**File:** `Modules/BusinessManagement/Config/config.php`

**Location on server:**
```
/var/www/html/Modules/BusinessManagement/Config/config.php
```

**Current config (already correct):**
```php
'lic1' => 'aHR0cHM6Ly9kZXYucG9zaWZ5bWUuY29tL2FwaS9saWNlbnNlL3ZhbGlkYXRl', // https://dev.posifyme.com/api/license/validate
```

## 🚀 Deployment Steps

### Step 1: Upload Files to Server

```bash
# On your local machine, upload these files:
scp Modules/BusinessManagement/Http/Controllers/Api/LicenseValidationController.php user@dev.posifyme.com:/var/www/html/Modules/BusinessManagement/Http/Controllers/Api/

# Or use FTP/SFTP to upload:
# - LicenseValidationController.php
# - routes/api.php (update with new routes)
```

### Step 2: Update routes/api.php on Server

**On server, edit:** `/var/www/html/routes/api.php`

**Add these routes:**
```php
<?php

use Illuminate\Http\Request;
use Modules\BusinessManagement\Http\Controllers\Api\LicenseValidationController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
*/

// Public License Validation API (No authentication required)
Route::post('/license/validate', [LicenseValidationController::class, 'validate'])
    ->name('api.license.validate')
    ->middleware('api');

// API endpoint info (GET request for testing/documentation)
Route::get('/license/validate', [LicenseValidationController::class, 'info'])
    ->name('api.license.info')
    ->middleware('api');

// Other existing routes...
```

### Step 3: Clear Caches on Server

```bash
# SSH into server
ssh user@dev.posifyme.com

# Navigate to project directory
cd /var/www/html

# Clear all caches
php artisan route:clear
php artisan config:clear
php artisan cache:clear
php artisan view:clear
```

### Step 4: Run Migration (if not done)

```bash
# On server
php artisan migrate

# This will create the module_licenses table
```

### Step 5: Test the Endpoint

```bash
# Test GET (should return API info)
curl https://dev.posifyme.com/api/license/validate

# Test POST (should validate license)
curl -X POST https://dev.posifyme.com/api/license/validate \
  -H "Content-Type: application/x-www-form-urlencoded" \
  -d "license_code=TEST-123" \
  -d "username=testuser" \
  -d "product_id=25"
```

## ✅ Expected Results

### GET Request:
```json
{
    "endpoint": "/api/license/validate",
    "method": "POST",
    "description": "Public license validation API...",
    ...
}
```

### POST Request (Valid License):
```json
{
    "flag": "valid",
    "msg": "License is valid",
    "data": {...}
}
```

### POST Request (Invalid License):
```json
{
    "flag": "invalid",
    "msg": "License not found. Please generate a license via License Management or contact support."
}
```

## 🔧 Troubleshooting

### If HTTP 500 persists:

1. **Check Laravel Logs:**
   ```bash
   tail -f /var/www/html/storage/logs/laravel.log | grep LicenseValidation
   ```

2. **Check Database Connection:**
   ```bash
   php artisan tinker
   >>> DB::connection()->getPdo();
   ```

3. **Check Table Exists:**
   ```bash
   php artisan tinker
   >>> Schema::hasTable('module_licenses');
   ```

4. **Check PHP Errors:**
   ```bash
   tail -f /var/log/php_errors.log
   # or
   tail -f /var/www/html/storage/logs/laravel.log
   ```

5. **Test Endpoint Directly:**
   ```bash
   # Upload test_api_endpoint.php to server
   php test_api_endpoint.php
   ```

## 📝 Important Notes

1. **The endpoint is PUBLIC** - No authentication required
2. **Auto-creates table** - If `module_licenses` table doesn't exist, it will be created automatically
3. **Returns HTTP 200** - All errors return HTTP 200 with `flag: "invalid"` to prevent generic "HTTP 500" errors
4. **Better error messages** - Shows actual error messages instead of generic errors

## 🎯 Success Criteria

- ✅ GET request returns API documentation
- ✅ POST request with valid license returns `flag: "valid"`
- ✅ POST request with invalid license returns `flag: "invalid"` with helpful message
- ✅ No HTTP 500 errors
- ✅ Module installation works

**Once deployed, the module installation should work without HTTP 500 errors!**

