<?php

namespace Modules\BusinessManagement\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Transaction;
use App\BusinessLocation;
use App\Contact;
use App\PurchaseLine;
use App\Category;
use App\Product;
use App\Brands;
use App\TransactionPayment;
use App\TransactionSellLine;
use App\TransactionSellLinesPurchaseLines;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;
use App\Utils\Util;
use DB;

class PurchaseRegisterController extends Controller
{
    /**
     * Display a listing of the purchase register.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id);

        if ($request->ajax()) {
            $purchases = Transaction::where('business_id', $business_id)
                ->whereIn('type', ['purchase', 'opening_stock'])
                ->with(['contact', 'location'])
                ->select('transactions.*');

            // Apply location filter based on user permissions
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $purchases->whereIn('location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                // Validate that user has access to the requested location
                if ($permitted_locations != 'all' && !in_array($request->location_id, $permitted_locations)) {
                    abort(403, 'Unauthorized location access.');
                }
                $purchases->where('location_id', $request->location_id);
            }

            if (!empty($request->start_date) && !empty($request->end_date)) {
                $purchases->whereBetween('transaction_date', [$request->start_date, $request->end_date]);
            }

            // Permission check for view own purchases only
            if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
                $purchases->where('transactions.created_by', request()->session()->get('user.id'));
            }

            $util = new Util();
            
            return DataTables::of($purchases)
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row->transaction_date, true);
                })
                ->addColumn('contact_name', function ($row) {
                    return $row->contact->name ?? '';
                })
                ->addColumn('location_name', function ($row) {
                    return $row->location->name ?? '';
                })
                ->editColumn('final_total', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->final_total . '</span>';
                })
                ->addColumn('action', function ($row) {
                    $html = '<div class="btn-group">
                        <button type="button" class="btn btn-info btn-xs view-purchase" data-href="' . action([\App\Http\Controllers\PurchaseController::class, 'show'], [$row->id]) . '">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>';
                    return $html;
                })
                ->rawColumns(['action', 'final_total'])
                ->make(true);
        }

        return view('businessmanagement::purchase_register.index', compact('locations'));
    }

    /**
     * Display Date Wise Purchase Report
     *
     * @return \Illuminate\Http\Response
     */
    public function dateWisePurchase(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $suppliers = Contact::suppliersDropdown($business_id, true);
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            $purchases = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')
                ->leftJoin('users', 'transactions.created_by', '=', 'users.id')
                ->where('transactions.business_id', $business_id)
                ->whereIn('transactions.type', ['purchase', 'opening_stock'])
                ->select(
                    'transactions.id',
                    'transactions.ref_no',
                    'transactions.invoice_no',
                    'transactions.transaction_date',
                    'transactions.discount_amount',
                    'transactions.discount_type',
                    'transactions.total_before_tax',
                    'transactions.tax_amount',
                    'transactions.final_total',
                    'transactions.status',
                    'transactions.payment_status',
                    'transactions.shipping_charges',
                    'transactions.additional_expense_value_1',
                    'transactions.additional_expense_value_2',
                    'transactions.additional_expense_value_3',
                    'transactions.additional_expense_value_4',
                    'contacts.name as supplier_name',
                    'contacts.supplier_business_name',
                    'contacts.contact_id',
                    DB::raw("CONCAT(COALESCE(users.surname, ''), ' ', COALESCE(users.first_name, ''), ' ', COALESCE(users.last_name, '')) as entered_by")
                );

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $purchases->whereIn('transactions.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                $purchases->where('transactions.location_id', $request->location_id);
            }

            // Apply supplier filter
            if (!empty($request->supplier_id)) {
                $purchases->where('contacts.id', $request->supplier_id);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $purchases->whereDate('transactions.transaction_date', '>=', $request->start_date)
                    ->whereDate('transactions.transaction_date', '<=', $request->end_date);
            }

            // Permission check
            if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
                $purchases->where('transactions.created_by', request()->session()->get('user.id'));
            }

            $util = new Util();

            return DataTables::of($purchases)
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row->transaction_date, true);
                })
                ->addColumn('supplier', function ($row) {
                    $supplier = $row->supplier_name;
                    if (!empty($row->supplier_business_name)) {
                        $supplier .= ' (' . $row->supplier_business_name . ')';
                    }
                    return $supplier;
                })
                ->editColumn('ref_no', function ($row) {
                    if (!empty($row->ref_no)) {
                        return $row->ref_no;
                    }
                    return $row->invoice_no ?? '';
                })
                ->addColumn('bill_type', function ($row) {
                    if ($row->payment_status == 'paid') {
                        return '<span class="label label-success">Cash</span>';
                    } elseif ($row->payment_status == 'due') {
                        return '<span class="label label-warning">Credit</span>';
                    } elseif ($row->payment_status == 'partial') {
                        return '<span class="label label-info">Partial</span>';
                    }
                    return '<span class="label label-default">' . ucfirst($row->payment_status) . '</span>';
                })
                ->editColumn('discount_amount', function ($row) {
                    $discount = $row->discount_amount ?? 0;
                    if (!empty($discount) && $row->discount_type == 'percentage') {
                        $discount = $row->total_before_tax * ($discount / 100);
                    }
                    return '<span class="display_currency" data-currency_symbol="true">' . $discount . '</span>';
                })
                ->editColumn('tax_amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . ($row->tax_amount ?? 0) . '</span>';
                })
                ->addColumn('extra_charges', function ($row) {
                    $extra_charges = $row->shipping_charges ?? 0;
                    $extra_charges += $row->additional_expense_value_1 ?? 0;
                    $extra_charges += $row->additional_expense_value_2 ?? 0;
                    $extra_charges += $row->additional_expense_value_3 ?? 0;
                    $extra_charges += $row->additional_expense_value_4 ?? 0;
                    return '<span class="display_currency" data-currency_symbol="true">' . $extra_charges . '</span>';
                })
                ->editColumn('final_total', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->final_total . '</span>';
                })
                ->addColumn('link', function ($row) {
                    return '<a href="' . action([\App\Http\Controllers\PurchaseController::class, 'show'], [$row->id]) . '" target="_blank" class="btn btn-xs btn-info"><i class="fas fa-eye"></i></a>';
                })
                ->rawColumns(['bill_type', 'discount_amount', 'tax_amount', 'extra_charges', 'final_total', 'link'])
                ->make(true);
        }

        return view('businessmanagement::purchase_register.date_wise_purchase', compact('suppliers', 'locations'));
    }

    /**
     * Print Date Wise Purchase Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printDateWisePurchase(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        $location = null;
        
        if (!empty($request->location_id)) {
            $location = BusinessLocation::find($request->location_id);
        }
        
        // Get invoice layout logo
        $logo = false;
        if ($location) {
            $businessUtil = new \App\Utils\BusinessUtil();
            $invoice_layout = $businessUtil->invoiceLayout($business_id, $location->invoice_layout_id);
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        } else {
            // Get default invoice layout
            $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        }

        $purchases = Transaction::leftJoin('contacts', 'transactions.contact_id', '=', 'contacts.id')
            ->leftJoin('users', 'transactions.created_by', '=', 'users.id')
            ->where('transactions.business_id', $business_id)
            ->whereIn('transactions.type', ['purchase', 'opening_stock'])
            ->select(
                'transactions.id',
                'transactions.ref_no',
                'transactions.invoice_no',
                'transactions.transaction_date',
                'transactions.discount_amount',
                'transactions.discount_type',
                'transactions.total_before_tax',
                'transactions.tax_amount',
                'transactions.final_total',
                'transactions.status',
                'transactions.payment_status',
                'transactions.shipping_charges',
                'transactions.additional_expense_value_1',
                'transactions.additional_expense_value_2',
                'transactions.additional_expense_value_3',
                'transactions.additional_expense_value_4',
                'contacts.name as supplier_name',
                'contacts.supplier_business_name',
                'contacts.contact_id',
                DB::raw("CONCAT(COALESCE(users.surname, ''), ' ', COALESCE(users.first_name, ''), ' ', COALESCE(users.last_name, '')) as entered_by")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $purchases->whereIn('transactions.location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            $purchases->where('transactions.location_id', $request->location_id);
        }

        // Apply supplier filter
        if (!empty($request->supplier_id)) {
            $purchases->where('contacts.id', $request->supplier_id);
        }

        // Apply date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
            $purchases->whereDate('transactions.transaction_date', '>=', $start_date)
                ->whereDate('transactions.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
            $purchases->where('transactions.created_by', request()->session()->get('user.id'));
        }

        // Get supplier name for filter
        $supplier_name = __('lang_v1.all_suppliers');
        $is_supplier_wise = false;
        if (!empty($request->supplier_id)) {
            $supplier = Contact::find($request->supplier_id);
            if ($supplier) {
                $supplier_name = $supplier->name;
                if (!empty($supplier->supplier_business_name)) {
                    $supplier_name .= ' (' . $supplier->supplier_business_name . ')';
                }
                $is_supplier_wise = true;
            }
        }

        $purchases = $purchases->orderBy('transactions.transaction_date', 'asc')
            ->orderBy('transactions.invoice_no', 'asc')
            ->get();
        
        // Group purchases by date only if not supplier-wise
        $purchases_by_date = [];
        if (!$is_supplier_wise) {
            foreach ($purchases as $purchase) {
                $date_key = date('Y-m-d', strtotime($purchase->transaction_date));
                if (!isset($purchases_by_date[$date_key])) {
                    $purchases_by_date[$date_key] = [];
                }
                $purchases_by_date[$date_key][] = $purchase;
            }
        }

        $util = new Util();
        $currency = session('currency');

        // Calculate totals
        $total_discount = 0;
        $total_extra_charges = 0;
        $total_tax = 0;
        $total_amount = 0;

        foreach ($purchases as $purchase) {
            $discount = $purchase->discount_amount ?? 0;
            if (!empty($discount) && $purchase->discount_type == 'percentage') {
                $discount = $purchase->total_before_tax * ($discount / 100);
            }
            $total_discount += $discount;

            $extra_charges = $purchase->shipping_charges ?? 0;
            $extra_charges += $purchase->additional_expense_value_1 ?? 0;
            $extra_charges += $purchase->additional_expense_value_2 ?? 0;
            $extra_charges += $purchase->additional_expense_value_3 ?? 0;
            $extra_charges += $purchase->additional_expense_value_4 ?? 0;
            $total_extra_charges += $extra_charges;

            $total_tax += $purchase->tax_amount ?? 0;
            $total_amount += $purchase->final_total;
        }

        return view('businessmanagement::purchase_register.print_date_wise_purchase', compact(
            'purchases',
            'purchases_by_date',
            'business',
            'location',
            'logo',
            'start_date',
            'end_date',
            'supplier_name',
            'is_supplier_wise',
            'util',
            'currency',
            'total_discount',
            'total_extra_charges',
            'total_tax',
            'total_amount'
        ));
    }

    /**
     * Display Supplier Wise Purchase Report
     *
     * @return \Illuminate\Http\Response
     */
    public function supplierWisePurchase(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $suppliers = Contact::suppliersDropdown($business_id, true);
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            // Get purchase lines with product details
            $query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->where('t.business_id', $business_id)
                ->whereIn('t.type', ['purchase', 'opening_stock'])
                ->select(
                    't.id as transaction_id',
                    't.invoice_no',
                    't.ref_no',
                    't.transaction_date',
                    'c.name as supplier_name',
                    'c.supplier_business_name',
                    'purchase_lines.id as purchase_line_id',
                    'purchase_lines.quantity',
                    'purchase_lines.purchase_price_inc_tax',
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_description"),
                    DB::raw("(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as line_total")
                );

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                $query->where('t.location_id', $request->location_id);
            }

            // Apply supplier filter
            if (!empty($request->supplier_id)) {
                $query->where('c.id', $request->supplier_id);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
            }

            // Permission check
            if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
                $query->where('t.created_by', request()->session()->get('user.id'));
            }

            $util = new Util();
            
            return DataTables::of($query)
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row->transaction_date, true);
                })
                ->addColumn('supplier_name', function ($row) {
                    $supplier = $row->supplier_name ?? '';
                    if (!empty($row->supplier_business_name)) {
                        $supplier .= ' (' . $row->supplier_business_name . ')';
                    }
                    return $supplier;
                })
                ->editColumn('ref_no', function ($row) {
                    return $row->ref_no ?? '';
                })
                ->editColumn('quantity', function ($row) use ($util) {
                    return $util->num_f($row->quantity);
                })
                ->addColumn('unit_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->purchase_price_inc_tax . '</span>';
                })
                ->editColumn('line_total', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->line_total . '</span>';
                })
                ->rawColumns(['unit_price', 'line_total'])
                ->make(true);
        }

        return view('businessmanagement::purchase_register.supplier_wise_purchase', compact('suppliers', 'locations'));
    }

    /**
     * Print Supplier Wise Purchase Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printSupplierWisePurchase(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        $location = null;
        
        if (!empty($request->location_id)) {
            $location = BusinessLocation::find($request->location_id);
        }
        
        // Get invoice layout logo
        $logo = false;
        if ($location) {
            $businessUtil = new \App\Utils\BusinessUtil();
            $invoice_layout = $businessUtil->invoiceLayout($business_id, $location->invoice_layout_id);
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        } else {
            // Get default invoice layout
            $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        }

        // Get purchase lines with product details
        $query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->select(
                't.id as transaction_id',
                't.invoice_no',
                't.ref_no',
                't.transaction_date',
                'c.id as supplier_id',
                'c.name as supplier_name',
                'c.supplier_business_name',
                'purchase_lines.id as purchase_line_id',
                'purchase_lines.quantity',
                'purchase_lines.purchase_price',
                'purchase_lines.purchase_price_inc_tax',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw("(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as line_total")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            $query->where('t.location_id', $request->location_id);
        }

        // Apply supplier filter
        if (!empty($request->supplier_id)) {
            $query->where('c.id', $request->supplier_id);
        }

        // Apply date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $purchase_lines = $query->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        // Group by supplier and date
        $suppliers_data = [];
        foreach ($purchase_lines as $line) {
            $supplier_id = $line->supplier_id;
            if (!isset($suppliers_data[$supplier_id])) {
                $suppliers_data[$supplier_id] = [
                    'supplier_name' => $line->supplier_name,
                    'supplier_business_name' => $line->supplier_business_name,
                    'lines' => [],
                    'total_qty' => 0,
                    'total_price' => 0
                ];
            }
            
            // Group by date within supplier
            $date_key = date('Y-m-d', strtotime($line->transaction_date));
            if (!isset($suppliers_data[$supplier_id]['lines'][$date_key])) {
                $suppliers_data[$supplier_id]['lines'][$date_key] = [];
            }
            
            $suppliers_data[$supplier_id]['lines'][$date_key][] = $line;
            $suppliers_data[$supplier_id]['total_qty'] += $line->quantity;
            $suppliers_data[$supplier_id]['total_price'] += $line->line_total;
        }
        
        // Sort suppliers by name
        ksort($suppliers_data);

        // Get supplier name for single supplier
        $supplier_name = null;
        $is_single_supplier = false;
        if (!empty($request->supplier_id) && count($suppliers_data) == 1) {
            $is_single_supplier = true;
            $supplier = Contact::find($request->supplier_id);
            if ($supplier) {
                $supplier_name = $supplier->name;
                if (!empty($supplier->supplier_business_name)) {
                    $supplier_name .= ' (' . $supplier->supplier_business_name . ')';
                }
            }
        }

        // Calculate grand totals
        $grand_total_qty = 0;
        $grand_total_price = 0;
        foreach ($suppliers_data as $supplier_data) {
            $grand_total_qty += $supplier_data['total_qty'];
            $grand_total_price += $supplier_data['total_price'];
        }

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::purchase_register.print_supplier_wise_purchase', compact(
            'suppliers_data',
            'business',
            'location',
            'logo',
            'start_date',
            'end_date',
            'supplier_name',
            'is_single_supplier',
            'grand_total_qty',
            'grand_total_price',
            'util',
            'currency'
        ));
    }

    /**
     * Display Supplier Wise Purchase Without Price Report
     *
     * @return \Illuminate\Http\Response
     */
    public function supplierWisePurchaseWOP(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $suppliers = Contact::suppliersDropdown($business_id, true);
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            // Get purchase lines with product details
            $query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
                ->where('t.business_id', $business_id)
                ->whereIn('t.type', ['purchase', 'opening_stock'])
                ->select(
                    't.id as transaction_id',
                    DB::raw('COALESCE(t.invoice_no, t.ref_no, "") as invoice_no'),
                    't.ref_no',
                    't.transaction_date',
                    'c.name as supplier_name',
                    'c.supplier_business_name',
                    'purchase_lines.id as purchase_line_id',
                    'purchase_lines.quantity',
                    'u.short_name as unit_name',
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_description")
                );

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                $query->where('t.location_id', $request->location_id);
            }

            // Apply supplier filter
            if (!empty($request->supplier_id)) {
                $query->where('c.id', $request->supplier_id);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
            }

            // Permission check
            if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
                $query->where('t.created_by', request()->session()->get('user.id'));
            }

            $util = new Util();
            
            return DataTables::of($query)
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row->transaction_date, true);
                })
                ->addColumn('supplier_name', function ($row) {
                    $supplier = $row->supplier_name ?? '';
                    if (!empty($row->supplier_business_name)) {
                        $supplier .= ' (' . $row->supplier_business_name . ')';
                    }
                    return $supplier;
                })
                ->editColumn('ref_no', function ($row) {
                    return !empty($row->invoice_no) ? $row->invoice_no : ($row->ref_no ?? '');
                })
                ->editColumn('quantity', function ($row) use ($util) {
                    return $util->num_f($row->quantity);
                })
                ->make(true);
        }

        return view('businessmanagement::purchase_register.supplier_wise_purchase_wop', compact('suppliers', 'locations'));
    }

    /**
     * Print Supplier Wise Purchase Without Price Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printSupplierWisePurchaseWOP(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        $location = null;
        
        if (!empty($request->location_id)) {
            $location = BusinessLocation::find($request->location_id);
        }
        
        // Get invoice layout logo
        $logo = false;
        if ($location) {
            $businessUtil = new \App\Utils\BusinessUtil();
            $invoice_layout = $businessUtil->invoiceLayout($business_id, $location->invoice_layout_id);
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        } else {
            // Get default invoice layout
            $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        }

        // Get purchase lines with product details
        $query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->select(
                't.id as transaction_id',
                DB::raw('COALESCE(t.invoice_no, t.ref_no, "") as invoice_no'),
                't.ref_no',
                't.transaction_date',
                'c.id as supplier_id',
                'c.name as supplier_name',
                'c.supplier_business_name',
                'purchase_lines.id as purchase_line_id',
                'purchase_lines.quantity',
                'u.short_name as unit_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            $query->where('t.location_id', $request->location_id);
        }

        // Apply supplier filter
        if (!empty($request->supplier_id)) {
            $query->where('c.id', $request->supplier_id);
        }

        // Apply date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $purchase_lines = $query->orderBy('c.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        // Group by supplier and date
        $suppliers_data = [];
        foreach ($purchase_lines as $line) {
            $supplier_id = $line->supplier_id;
            if (!isset($suppliers_data[$supplier_id])) {
                $suppliers_data[$supplier_id] = [
                    'supplier_name' => $line->supplier_name,
                    'supplier_business_name' => $line->supplier_business_name,
                    'lines' => [],
                    'total_qty' => 0
                ];
            }
            
            // Group by date within supplier
            $date_key = date('Y-m-d', strtotime($line->transaction_date));
            if (!isset($suppliers_data[$supplier_id]['lines'][$date_key])) {
                $suppliers_data[$supplier_id]['lines'][$date_key] = [];
            }
            
            $suppliers_data[$supplier_id]['lines'][$date_key][] = $line;
            $suppliers_data[$supplier_id]['total_qty'] += $line->quantity;
        }
        
        // Sort suppliers by name
        ksort($suppliers_data);

        // Get supplier name for single supplier
        $supplier_name = null;
        $is_single_supplier = false;
        if (!empty($request->supplier_id) && count($suppliers_data) == 1) {
            $is_single_supplier = true;
            $supplier = Contact::find($request->supplier_id);
            if ($supplier) {
                $supplier_name = $supplier->name;
                if (!empty($supplier->supplier_business_name)) {
                    $supplier_name .= ' (' . $supplier->supplier_business_name . ')';
                }
            }
        }

        // Calculate grand totals
        $grand_total_qty = 0;
        foreach ($suppliers_data as $supplier_data) {
            $grand_total_qty += $supplier_data['total_qty'];
        }

        $util = new Util();

        return view('businessmanagement::purchase_register.print_supplier_wise_purchase_wop', compact(
            'suppliers_data',
            'business',
            'location',
            'logo',
            'start_date',
            'end_date',
            'supplier_name',
            'is_single_supplier',
            'grand_total_qty',
            'util'
        ));
    }

    /**
     * Display Product Wise Purchase Report
     *
     * @return \Illuminate\Http\Response
     */
    public function productWisePurchase(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            // Get purchase lines with product details
            $query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
                ->where('t.business_id', $business_id)
                ->whereIn('t.type', ['purchase', 'opening_stock'])
                ->select(
                    't.id as transaction_id',
                    't.invoice_no',
                    't.ref_no',
                    't.transaction_date',
                    'c.name as supplier_name',
                    'c.supplier_business_name',
                    'p.id as product_id',
                    'p.category_id',
                    'cat.name as category_name',
                    'purchase_lines.id as purchase_line_id',
                    'purchase_lines.quantity',
                    'purchase_lines.purchase_price_inc_tax',
                    'u.short_name as unit_name',
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_description"),
                    DB::raw("(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as line_total")
                );

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                $query->where('t.location_id', $request->location_id);
            }

            // Apply category filter
            if (!empty($request->category_id)) {
                $query->where('p.category_id', $request->category_id);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
            }

            // Apply amount filter
            if (!empty($request->min_amount) || !empty($request->max_amount)) {
                $min_amount = !empty($request->min_amount) ? (float)$request->min_amount : 0;
                $max_amount = !empty($request->max_amount) ? (float)$request->max_amount : PHP_INT_MAX;
                $query->whereRaw('(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) >= ?', [$min_amount]);
                if (!empty($request->max_amount)) {
                    $query->whereRaw('(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) <= ?', [$max_amount]);
                }
            }

            // Permission check
            if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
                $query->where('t.created_by', request()->session()->get('user.id'));
            }

            $util = new Util();
            
            return DataTables::of($query)
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row->transaction_date, true);
                })
                ->addColumn('supplier_name', function ($row) {
                    $supplier = $row->supplier_name ?? '';
                    if (!empty($row->supplier_business_name)) {
                        $supplier .= ' (' . $row->supplier_business_name . ')';
                    }
                    return $supplier;
                })
                ->editColumn('category_name', function ($row) {
                    return $row->category_name ?? 'UNCATEGORIZED';
                })
                ->editColumn('ref_no', function ($row) {
                    return $row->ref_no ?? '';
                })
                ->editColumn('quantity', function ($row) use ($util) {
                    return $util->num_f($row->quantity);
                })
                ->addColumn('unit_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->purchase_price_inc_tax . '</span>';
                })
                ->editColumn('line_total', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->line_total . '</span>';
                })
                ->rawColumns(['unit_price', 'line_total'])
                ->make(true);
        }

        return view('businessmanagement::purchase_register.product_wise_purchase', compact('categories', 'locations'));
    }

    /**
     * Print Product Wise Purchase Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printProductWisePurchase(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        $location = null;
        
        if (!empty($request->location_id)) {
            $location = BusinessLocation::find($request->location_id);
        }
        
        // Get invoice layout logo
        $logo = false;
        if ($location) {
            $businessUtil = new \App\Utils\BusinessUtil();
            $invoice_layout = $businessUtil->invoiceLayout($business_id, $location->invoice_layout_id);
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        } else {
            // Get default invoice layout
            $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        }

        // Get purchase lines with product details
        $query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->select(
                't.id as transaction_id',
                DB::raw('COALESCE(t.invoice_no, t.ref_no, "") as invoice_no'),
                't.ref_no',
                't.transaction_date',
                'c.name as supplier_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'purchase_lines.id as purchase_line_id',
                'purchase_lines.quantity',
                'purchase_lines.purchase_price_inc_tax',
                'u.short_name as unit_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw("(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as line_total")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            $query->where('t.location_id', $request->location_id);
        }

        // Apply category filter
        if (!empty($request->category_id)) {
            $query->where('p.category_id', $request->category_id);
        }

        // Apply date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $purchase_lines = $query->orderBy('cat.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        // Apply amount filter after getting results (since we need to filter by line_total)
        if (!empty($request->min_amount) || !empty($request->max_amount)) {
            $min_amount = !empty($request->min_amount) ? (float)$request->min_amount : 0;
            $max_amount = !empty($request->max_amount) ? (float)$request->max_amount : PHP_INT_MAX;
            
            $purchase_lines = $purchase_lines->filter(function($line) use ($min_amount, $max_amount) {
                $line_total = (float)$line->line_total;
                return $line_total >= $min_amount && $line_total <= $max_amount;
            })->values();
        }

        // Group by category, then by product
        $categories_data = [];
        foreach ($purchase_lines as $line) {
            $category_id = $line->category_id ?? 0;
            $category_name = $line->category_name ?? 'UNCATEGORIZED';
            
            if (!isset($categories_data[$category_id])) {
                $categories_data[$category_id] = [
                    'category_name' => $category_name,
                    'products' => [],
                    'total_qty' => 0,
                    'total_price' => 0
                ];
            }
            
            // Group by product
            $product_key = $line->product_id . '_' . $line->product_description;
            if (!isset($categories_data[$category_id]['products'][$product_key])) {
                $categories_data[$category_id]['products'][$product_key] = [
                    'product_description' => $line->product_description,
                    'unit_name' => $line->unit_name ?? '',
                    'lines' => [],
                    'total_qty' => 0,
                    'total_price' => 0
                ];
            }
            
            $categories_data[$category_id]['products'][$product_key]['lines'][] = $line;
            $categories_data[$category_id]['products'][$product_key]['total_qty'] += $line->quantity;
            $categories_data[$category_id]['products'][$product_key]['total_price'] += $line->line_total;
            
            $categories_data[$category_id]['total_qty'] += $line->quantity;
            $categories_data[$category_id]['total_price'] += $line->line_total;
        }
        
        // Sort categories by name
        uasort($categories_data, function($a, $b) {
            return strcmp($a['category_name'], $b['category_name']);
        });

        // Calculate grand totals
        $grand_total_qty = 0;
        $grand_total_price = 0;
        foreach ($categories_data as $category_data) {
            $grand_total_qty += $category_data['total_qty'];
            $grand_total_price += $category_data['total_price'];
        }

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::purchase_register.print_product_wise_purchase', compact(
            'categories_data',
            'business',
            'location',
            'logo',
            'start_date',
            'end_date',
            'grand_total_qty',
            'grand_total_price',
            'util',
            'currency'
        ));
    }

    /**
     * Display Product Wise Purchase Without Price Report
     *
     * @return \Illuminate\Http\Response
     */
    public function productWisePurchaseWOP(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $locations = BusinessLocation::forDropdown($business_id, true);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->mapWithKeys(function ($product) {
                $name = $product->name;
                if (!empty($product->sku)) {
                    $name .= ' (' . $product->sku . ')';
                }
                return [$product->id => $name];
            });

        if ($request->ajax()) {
            // Get purchase lines with product details
            $query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
                ->where('t.business_id', $business_id)
                ->whereIn('t.type', ['purchase', 'opening_stock'])
                ->select(
                    't.id as transaction_id',
                    DB::raw('COALESCE(t.invoice_no, t.ref_no, "") as invoice_no'),
                    't.ref_no',
                    't.transaction_date',
                    'c.name as supplier_name',
                    'c.supplier_business_name',
                    'p.id as product_id',
                    'p.category_id',
                    'cat.name as category_name',
                    'purchase_lines.id as purchase_line_id',
                    'purchase_lines.quantity',
                    'u.short_name as unit_name',
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_description")
                );

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                $query->where('t.location_id', $request->location_id);
            }

            // Apply category filter
            if (!empty($request->category_id)) {
                $query->where('p.category_id', $request->category_id);
            }

            // Apply product filter
            if (!empty($request->product_id)) {
                $query->where('p.id', $request->product_id);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
            }

            // Apply amount filter
            if (!empty($request->min_amount) || !empty($request->max_amount)) {
                $min_amount = !empty($request->min_amount) ? (float)$request->min_amount : 0;
                $max_amount = !empty($request->max_amount) ? (float)$request->max_amount : PHP_INT_MAX;
                $query->whereRaw('(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) >= ?', [$min_amount]);
                if (!empty($request->max_amount)) {
                    $query->whereRaw('(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) <= ?', [$max_amount]);
                }
            }

            // Permission check
            if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
                $query->where('t.created_by', request()->session()->get('user.id'));
            }

            $util = new Util();
            
            return DataTables::of($query)
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row->transaction_date, true);
                })
                ->addColumn('supplier_name', function ($row) {
                    $supplier = $row->supplier_name ?? '';
                    if (!empty($row->supplier_business_name)) {
                        $supplier .= ' (' . $row->supplier_business_name . ')';
                    }
                    return $supplier;
                })
                ->editColumn('category_name', function ($row) {
                    return $row->category_name ?? 'UNCATEGORIZED';
                })
                ->editColumn('quantity', function ($row) use ($util) {
                    return $util->num_f($row->quantity);
                })
                ->make(true);
        }

        return view('businessmanagement::purchase_register.product_wise_purchase_wop', compact('categories', 'locations', 'products'));
    }

    /**
     * Print Product Wise Purchase Without Price Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printProductWisePurchaseWOP(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        $location = null;
        
        if (!empty($request->location_id)) {
            $location = BusinessLocation::find($request->location_id);
        }
        
        // Get invoice layout logo
        $logo = false;
        if ($location) {
            $businessUtil = new \App\Utils\BusinessUtil();
            $invoice_layout = $businessUtil->invoiceLayout($business_id, $location->invoice_layout_id);
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        } else {
            // Get default invoice layout
            $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        }

        // Get purchase lines with product details
        $query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->select(
                't.id as transaction_id',
                DB::raw('COALESCE(t.invoice_no, t.ref_no, "") as invoice_no'),
                't.ref_no',
                't.transaction_date',
                'c.name as supplier_name',
                'c.supplier_business_name',
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'purchase_lines.id as purchase_line_id',
                'purchase_lines.quantity',
                'purchase_lines.purchase_price_inc_tax',
                'u.short_name as unit_name',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description"),
                DB::raw("(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as line_total")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            $query->where('t.location_id', $request->location_id);
        }

        // Apply category filter
        if (!empty($request->category_id)) {
            $query->where('p.category_id', $request->category_id);
        }

        // Apply product filter
        if (!empty($request->product_id)) {
            $query->where('p.id', $request->product_id);
        }

        // Apply date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $purchase_lines = $query->orderBy('cat.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        // Apply amount filter after getting results
        if (!empty($request->min_amount) || !empty($request->max_amount)) {
            $min_amount = !empty($request->min_amount) ? (float)$request->min_amount : 0;
            $max_amount = !empty($request->max_amount) ? (float)$request->max_amount : PHP_INT_MAX;
            
            $purchase_lines = $purchase_lines->filter(function($line) use ($min_amount, $max_amount) {
                $line_total = (float)$line->line_total;
                return $line_total >= $min_amount && $line_total <= $max_amount;
            })->values();
        }

        // Group by category, then by product
        $categories_data = [];
        foreach ($purchase_lines as $line) {
            $category_id = $line->category_id ?? 0;
            $category_name = $line->category_name ?? 'UNCATEGORIZED';
            
            if (!isset($categories_data[$category_id])) {
                $categories_data[$category_id] = [
                    'category_name' => $category_name,
                    'products' => [],
                    'total_qty' => 0
                ];
            }
            
            // Group by product
            $product_key = $line->product_id . '_' . $line->product_description;
            if (!isset($categories_data[$category_id]['products'][$product_key])) {
                $categories_data[$category_id]['products'][$product_key] = [
                    'product_description' => $line->product_description,
                    'unit_name' => $line->unit_name ?? '',
                    'lines' => [],
                    'total_qty' => 0
                ];
            }
            
            $categories_data[$category_id]['products'][$product_key]['lines'][] = $line;
            $categories_data[$category_id]['products'][$product_key]['total_qty'] += $line->quantity;
            
            $categories_data[$category_id]['total_qty'] += $line->quantity;
        }
        
        // Sort categories by name
        uasort($categories_data, function($a, $b) {
            return strcmp($a['category_name'], $b['category_name']);
        });

        // Calculate grand totals
        $grand_total_qty = 0;
        foreach ($categories_data as $category_data) {
            $grand_total_qty += $category_data['total_qty'];
        }

        $util = new Util();

        return view('businessmanagement::purchase_register.print_product_wise_purchase_wop', compact(
            'categories_data',
            'business',
            'location',
            'logo',
            'start_date',
            'end_date',
            'grand_total_qty',
            'util'
        ));
    }

    /**
     * Display Location Wise Net Product Purchase Report
     *
     * @return \Illuminate\Http\Response
     */
    public function locationWiseNetProductPurchase(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $locations = BusinessLocation::forDropdown($business_id, true);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->mapWithKeys(function ($product) {
                $name = $product->name;
                if (!empty($product->sku)) {
                    $name .= ' (' . $product->sku . ')';
                }
                return [$product->id => $name];
            });

        if ($request->ajax()) {
            // Get purchase lines
            $purchase_query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
                ->leftJoin('business_locations as bl', 't.location_id', '=', 'bl.id')
                ->where('t.business_id', $business_id)
                ->whereIn('t.type', ['purchase', 'opening_stock'])
                ->select(
                    't.location_id',
                    'bl.name as location_name',
                    'p.id as product_id',
                    'p.category_id',
                    'cat.name as category_name',
                    'p.brand_id',
                    'b.name as brand_name',
                    'purchase_lines.quantity as purchase_qty',
                    'purchase_lines.purchase_price_inc_tax as unit_price',
                    DB::raw('(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as purchase_amount'),
                    DB::raw('COALESCE(purchase_lines.quantity_returned, 0) as return_qty'),
                    DB::raw('COALESCE(purchase_lines.purchase_price_inc_tax, 0) as return_price'),
                    DB::raw('(COALESCE(purchase_lines.quantity_returned, 0) * COALESCE(purchase_lines.purchase_price_inc_tax, 0)) as return_amount'),
                    DB::raw('(purchase_lines.quantity - COALESCE(purchase_lines.quantity_returned, 0)) as net_qty'),
                    DB::raw('((purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) - (COALESCE(purchase_lines.quantity_returned, 0) * COALESCE(purchase_lines.purchase_price_inc_tax, 0))) as net_amount'),
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_description")
                );

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $purchase_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                $purchase_query->where('t.location_id', $request->location_id);
            }

            // Apply category filter
            if (!empty($request->category_id)) {
                $purchase_query->where('p.category_id', $request->category_id);
            }

            // Apply product filter
            if (!empty($request->product_id)) {
                $purchase_query->where('p.id', $request->product_id);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $purchase_query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
            }

            // Permission check
            if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
                $purchase_query->where('t.created_by', request()->session()->get('user.id'));
            }

            $purchase_lines = $purchase_query->get();

            // Group and aggregate by location, category, brand, product
            $grouped_data = [];
            foreach ($purchase_lines as $line) {
                $location_id = $line->location_id ?? 0;
                $location_name = $line->location_name ?? 'UNKNOWN';
                $category_id = $line->category_id ?? 0;
                $category_name = $line->category_name ?? 'UNCATEGORIZED';
                $brand_id = $line->brand_id ?? 0;
                $brand_name = $line->brand_name ?? '';
                $product_key = $line->product_id . '_' . $line->product_description;

                if (!isset($grouped_data[$location_id])) {
                    $grouped_data[$location_id] = [
                        'location_name' => $location_name,
                        'categories' => []
                    ];
                }

                if (!isset($grouped_data[$location_id]['categories'][$category_id])) {
                    $grouped_data[$location_id]['categories'][$category_id] = [
                        'category_name' => $category_name,
                        'brands' => []
                    ];
                }

                if (!isset($grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id])) {
                    $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id] = [
                        'brand_name' => $brand_name,
                        'products' => []
                    ];
                }

                if (!isset($grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key])) {
                    $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key] = [
                        'product_description' => $line->product_description,
                        'purchase_qty' => 0,
                        'unit_price' => 0,
                        'purchase_amount' => 0,
                        'return_qty' => 0,
                        'return_price' => 0,
                        'return_amount' => 0,
                        'net_qty' => 0,
                        'net_amount' => 0
                    ];
                }

                $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_qty'] += $line->purchase_qty;
                $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_amount'] += $line->purchase_amount;
                $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_qty'] += $line->return_qty;
                $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_amount'] += $line->return_amount;
                $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['net_qty'] += $line->net_qty;
                $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['net_amount'] += $line->net_amount;
                
                // Calculate average unit price
                if ($grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_qty'] > 0) {
                    $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['unit_price'] = 
                        $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_amount'] / 
                        $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_qty'];
                }
                
                // Calculate return price
                if ($grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_qty'] > 0) {
                    $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_price'] = 
                        $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_amount'] / 
                        $grouped_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_qty'];
                }
            }

            // Flatten for DataTable
            $table_data = [];
            foreach ($grouped_data as $location_id => $location_data) {
                foreach ($location_data['categories'] as $category_id => $category_data) {
                    foreach ($category_data['brands'] as $brand_id => $brand_data) {
                        foreach ($brand_data['products'] as $product_key => $product_data) {
                            $table_data[] = [
                                'location_name' => $location_data['location_name'],
                                'category_name' => $category_data['category_name'],
                                'product_description' => $product_data['product_description'],
                                'purchase_qty' => $product_data['purchase_qty'],
                                'unit_price' => $product_data['unit_price'],
                                'purchase_amount' => $product_data['purchase_amount'],
                                'return_qty' => $product_data['return_qty'],
                                'return_price' => $product_data['return_price'],
                                'return_amount' => $product_data['return_amount'],
                                'net_qty' => $product_data['net_qty'],
                                'net_amount' => $product_data['net_amount'],
                            ];
                        }
                    }
                }
            }

            $util = new Util();
            
            return DataTables::of(collect($table_data))
                ->editColumn('purchase_qty', function ($row) use ($util) {
                    return $util->num_f($row['purchase_qty']);
                })
                ->addColumn('unit_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['unit_price'] . '</span>';
                })
                ->addColumn('purchase_amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['purchase_amount'] . '</span>';
                })
                ->editColumn('return_qty', function ($row) use ($util) {
                    return $util->num_f($row['return_qty']);
                })
                ->addColumn('return_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['return_price'] . '</span>';
                })
                ->addColumn('return_amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['return_amount'] . '</span>';
                })
                ->editColumn('net_qty', function ($row) use ($util) {
                    return $util->num_f($row['net_qty']);
                })
                ->addColumn('net_amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['net_amount'] . '</span>';
                })
                ->rawColumns(['unit_price', 'purchase_amount', 'return_price', 'return_amount', 'net_amount'])
                ->make(true);
        }

        return view('businessmanagement::purchase_register.location_wise_net_product_purchase', compact('categories', 'locations', 'products'));
    }

    /**
     * Print Location Wise Net Product Purchase Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printLocationWiseNetProductPurchase(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        $location = null;
        $location_name = null;
        $is_single_location = false;
        
        if (!empty($request->location_id)) {
            $location = BusinessLocation::find($request->location_id);
            if ($location) {
                $location_name = $location->name;
                $is_single_location = true;
            }
        }
        
        // Get invoice layout logo
        $logo = false;
        if ($location) {
            $businessUtil = new \App\Utils\BusinessUtil();
            $invoice_layout = $businessUtil->invoiceLayout($business_id, $location->invoice_layout_id);
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        } else {
            // Get default invoice layout
            $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
                ->where('is_default', 1)
                ->first();
            if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
                $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
            }
        }

        // Get purchase lines
        $purchase_query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->leftJoin('business_locations as bl', 't.location_id', '=', 'bl.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->select(
                't.location_id',
                'bl.name as location_name',
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'p.brand_id',
                'b.name as brand_name',
                'purchase_lines.quantity as purchase_qty',
                'purchase_lines.purchase_price_inc_tax as unit_price',
                DB::raw('(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as purchase_amount'),
                DB::raw('COALESCE(purchase_lines.quantity_returned, 0) as return_qty'),
                DB::raw('COALESCE(purchase_lines.purchase_price_inc_tax, 0) as return_price'),
                DB::raw('(COALESCE(purchase_lines.quantity_returned, 0) * COALESCE(purchase_lines.purchase_price_inc_tax, 0)) as return_amount'),
                DB::raw('(purchase_lines.quantity - COALESCE(purchase_lines.quantity_returned, 0)) as net_qty'),
                DB::raw('((purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) - (COALESCE(purchase_lines.quantity_returned, 0) * COALESCE(purchase_lines.purchase_price_inc_tax, 0))) as net_amount'),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $purchase_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            $purchase_query->where('t.location_id', $request->location_id);
        }

        // Apply category filter
        if (!empty($request->category_id)) {
            $purchase_query->where('p.category_id', $request->category_id);
        }

        // Apply product filter
        if (!empty($request->product_id)) {
            $purchase_query->where('p.id', $request->product_id);
        }

        // Apply date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
            $purchase_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
            $purchase_query->where('t.created_by', request()->session()->get('user.id'));
        }

        $purchase_lines = $purchase_query->orderBy('bl.name', 'asc')
            ->orderBy('cat.name', 'asc')
            ->orderBy('b.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Group by location, category, brand, product
        $locations_data = [];
        foreach ($purchase_lines as $line) {
            $location_id = $line->location_id ?? 0;
            $location_name = $line->location_name ?? 'UNKNOWN';
            $category_id = $line->category_id ?? 0;
            $category_name = $line->category_name ?? 'UNCATEGORIZED';
            $brand_id = $line->brand_id ?? 0;
            $brand_name = $line->brand_name ?? '';
            $product_key = $line->product_id . '_' . $line->product_description;

            if (!isset($locations_data[$location_id])) {
                $locations_data[$location_id] = [
                    'location_name' => $location_name,
                    'categories' => []
                ];
            }

            if (!isset($locations_data[$location_id]['categories'][$category_id])) {
                $locations_data[$location_id]['categories'][$category_id] = [
                    'category_name' => $category_name,
                    'brands' => [],
                    'total_purchase_qty' => 0,
                    'total_purchase_amount' => 0,
                    'total_return_qty' => 0,
                    'total_return_amount' => 0,
                    'total_net_qty' => 0,
                    'total_net_amount' => 0
                ];
            }

            if (!isset($locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id])) {
                $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id] = [
                    'brand_name' => $brand_name,
                    'products' => [],
                    'total_purchase_qty' => 0,
                    'total_purchase_amount' => 0,
                    'total_return_qty' => 0,
                    'total_return_amount' => 0,
                    'total_net_qty' => 0,
                    'total_net_amount' => 0
                ];
            }

            if (!isset($locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key])) {
                $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key] = [
                    'product_description' => $line->product_description,
                    'purchase_qty' => 0,
                    'unit_price' => 0,
                    'purchase_amount' => 0,
                    'return_qty' => 0,
                    'return_price' => 0,
                    'return_amount' => 0,
                    'net_qty' => 0,
                    'net_amount' => 0
                ];
            }

            $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_qty'] += $line->purchase_qty;
            $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_amount'] += $line->purchase_amount;
            $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_qty'] += $line->return_qty;
            $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_amount'] += $line->return_amount;
            $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['net_qty'] += $line->net_qty;
            $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['net_amount'] += $line->net_amount;

            // Calculate average unit price
            if ($locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_qty'] > 0) {
                $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['unit_price'] = 
                    $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_amount'] / 
                    $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_qty'];
            }

            // Calculate return price
            if ($locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_qty'] > 0) {
                $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_price'] = 
                    $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_amount'] / 
                    $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_qty'];
            }

            // Update brand totals
            $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['total_purchase_qty'] += $line->purchase_qty;
            $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['total_purchase_amount'] += $line->purchase_amount;
            $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['total_return_qty'] += $line->return_qty;
            $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['total_return_amount'] += $line->return_amount;
            $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['total_net_qty'] += $line->net_qty;
            $locations_data[$location_id]['categories'][$category_id]['brands'][$brand_id]['total_net_amount'] += $line->net_amount;

            // Update category totals
            $locations_data[$location_id]['categories'][$category_id]['total_purchase_qty'] += $line->purchase_qty;
            $locations_data[$location_id]['categories'][$category_id]['total_purchase_amount'] += $line->purchase_amount;
            $locations_data[$location_id]['categories'][$category_id]['total_return_qty'] += $line->return_qty;
            $locations_data[$location_id]['categories'][$category_id]['total_return_amount'] += $line->return_amount;
            $locations_data[$location_id]['categories'][$category_id]['total_net_qty'] += $line->net_qty;
            $locations_data[$location_id]['categories'][$category_id]['total_net_amount'] += $line->net_amount;
        }

        // Sort locations by name
        uasort($locations_data, function($a, $b) {
            return strcmp($a['location_name'], $b['location_name']);
        });

        // Calculate grand totals
        $grand_total_purchase_qty = 0;
        $grand_total_purchase_amount = 0;
        $grand_total_return_qty = 0;
        $grand_total_return_amount = 0;
        $grand_total_net_qty = 0;
        $grand_total_net_amount = 0;

        foreach ($locations_data as $location_data) {
            foreach ($location_data['categories'] as $category_data) {
                $grand_total_purchase_qty += $category_data['total_purchase_qty'];
                $grand_total_purchase_amount += $category_data['total_purchase_amount'];
                $grand_total_return_qty += $category_data['total_return_qty'];
                $grand_total_return_amount += $category_data['total_return_amount'];
                $grand_total_net_qty += $category_data['total_net_qty'];
                $grand_total_net_amount += $category_data['total_net_amount'];
            }
        }

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::purchase_register.print_location_wise_net_product_purchase', compact(
            'locations_data',
            'business',
            'location',
            'logo',
            'start_date',
            'end_date',
            'location_name',
            'is_single_location',
            'grand_total_purchase_qty',
            'grand_total_purchase_amount',
            'grand_total_return_qty',
            'grand_total_return_amount',
            'grand_total_net_qty',
            'grand_total_net_amount',
            'util',
            'currency'
        ));
    }

    /**
     * Display Date Wise Net Product Purchase Report
     *
     * @return \Illuminate\Http\Response
     */
    public function dateWiseNetProductPurchase(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $suppliers = Contact::suppliersDropdown($business_id, false);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->mapWithKeys(function ($product) {
                $name = $product->name;
                if (!empty($product->sku)) {
                    $name .= ' (' . $product->sku . ')';
                }
                return [$product->id => $name];
            });

        if ($request->ajax()) {
            // Get purchase lines
            $purchase_query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->whereIn('t.type', ['purchase', 'opening_stock'])
                ->select(
                    DB::raw('DATE(t.transaction_date) as transaction_date'),
                    'p.id as product_id',
                    'p.category_id',
                    'cat.name as category_name',
                    'p.brand_id',
                    'b.name as brand_name',
                    'purchase_lines.quantity as purchase_qty',
                    'purchase_lines.purchase_price_inc_tax as unit_price',
                    DB::raw('(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as purchase_amount'),
                    DB::raw('COALESCE(purchase_lines.quantity_returned, 0) as return_qty'),
                    DB::raw('COALESCE(purchase_lines.purchase_price_inc_tax, 0) as return_price'),
                    DB::raw('(COALESCE(purchase_lines.quantity_returned, 0) * COALESCE(purchase_lines.purchase_price_inc_tax, 0)) as return_amount'),
                    DB::raw('(purchase_lines.quantity - COALESCE(purchase_lines.quantity_returned, 0)) as net_qty'),
                    DB::raw('((purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) - (COALESCE(purchase_lines.quantity_returned, 0) * COALESCE(purchase_lines.purchase_price_inc_tax, 0))) as net_amount'),
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_description")
                );

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $purchase_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                $purchase_query->where('t.location_id', $request->location_id);
            }

            // Apply category filter
            if (!empty($request->category_id)) {
                $purchase_query->where('p.category_id', $request->category_id);
            }

            // Apply product filter
            if (!empty($request->product_id)) {
                $purchase_query->where('p.id', $request->product_id);
            }

            // Apply supplier filter
            if (!empty($request->supplier_id)) {
                $purchase_query->where('t.contact_id', $request->supplier_id);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $purchase_query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
            }

            // Permission check
            if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
                $purchase_query->where('t.created_by', request()->session()->get('user.id'));
            }

            $purchase_lines = $purchase_query->orderBy('t.transaction_date', 'asc')
                ->orderBy('cat.name', 'asc')
                ->orderBy('b.name', 'asc')
                ->orderBy('p.name', 'asc')
                ->get();

            // Group by date, category, brand, product
            $table_data = [];
            foreach ($purchase_lines as $line) {
                $date_key = $line->transaction_date;
                $category_id = $line->category_id ?? 0;
                $category_name = $line->category_name ?? 'UNCATEGORIZED';
                $brand_id = $line->brand_id ?? 0;
                $brand_name = $line->brand_name ?? '';
                $product_key = $line->product_id . '_' . $line->product_description;

                $key = $date_key . '_' . $category_id . '_' . $brand_id . '_' . $product_key;

                if (!isset($table_data[$key])) {
                    $table_data[$key] = [
                        'transaction_date' => $date_key,
                        'category_name' => $category_name,
                        'brand_name' => $brand_name,
                        'product_description' => $line->product_description,
                        'purchase_qty' => 0,
                        'unit_price' => 0,
                        'purchase_amount' => 0,
                        'return_qty' => 0,
                        'return_price' => 0,
                        'return_amount' => 0,
                        'net_qty' => 0,
                        'net_amount' => 0
                    ];
                }

                $table_data[$key]['purchase_qty'] += $line->purchase_qty;
                $table_data[$key]['purchase_amount'] += $line->purchase_amount;
                $table_data[$key]['return_qty'] += $line->return_qty;
                $table_data[$key]['return_amount'] += $line->return_amount;
                $table_data[$key]['net_qty'] += $line->net_qty;
                $table_data[$key]['net_amount'] += $line->net_amount;

                // Calculate average unit price
                if ($table_data[$key]['purchase_qty'] > 0) {
                    $table_data[$key]['unit_price'] = $table_data[$key]['purchase_amount'] / $table_data[$key]['purchase_qty'];
                }

                // Calculate return price
                if ($table_data[$key]['return_qty'] > 0) {
                    $table_data[$key]['return_price'] = $table_data[$key]['return_amount'] / $table_data[$key]['return_qty'];
                }
            }

            $util = new Util();
            
            return DataTables::of(collect($table_data))
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row['transaction_date'], true);
                })
                ->editColumn('purchase_qty', function ($row) use ($util) {
                    return $util->num_f($row['purchase_qty']);
                })
                ->addColumn('unit_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['unit_price'] . '</span>';
                })
                ->addColumn('purchase_amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['purchase_amount'] . '</span>';
                })
                ->editColumn('return_qty', function ($row) use ($util) {
                    return $util->num_f($row['return_qty']);
                })
                ->addColumn('return_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['return_price'] . '</span>';
                })
                ->addColumn('return_amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['return_amount'] . '</span>';
                })
                ->editColumn('net_qty', function ($row) use ($util) {
                    return $util->num_f($row['net_qty']);
                })
                ->addColumn('net_amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['net_amount'] . '</span>';
                })
                ->rawColumns(['unit_price', 'purchase_amount', 'return_price', 'return_amount', 'net_amount'])
                ->make(true);
        }

        return view('businessmanagement::purchase_register.date_wise_net_product_purchase', compact('categories', 'locations', 'products', 'suppliers'));
    }

    /**
     * Print Date Wise Net Product Purchase Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printDateWiseNetProductPurchase(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get invoice layout logo
        $logo = false;
        $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
            ->where('is_default', 1)
            ->first();
        if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
            $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
        }

        // Get purchase lines
        $purchase_query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->select(
                DB::raw('DATE(t.transaction_date) as transaction_date'),
                'p.id as product_id',
                'p.category_id',
                'cat.name as category_name',
                'p.brand_id',
                'b.name as brand_name',
                'purchase_lines.quantity as purchase_qty',
                'purchase_lines.purchase_price_inc_tax as unit_price',
                DB::raw('(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as purchase_amount'),
                DB::raw('COALESCE(purchase_lines.quantity_returned, 0) as return_qty'),
                DB::raw('COALESCE(purchase_lines.purchase_price_inc_tax, 0) as return_price'),
                DB::raw('(COALESCE(purchase_lines.quantity_returned, 0) * COALESCE(purchase_lines.purchase_price_inc_tax, 0)) as return_amount'),
                DB::raw('(purchase_lines.quantity - COALESCE(purchase_lines.quantity_returned, 0)) as net_qty'),
                DB::raw('((purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) - (COALESCE(purchase_lines.quantity_returned, 0) * COALESCE(purchase_lines.purchase_price_inc_tax, 0))) as net_amount'),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $purchase_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($request->location_id)) {
            $purchase_query->where('t.location_id', $request->location_id);
        }

        // Apply category filter
        if (!empty($request->category_id)) {
            $purchase_query->where('p.category_id', $request->category_id);
        }

        // Apply product filter
        if (!empty($request->product_id)) {
            $purchase_query->where('p.id', $request->product_id);
        }

        // Apply supplier filter
        if (!empty($request->supplier_id)) {
            $purchase_query->where('t.contact_id', $request->supplier_id);
        }

        // Apply date filter
        $start_date = null;
        $end_date = null;
        if (!empty($request->start_date) && !empty($request->end_date)) {
            $start_date = $request->start_date;
            $end_date = $request->end_date;
            $purchase_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
            $purchase_query->where('t.created_by', request()->session()->get('user.id'));
        }

        $purchase_lines = $purchase_query->orderBy('t.transaction_date', 'asc')
            ->orderBy('cat.name', 'asc')
            ->orderBy('b.name', 'asc')
            ->orderBy('p.name', 'asc')
            ->get();

        // Group by date, category, brand, product
        $dates_data = [];
        foreach ($purchase_lines as $line) {
            $date_key = $line->transaction_date;
            $category_id = $line->category_id ?? 0;
            $category_name = $line->category_name ?? 'UNCATEGORIZED';
            $brand_id = $line->brand_id ?? 0;
            $brand_name = $line->brand_name ?? '';
            $product_key = $line->product_id . '_' . $line->product_description;

            if (!isset($dates_data[$date_key])) {
                $dates_data[$date_key] = [
                    'categories' => []
                ];
            }

            if (!isset($dates_data[$date_key]['categories'][$category_id])) {
                $dates_data[$date_key]['categories'][$category_id] = [
                    'category_name' => $category_name,
                    'brands' => [],
                    'total_purchase_qty' => 0,
                    'total_purchase_amount' => 0,
                    'total_return_qty' => 0,
                    'total_return_amount' => 0,
                    'total_net_qty' => 0,
                    'total_net_amount' => 0
                ];
            }

            if (!isset($dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id])) {
                $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id] = [
                    'brand_name' => $brand_name,
                    'products' => [],
                    'total_purchase_qty' => 0,
                    'total_purchase_amount' => 0,
                    'total_return_qty' => 0,
                    'total_return_amount' => 0,
                    'total_net_qty' => 0,
                    'total_net_amount' => 0
                ];
            }

            if (!isset($dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key])) {
                $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key] = [
                    'product_description' => $line->product_description,
                    'purchase_qty' => 0,
                    'unit_price' => 0,
                    'purchase_amount' => 0,
                    'return_qty' => 0,
                    'return_price' => 0,
                    'return_amount' => 0,
                    'net_qty' => 0,
                    'net_amount' => 0
                ];
            }

            $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_qty'] += $line->purchase_qty;
            $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_amount'] += $line->purchase_amount;
            $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_qty'] += $line->return_qty;
            $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_amount'] += $line->return_amount;
            $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['net_qty'] += $line->net_qty;
            $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['net_amount'] += $line->net_amount;

            // Calculate average unit price
            if ($dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_qty'] > 0) {
                $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['unit_price'] = 
                    $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_amount'] / 
                    $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['purchase_qty'];
            }

            // Calculate return price
            if ($dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_qty'] > 0) {
                $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_price'] = 
                    $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_amount'] / 
                    $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['products'][$product_key]['return_qty'];
            }

            // Update brand totals
            $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['total_purchase_qty'] += $line->purchase_qty;
            $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['total_purchase_amount'] += $line->purchase_amount;
            $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['total_return_qty'] += $line->return_qty;
            $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['total_return_amount'] += $line->return_amount;
            $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['total_net_qty'] += $line->net_qty;
            $dates_data[$date_key]['categories'][$category_id]['brands'][$brand_id]['total_net_amount'] += $line->net_amount;

            // Update category totals
            $dates_data[$date_key]['categories'][$category_id]['total_purchase_qty'] += $line->purchase_qty;
            $dates_data[$date_key]['categories'][$category_id]['total_purchase_amount'] += $line->purchase_amount;
            $dates_data[$date_key]['categories'][$category_id]['total_return_qty'] += $line->return_qty;
            $dates_data[$date_key]['categories'][$category_id]['total_return_amount'] += $line->return_amount;
            $dates_data[$date_key]['categories'][$category_id]['total_net_qty'] += $line->net_qty;
            $dates_data[$date_key]['categories'][$category_id]['total_net_amount'] += $line->net_amount;
        }

        // Sort dates
        ksort($dates_data);

        // Calculate grand totals
        $grand_total_purchase_qty = 0;
        $grand_total_purchase_amount = 0;
        $grand_total_return_qty = 0;
        $grand_total_return_amount = 0;
        $grand_total_net_qty = 0;
        $grand_total_net_amount = 0;

        foreach ($dates_data as $date_data) {
            foreach ($date_data['categories'] as $category_data) {
                $grand_total_purchase_qty += $category_data['total_purchase_qty'];
                $grand_total_purchase_amount += $category_data['total_purchase_amount'];
                $grand_total_return_qty += $category_data['total_return_qty'];
                $grand_total_return_amount += $category_data['total_return_amount'];
                $grand_total_net_qty += $category_data['total_net_qty'];
                $grand_total_net_amount += $category_data['total_net_amount'];
            }
        }

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::purchase_register.print_date_wise_net_product_purchase', compact(
            'dates_data',
            'business',
            'logo',
            'start_date',
            'end_date',
            'grand_total_purchase_qty',
            'grand_total_purchase_amount',
            'grand_total_return_qty',
            'grand_total_return_amount',
            'grand_total_net_qty',
            'grand_total_net_amount',
            'util',
            'currency'
        ));
    }

    /**
     * Display Supplier Wise Statement Report
     *
     * @return \Illuminate\Http\Response
     */
    public function supplierWiseStatement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $suppliers = Contact::suppliersDropdown($business_id, false);
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            $start_date = $request->start_date ?? null;
            $end_date = $request->end_date ?? null;
            $location_id = $request->location_id ?? null;
            $supplier_id = $request->supplier_id ?? null;

            // Get all suppliers or specific supplier
            $suppliers_query = Contact::where('business_id', $business_id)
                ->onlySuppliers();
            
            if (!empty($supplier_id)) {
                $suppliers_query->where('id', $supplier_id);
            }

            $suppliers_list = $suppliers_query->get();

            $table_data = [];
            foreach ($suppliers_list as $supplier) {
                // Calculate opening balance (before start_date)
                $opening_balance_query = Transaction::where('business_id', $business_id)
                    ->where('contact_id', $supplier->id)
                    ->whereIn('type', ['purchase', 'purchase_return', 'opening_balance', 'ledger_discount']);

                if (!empty($location_id)) {
                    $opening_balance_query->where('location_id', $location_id);
                }

                $permitted_locations = auth()->user()->permitted_locations();
                if ($permitted_locations != 'all') {
                    $opening_balance_query->whereIn('location_id', $permitted_locations);
                }

                if (!empty($start_date)) {
                    $opening_balance_query->whereDate('transaction_date', '<', $start_date);
                }

                $opening_balance_data = $opening_balance_query->select(
                    DB::raw("SUM(IF(type = 'purchase', final_total, 0)) as total_purchase"),
                    DB::raw("SUM(IF(type = 'purchase_return', final_total, 0)) as total_purchase_return"),
                    DB::raw("SUM(IF(type = 'opening_balance', final_total, 0)) as total_opening_balance"),
                    DB::raw("SUM(IF(type = 'ledger_discount', final_total, 0)) as total_ledger_discount")
                )->first();

                // Get payments before start_date
                $opening_payments = 0;
                if (!empty($start_date)) {
                    $opening_payments_query = TransactionPayment::join('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                        ->where('t.business_id', $business_id)
                        ->where('t.contact_id', $supplier->id)
                        ->whereIn('t.type', ['purchase', 'purchase_return', 'opening_balance'])
                        ->whereDate('transaction_payments.paid_on', '<', $start_date);

                    if (!empty($location_id)) {
                        $opening_payments_query->where('t.location_id', $location_id);
                    }

                    if ($permitted_locations != 'all') {
                        $opening_payments_query->whereIn('t.location_id', $permitted_locations);
                    }

                    $opening_payments = $opening_payments_query->sum(DB::raw('IF(transaction_payments.is_return = 1, -1 * transaction_payments.amount, transaction_payments.amount)')) ?? 0;
                }

                $opening_balance = ($opening_balance_data->total_purchase ?? 0) 
                    - ($opening_balance_data->total_purchase_return ?? 0)
                    + ($opening_balance_data->total_opening_balance ?? 0)
                    - ($opening_balance_data->total_ledger_discount ?? 0)
                    - $opening_payments;

                // Get transactions in date range
                $period_query = Transaction::where('business_id', $business_id)
                    ->where('contact_id', $supplier->id)
                    ->whereIn('type', ['purchase', 'purchase_return', 'ledger_discount']);

                if (!empty($location_id)) {
                    $period_query->where('location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $period_query->whereIn('location_id', $permitted_locations);
                }

                if (!empty($start_date) && !empty($end_date)) {
                    $period_query->whereDate('transaction_date', '>=', $start_date)
                        ->whereDate('transaction_date', '<=', $end_date);
                }

                $period_data = $period_query->select(
                    DB::raw("SUM(IF(type = 'purchase', final_total, 0)) as total_purchase"),
                    DB::raw("SUM(IF(type = 'purchase_return', final_total, 0)) as total_purchase_return"),
                    DB::raw("SUM(IF(type = 'ledger_discount', final_total, 0)) as total_ledger_discount")
                )->first();

                $purchase = $period_data->total_purchase ?? 0;
                $purchase_return = $period_data->total_purchase_return ?? 0;
                $net_purchase = $purchase - $purchase_return;
                $other_dr = $period_data->total_ledger_discount ?? 0;
                $other_cr = 0; // Other credits (can be extended later)

                // Get payments in date range
                $payment = 0;
                if (!empty($start_date) && !empty($end_date)) {
                    $payment_query = TransactionPayment::join('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                        ->where('t.business_id', $business_id)
                        ->where('t.contact_id', $supplier->id)
                        ->whereIn('t.type', ['purchase', 'purchase_return'])
                        ->whereDate('transaction_payments.paid_on', '>=', $start_date)
                        ->whereDate('transaction_payments.paid_on', '<=', $end_date);

                    if (!empty($location_id)) {
                        $payment_query->where('t.location_id', $location_id);
                    }

                    if ($permitted_locations != 'all') {
                        $payment_query->whereIn('t.location_id', $permitted_locations);
                    }

                    $payment = $payment_query->sum(DB::raw('IF(transaction_payments.is_return = 1, -1 * transaction_payments.amount, transaction_payments.amount)')) ?? 0;
                }

                // Calculate closing balance
                $closing_balance = $opening_balance + $net_purchase - $payment + $other_cr - $other_dr;

                $supplier_name = $supplier->name;
                if (!empty($supplier->supplier_business_name)) {
                    $supplier_name .= ' (' . $supplier->supplier_business_name . ')';
                }

                $table_data[] = [
                    'supplier_name' => $supplier_name,
                    'opening_balance' => $opening_balance,
                    'purchase' => $purchase,
                    'purchase_return' => $purchase_return,
                    'net_purchase' => $net_purchase,
                    'payment' => $payment,
                    'other_cr' => $other_cr,
                    'other_dr' => $other_dr,
                    'closing_balance' => $closing_balance,
                ];
            }

            $util = new Util();
            
            return DataTables::of(collect($table_data))
                ->addColumn('opening_balance', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['opening_balance'] . '</span>';
                })
                ->addColumn('purchase', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['purchase'] . '</span>';
                })
                ->addColumn('purchase_return', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['purchase_return'] . '</span>';
                })
                ->addColumn('net_purchase', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['net_purchase'] . '</span>';
                })
                ->addColumn('payment', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['payment'] . '</span>';
                })
                ->addColumn('other_cr', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['other_cr'] . '</span>';
                })
                ->addColumn('other_dr', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['other_dr'] . '</span>';
                })
                ->addColumn('closing_balance', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['closing_balance'] . '</span>';
                })
                ->rawColumns(['opening_balance', 'purchase', 'purchase_return', 'net_purchase', 'payment', 'other_cr', 'other_dr', 'closing_balance'])
                ->make(true);
        }

        return view('businessmanagement::purchase_register.supplier_wise_statement', compact('suppliers', 'locations'));
    }

    /**
     * Print Supplier Wise Statement Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printSupplierWiseStatement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get invoice layout logo
        $logo = false;
        $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
            ->where('is_default', 1)
            ->first();
        if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
            $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
        }

        $start_date = $request->start_date ?? null;
        $end_date = $request->end_date ?? null;
        $location_id = $request->location_id ?? null;
        $supplier_id = $request->supplier_id ?? null;

        // Get all suppliers or specific supplier
        $suppliers_query = Contact::where('business_id', $business_id)
            ->onlySuppliers();
        
        if (!empty($supplier_id)) {
            $suppliers_query->where('id', $supplier_id);
        }

        $suppliers_list = $suppliers_query->get();

        $suppliers_data = [];
        $grand_total_opening_balance = 0;
        $grand_total_purchase = 0;
        $grand_total_purchase_return = 0;
        $grand_total_net_purchase = 0;
        $grand_total_payment = 0;
        $grand_total_other_cr = 0;
        $grand_total_other_dr = 0;
        $grand_total_closing_balance = 0;

        foreach ($suppliers_list as $supplier) {
            // Calculate opening balance (before start_date)
            $opening_balance_query = Transaction::where('business_id', $business_id)
                ->where('contact_id', $supplier->id)
                ->whereIn('type', ['purchase', 'purchase_return', 'opening_balance', 'ledger_discount']);

            if (!empty($location_id)) {
                $opening_balance_query->where('location_id', $location_id);
            }

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $opening_balance_query->whereIn('location_id', $permitted_locations);
            }

            if (!empty($start_date)) {
                $opening_balance_query->whereDate('transaction_date', '<', $start_date);
            }

            $opening_balance_data = $opening_balance_query->select(
                DB::raw("SUM(IF(type = 'purchase', final_total, 0)) as total_purchase"),
                DB::raw("SUM(IF(type = 'purchase_return', final_total, 0)) as total_purchase_return"),
                DB::raw("SUM(IF(type = 'opening_balance', final_total, 0)) as total_opening_balance"),
                DB::raw("SUM(IF(type = 'ledger_discount', final_total, 0)) as total_ledger_discount")
            )->first();

            // Get payments before start_date
            $opening_payments = 0;
            if (!empty($start_date)) {
                $opening_payments_query = TransactionPayment::join('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $supplier->id)
                    ->whereIn('t.type', ['purchase', 'purchase_return', 'opening_balance'])
                    ->whereDate('transaction_payments.paid_on', '<', $start_date);

                if (!empty($location_id)) {
                    $opening_payments_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $opening_payments_query->whereIn('t.location_id', $permitted_locations);
                }

                $opening_payments = $opening_payments_query->sum(DB::raw('IF(transaction_payments.is_return = 1, -1 * transaction_payments.amount, transaction_payments.amount)')) ?? 0;
            }

            $opening_balance = ($opening_balance_data->total_purchase ?? 0) 
                - ($opening_balance_data->total_purchase_return ?? 0)
                + ($opening_balance_data->total_opening_balance ?? 0)
                - ($opening_balance_data->total_ledger_discount ?? 0)
                - $opening_payments;

            // Get transactions in date range
            $period_query = Transaction::where('business_id', $business_id)
                ->where('contact_id', $supplier->id)
                ->whereIn('type', ['purchase', 'purchase_return', 'ledger_discount']);

            if (!empty($location_id)) {
                $period_query->where('location_id', $location_id);
            }

            if ($permitted_locations != 'all') {
                $period_query->whereIn('location_id', $permitted_locations);
            }

            if (!empty($start_date) && !empty($end_date)) {
                $period_query->whereDate('transaction_date', '>=', $start_date)
                    ->whereDate('transaction_date', '<=', $end_date);
            }

            $period_data = $period_query->select(
                DB::raw("SUM(IF(type = 'purchase', final_total, 0)) as total_purchase"),
                DB::raw("SUM(IF(type = 'purchase_return', final_total, 0)) as total_purchase_return"),
                DB::raw("SUM(IF(type = 'ledger_discount', final_total, 0)) as total_ledger_discount")
            )->first();

            $purchase = $period_data->total_purchase ?? 0;
            $purchase_return = $period_data->total_purchase_return ?? 0;
            $net_purchase = $purchase - $purchase_return;
            $other_dr = $period_data->total_ledger_discount ?? 0;
            $other_cr = 0; // Other credits (can be extended later)

            // Get payments in date range
            $payment = 0;
            if (!empty($start_date) && !empty($end_date)) {
                $payment_query = TransactionPayment::join('transactions as t', 'transaction_payments.transaction_id', '=', 't.id')
                    ->where('t.business_id', $business_id)
                    ->where('t.contact_id', $supplier->id)
                    ->whereIn('t.type', ['purchase', 'purchase_return'])
                    ->whereDate('transaction_payments.paid_on', '>=', $start_date)
                    ->whereDate('transaction_payments.paid_on', '<=', $end_date);

                if (!empty($location_id)) {
                    $payment_query->where('t.location_id', $location_id);
                }

                if ($permitted_locations != 'all') {
                    $payment_query->whereIn('t.location_id', $permitted_locations);
                }

                $payment = $payment_query->sum(DB::raw('IF(transaction_payments.is_return = 1, -1 * transaction_payments.amount, transaction_payments.amount)')) ?? 0;
            }

            // Calculate closing balance
            $closing_balance = $opening_balance + $net_purchase - $payment + $other_cr - $other_dr;

            $supplier_name = $supplier->name;
            if (!empty($supplier->supplier_business_name)) {
                $supplier_name .= ' (' . $supplier->supplier_business_name . ')';
            }

            // Group by category if needed (for all suppliers report)
            $category_name = 'ALL';
            // You can add category grouping logic here if needed

            if (!isset($suppliers_data[$category_name])) {
                $suppliers_data[$category_name] = [
                    'category_name' => $category_name,
                    'suppliers' => []
                ];
            }

            $suppliers_data[$category_name]['suppliers'][] = [
                'supplier_name' => $supplier_name,
                'opening_balance' => $opening_balance,
                'purchase' => $purchase,
                'purchase_return' => $purchase_return,
                'net_purchase' => $net_purchase,
                'payment' => $payment,
                'other_cr' => $other_cr,
                'other_dr' => $other_dr,
                'closing_balance' => $closing_balance,
            ];

            // Update grand totals
            $grand_total_opening_balance += $opening_balance;
            $grand_total_purchase += $purchase;
            $grand_total_purchase_return += $purchase_return;
            $grand_total_net_purchase += $net_purchase;
            $grand_total_payment += $payment;
            $grand_total_other_cr += $other_cr;
            $grand_total_other_dr += $other_dr;
            $grand_total_closing_balance += $closing_balance;
        }

        // Calculate category subtotals
        foreach ($suppliers_data as $category_name => $category_data) {
            $subtotal_opening_balance = 0;
            $subtotal_purchase = 0;
            $subtotal_purchase_return = 0;
            $subtotal_net_purchase = 0;
            $subtotal_payment = 0;
            $subtotal_other_cr = 0;
            $subtotal_other_dr = 0;
            $subtotal_closing_balance = 0;

            foreach ($category_data['suppliers'] as $supplier_data) {
                $subtotal_opening_balance += $supplier_data['opening_balance'];
                $subtotal_purchase += $supplier_data['purchase'];
                $subtotal_purchase_return += $supplier_data['purchase_return'];
                $subtotal_net_purchase += $supplier_data['net_purchase'];
                $subtotal_payment += $supplier_data['payment'];
                $subtotal_other_cr += $supplier_data['other_cr'];
                $subtotal_other_dr += $supplier_data['other_dr'];
                $subtotal_closing_balance += $supplier_data['closing_balance'];
            }

            $suppliers_data[$category_name]['subtotal'] = [
                'opening_balance' => $subtotal_opening_balance,
                'purchase' => $subtotal_purchase,
                'purchase_return' => $subtotal_purchase_return,
                'net_purchase' => $subtotal_net_purchase,
                'payment' => $subtotal_payment,
                'other_cr' => $subtotal_other_cr,
                'other_dr' => $subtotal_other_dr,
                'closing_balance' => $subtotal_closing_balance,
            ];
        }

        $is_single_supplier = !empty($supplier_id) && count($suppliers_list) == 1;
        $supplier_name = null;
        if ($is_single_supplier && count($suppliers_list) > 0) {
            $supplier_name = $suppliers_list->first()->name;
            if (!empty($suppliers_list->first()->supplier_business_name)) {
                $supplier_name .= ' (' . $suppliers_list->first()->supplier_business_name . ')';
            }
        }

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::purchase_register.print_supplier_wise_statement', compact(
            'suppliers_data',
            'business',
            'logo',
            'start_date',
            'end_date',
            'is_single_supplier',
            'supplier_name',
            'grand_total_opening_balance',
            'grand_total_purchase',
            'grand_total_purchase_return',
            'grand_total_net_purchase',
            'grand_total_payment',
            'grand_total_other_cr',
            'grand_total_other_dr',
            'grand_total_closing_balance',
            'util',
            'currency'
        ));
    }

    /**
     * Display Supplier and Product Wise Purchase Report
     *
     * @return \Illuminate\Http\Response
     */
    public function supplierAndProductWisePurchase(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $suppliers = Contact::suppliersDropdown($business_id, false);
        
        // Get products for dropdown
        $products = Product::where('business_id', $business_id)
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get()
            ->mapWithKeys(function ($product) {
                $name = $product->name;
                if (!empty($product->sku)) {
                    $name .= ' (' . $product->sku . ')';
                }
                return [$product->id => $name];
            });

        if ($request->ajax()) {
            // Get purchase lines
            $query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->whereIn('t.type', ['purchase', 'opening_stock'])
                ->select(
                    't.transaction_date',
                    DB::raw('COALESCE(t.invoice_no, t.ref_no, "") as invoice_no'),
                    'c.id as supplier_id',
                    'c.name as supplier_name',
                    'c.supplier_business_name',
                    'cat.id as category_id',
                    'cat.name as category_name',
                    'b.id as brand_id',
                    'b.name as brand_name',
                    'purchase_lines.quantity',
                    'purchase_lines.purchase_price_inc_tax as unit_price',
                    DB::raw("(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as line_total"),
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_description")
                );

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                $query->where('t.location_id', $request->location_id);
            }

            // Apply category filter
            if (!empty($request->category_id)) {
                $query->where('p.category_id', $request->category_id);
            }

            // Apply product filter
            if (!empty($request->product_id)) {
                $query->where('p.id', $request->product_id);
            }

            // Apply supplier filter
            if (!empty($request->supplier_id)) {
                $query->where('c.id', $request->supplier_id);
            }

            // Apply date filter
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $query->whereDate('t.transaction_date', '>=', $request->start_date)
                    ->whereDate('t.transaction_date', '<=', $request->end_date);
            }

            // Permission check
            if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
                $query->where('t.created_by', request()->session()->get('user.id'));
            }

            $util = new Util();
            
            return DataTables::of($query)
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row->transaction_date, true);
                })
                ->addColumn('supplier_name', function ($row) {
                    $supplier = $row->supplier_name ?? '';
                    if (!empty($row->supplier_business_name)) {
                        $supplier .= ' (' . $row->supplier_business_name . ')';
                    }
                    return $supplier;
                })
                ->editColumn('quantity', function ($row) use ($util) {
                    return $util->num_f($row->quantity);
                })
                ->addColumn('unit_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->unit_price . '</span>';
                })
                ->addColumn('line_total', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->line_total . '</span>';
                })
                ->rawColumns(['unit_price', 'line_total'])
                ->make(true);
        }

        return view('businessmanagement::purchase_register.supplier_and_product_wise_purchase', compact('categories', 'locations', 'products', 'suppliers'));
    }

    /**
     * Print Supplier and Product Wise Purchase Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printSupplierAndProductWisePurchase(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get invoice layout logo
        $logo = false;
        $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
            ->where('is_default', 1)
            ->first();
        if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
            $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
        }

        $start_date = $request->start_date ?? null;
        $end_date = $request->end_date ?? null;
        $location_id = $request->location_id ?? null;
        $category_id = $request->category_id ?? null;
        $product_id = $request->product_id ?? null;
        $supplier_id = $request->supplier_id ?? null;

        // Get purchase lines
        $query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('brands as b', 'p.brand_id', '=', 'b.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->select(
                't.transaction_date',
                DB::raw('COALESCE(t.invoice_no, t.ref_no, "") as invoice_no'),
                'c.id as supplier_id',
                'c.name as supplier_name',
                'c.supplier_business_name',
                'cat.id as category_id',
                'cat.name as category_name',
                'b.id as brand_id',
                'b.name as brand_name',
                'purchase_lines.quantity',
                'purchase_lines.purchase_price_inc_tax as unit_price',
                DB::raw("(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as line_total"),
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_description")
            );

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $query->where('t.location_id', $location_id);
        }

        // Apply category filter
        if (!empty($category_id)) {
            $query->where('p.category_id', $category_id);
        }

        // Apply product filter
        if (!empty($product_id)) {
            $query->where('p.id', $product_id);
        }

        // Apply supplier filter
        if (!empty($supplier_id)) {
            $query->where('c.id', $supplier_id);
        }

        // Apply date filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        // Permission check
        if (!auth()->user()->can('purchase.view') && auth()->user()->can('view_own_purchase')) {
            $query->where('t.created_by', request()->session()->get('user.id'));
        }

        $purchase_lines = $query->orderBy('c.name', 'asc')
            ->orderBy('cat.name', 'asc')
            ->orderBy('t.transaction_date', 'asc')
            ->orderBy('t.invoice_no', 'asc')
            ->get();

        // Determine grouping structure
        $is_single_supplier = !empty($supplier_id);
        $is_single_category = !empty($category_id);
        $is_single_product = !empty($product_id);

        // Group data based on filters
        $grouped_data = [];
        $grand_total_qty = 0;
        $grand_total_amount = 0;

        foreach ($purchase_lines as $line) {
            $supplier_name = $line->supplier_name ?? '';
            if (!empty($line->supplier_business_name)) {
                $supplier_name .= ' (' . $line->supplier_business_name . ')';
            }
            $category_name = $line->category_name ?? 'UNCATEGORIZED';
            $brand_name = $line->brand_name ?? 'NO BRAND';

            // Determine grouping key
            if ($is_single_supplier && $is_single_category) {
                // Single supplier, single category - no grouping
                $group_key = 'all';
            } elseif ($is_single_supplier) {
                // Single supplier, multiple categories - group by category
                $group_key = 'category_' . ($line->category_id ?? 0);
            } elseif ($is_single_category) {
                // Multiple suppliers, single category - group by supplier
                $group_key = 'supplier_' . ($line->supplier_id ?? 0);
            } else {
                // Multiple suppliers, multiple categories - group by category then supplier
                $group_key = 'category_' . ($line->category_id ?? 0) . '_supplier_' . ($line->supplier_id ?? 0);
            }

            if (!isset($grouped_data[$group_key])) {
                $grouped_data[$group_key] = [
                    'category_id' => $line->category_id ?? 0,
                    'category_name' => $category_name,
                    'supplier_id' => $line->supplier_id ?? 0,
                    'supplier_name' => $supplier_name,
                    'lines' => [],
                    'total_qty' => 0,
                    'total_amount' => 0
                ];
            }

            $grouped_data[$group_key]['lines'][] = $line;
            $grouped_data[$group_key]['total_qty'] += $line->quantity;
            $grouped_data[$group_key]['total_amount'] += $line->line_total;
            $grand_total_qty += $line->quantity;
            $grand_total_amount += $line->line_total;
        }

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::purchase_register.print_supplier_and_product_wise_purchase', compact(
            'grouped_data',
            'business',
            'logo',
            'start_date',
            'end_date',
            'is_single_supplier',
            'is_single_category',
            'is_single_product',
            'grand_total_qty',
            'grand_total_amount',
            'util',
            'currency'
        ));
    }

    /**
     * Display Purchased Product Movement Report
     *
     * @return \Illuminate\Http\Response
     */
    public function purchasedProductMovement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = Category::forDropdown($business_id, 'product');
        $locations = BusinessLocation::forDropdown($business_id, true);
        $suppliers = Contact::suppliersDropdown($business_id, false);

        if ($request->ajax()) {
            // Parse date range
            $start_date = null;
            $end_date = null;
            if (!empty($request->date_range)) {
                $date_range = explode(' ~ ', $request->date_range);
                if (count($date_range) == 2) {
                    $start_date = $date_range[0];
                    $end_date = $date_range[1];
                }
            }

            // Get purchase lines - exclude opening stock
            $purchase_query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'purchase')
                ->where('t.status', 'received')
                ->select(
                    'purchase_lines.id as purchase_line_id',
                    't.transaction_date',
                    't.type as transaction_db_type',
                    DB::raw("'Purchase' as transaction_type"),
                    DB::raw('COALESCE(t.invoice_no, t.ref_no, "") as invoice_no'),
                    'c.id as supplier_id',
                    'c.name as supplier_name',
                    'c.supplier_business_name',
                    'cat.id as category_id',
                    'cat.name as category_name',
                    'p.id as product_id',
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_name"),
                    'purchase_lines.quantity',
                    'purchase_lines.purchase_price_inc_tax as unit_price',
                    'purchase_lines.purchase_price as unit_cost',
                    DB::raw("(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as amount"),
                    DB::raw("(purchase_lines.quantity * purchase_lines.purchase_price) as cost_amount")
                );

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $purchase_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                $purchase_query->where('t.location_id', $request->location_id);
            }

            // Apply category filter
            if (!empty($request->category_id)) {
                $purchase_query->where('p.category_id', $request->category_id);
            }

            // Apply supplier filter
            if (!empty($request->supplier_id)) {
                $purchase_query->where('c.id', $request->supplier_id);
            }

            // Apply invoice filter
            if (!empty($request->invoice_no)) {
                $purchase_query->where(function($q) use ($request) {
                    $q->where('t.invoice_no', 'like', '%' . $request->invoice_no . '%')
                      ->orWhere('t.ref_no', 'like', '%' . $request->invoice_no . '%');
                });
            }

            // Apply date filter
            if (!empty($start_date) && !empty($end_date)) {
                $purchase_query->whereDate('t.transaction_date', '>=', $start_date)
                    ->whereDate('t.transaction_date', '<=', $end_date);
            }

            $purchase_lines = $purchase_query->get();

            // Get sell lines linked to purchase lines
            // Exclude those with no supplier or no invoice (will be handled as opening stock)
            $sell_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
                ->join('transaction_sell_lines_purchase_lines as tspl', 'transaction_sell_lines.id', '=', 'tspl.sell_line_id')
                ->join('purchase_lines as pl', 'tspl.purchase_line_id', '=', 'pl.id')
                ->join('transactions as pt', 'pl.transaction_id', '=', 'pt.id')
                ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                ->leftJoin('contacts as c', 'pt.contact_id', '=', 'c.id')
                ->where('t.business_id', $business_id)
                ->where('t.type', 'sell')
                ->where('t.status', 'final')
                ->where('transaction_sell_lines.children_type', '!=', 'combo')
                ->whereNotNull('tspl.purchase_line_id')
                ->where('tspl.purchase_line_id', '!=', 0)
                ->whereNotNull('pt.contact_id')
                ->where(function($q) {
                    $q->whereNotNull('pt.invoice_no')
                      ->orWhereNotNull('pt.ref_no');
                })
                ->where(function($q) {
                    $q->where('pt.invoice_no', '!=', '')
                      ->orWhere('pt.ref_no', '!=', '');
                })
                ->select(
                    'pl.id as purchase_line_id',
                    't.transaction_date',
                    'pt.transaction_date as purchase_date',
                    DB::raw('COALESCE(pt.invoice_no, pt.ref_no, "") as invoice_no'),
                    'c.id as supplier_id',
                    'c.name as supplier_name',
                    'c.supplier_business_name',
                    'cat.id as category_id',
                    'cat.name as category_name',
                    'p.id as product_id',
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_name"),
                    DB::raw("(tspl.quantity - COALESCE(tspl.qty_returned, 0)) as quantity"),
                    'transaction_sell_lines.unit_price_inc_tax as unit_price',
                    'pl.purchase_price as unit_cost',
                    DB::raw("((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * transaction_sell_lines.unit_price_inc_tax) as amount"),
                    DB::raw("((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * pl.purchase_price) as cost_amount"),
                    DB::raw("((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * (transaction_sell_lines.unit_price_inc_tax - pl.purchase_price_inc_tax)) as profit"),
                    DB::raw("'Sales' as transaction_type")
                );

            // Apply same filters
            if ($permitted_locations != 'all') {
                $sell_query->whereIn('t.location_id', $permitted_locations);
            }

            if (!empty($request->location_id)) {
                $sell_query->where('t.location_id', $request->location_id);
            }

            if (!empty($request->category_id)) {
                $sell_query->where('p.category_id', $request->category_id);
            }

            if (!empty($request->supplier_id)) {
                $sell_query->where('c.id', $request->supplier_id);
            }

            if (!empty($request->invoice_no)) {
                $sell_query->where(function($q) use ($request) {
                    $q->where('pt.invoice_no', 'like', '%' . $request->invoice_no . '%')
                      ->orWhere('pt.ref_no', 'like', '%' . $request->invoice_no . '%');
                });
            }

            // Filter by sales transaction date, not purchase date
            if (!empty($start_date) && !empty($end_date)) {
                $sell_query->whereDate('t.transaction_date', '>=', $start_date)
                    ->whereDate('t.transaction_date', '<=', $end_date);
            }

            $sell_lines = $sell_query->get();

            // Combine and format data
            $data = [];
            
            // Add purchase lines - exclude opening stock
            foreach ($purchase_lines as $line) {
                // Build supplier name with business name for consistency
                $supplier_name = $line->supplier_name ?? '';
                if (!empty($line->supplier_business_name)) {
                    $supplier_name .= ' (' . $line->supplier_business_name . ')';
                }
                
                $data[] = [
                    'supplier_name' => $supplier_name,
                    'category_name' => $line->category_name ?? 'UNCATEGORIZED',
                    'product_name' => $line->product_name ?? 'N/A',
                    'invoice_no' => $line->invoice_no ?? '',
                    'transaction_date' => $line->transaction_date,
                    'transaction_type' => 'Purchase',
                    'quantity' => $line->quantity,
                    'unit_price' => $line->unit_price,
                    'unit_cost' => $line->unit_cost,
                    'cost_amount' => $line->cost_amount,
                    'amount' => $line->amount,
                    'profit' => 0
                ];
            }

            // Add sell lines - group by supplier+product to consolidate
            // This ensures sales from different suppliers are tracked separately
            $sales_grouped = [];
            foreach ($sell_lines as $line) {
                // Build supplier name with business name for consistency
                $supplier_name = $line->supplier_name ?? '';
                if (!empty($line->supplier_business_name)) {
                    $supplier_name .= ' (' . $line->supplier_business_name . ')';
                }
                
                // Group by supplier+product to track which supplier's stock was sold
                $product_key = $supplier_name . '|' . ($line->product_name ?? 'N/A');
                
                if (!isset($sales_grouped[$product_key])) {
                    $sales_grouped[$product_key] = [
                        'supplier_name' => $supplier_name,
                        'category_name' => $line->category_name ?? 'UNCATEGORIZED',
                        'product_name' => $line->product_name ?? 'N/A',
                        'invoice_no' => $line->invoice_no ?? '',
                        'transaction_date' => $line->transaction_date,
                        'transaction_type' => 'Sales',
                        'quantity' => 0,
                        'unit_price' => $line->unit_price ?? 0,
                        'unit_cost' => $line->unit_cost ?? 0,
                        'cost_amount' => 0,
                        'amount' => 0,
                        'profit' => 0
                    ];
                }
                
                $sales_grouped[$product_key]['quantity'] += $line->quantity ?? 0;
                $sales_grouped[$product_key]['cost_amount'] += $line->cost_amount ?? 0;
                $sales_grouped[$product_key]['amount'] += $line->amount ?? 0;
                $sales_grouped[$product_key]['profit'] += $line->profit ?? 0;
            }
            
            // Add consolidated sales
            foreach ($sales_grouped as $sales) {
                // Recalculate unit price based on total amount and quantity
                if ($sales['quantity'] > 0) {
                    $sales['unit_price'] = $sales['amount'] / $sales['quantity'];
                    $sales['unit_cost'] = $sales['cost_amount'] / $sales['quantity'];
                }
                $data[] = $sales;
            }


            $util = new Util();
            
            return DataTables::of(collect($data))
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row['transaction_date'], true);
                })
                ->addColumn('unit_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['unit_price'] . '</span>';
                })
                ->addColumn('unit_cost', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['unit_cost'] . '</span>';
                })
                ->addColumn('cost_amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['cost_amount'] . '</span>';
                })
                ->addColumn('amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['amount'] . '</span>';
                })
                ->addColumn('profit', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['profit'] . '</span>';
                })
                ->rawColumns(['unit_price', 'unit_cost', 'cost_amount', 'amount', 'profit'])
                ->make(true);
        }

        return view('businessmanagement::purchase_register.purchased_product_movement', compact('categories', 'locations', 'suppliers'));
    }

    /**
     * Print Purchased Product Movement Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printPurchasedProductMovement(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get invoice layout logo
        $logo = false;
        $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
            ->where('is_default', 1)
            ->first();
        if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
            $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
        }

        // Parse date range - handle both " ~ " and " - " separators
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            // Try tilde separator first
            $date_range = explode(' ~ ', $request->date_range);
            if (count($date_range) != 2) {
                // Try dash separator
                $date_range = explode(' - ', $request->date_range);
            }
            if (count($date_range) == 2) {
                $start_date_raw = trim($date_range[0]);
                $end_date_raw = trim($date_range[1]);
                
                // Convert date format from MM/DD/YYYY to YYYY-MM-DD
                try {
                    $start_date = \Carbon::createFromFormat('m/d/Y', $start_date_raw)->format('Y-m-d');
                } catch (\Exception $e) {
                    // Try other formats
                    try {
                        $start_date = \Carbon::parse($start_date_raw)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $start_date = $start_date_raw;
                    }
                }
                
                try {
                    $end_date = \Carbon::createFromFormat('m/d/Y', $end_date_raw)->format('Y-m-d');
                } catch (\Exception $e) {
                    // Try other formats
                    try {
                        $end_date = \Carbon::parse($end_date_raw)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $end_date = $end_date_raw;
                    }
                }
            }
        }
        
        // Also check for separate start_date and end_date parameters
        if (empty($start_date) && !empty($request->start_date)) {
            try {
                $start_date = \Carbon::createFromFormat('m/d/Y', $request->start_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $start_date = \Carbon::parse($request->start_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $start_date = $request->start_date;
                }
            }
        }
        if (empty($end_date) && !empty($request->end_date)) {
            try {
                $end_date = \Carbon::createFromFormat('m/d/Y', $request->end_date)->format('Y-m-d');
            } catch (\Exception $e) {
                try {
                    $end_date = \Carbon::parse($request->end_date)->format('Y-m-d');
                } catch (\Exception $e2) {
                    $end_date = $request->end_date;
                }
            }
        }

        $location_id = $request->location_id ?? null;
        $category_id = $request->category_id ?? null;
        $supplier_filter = $request->supplier_filter ?? 'all';
        $supplier_id = ($supplier_filter == 'single' && !empty($request->supplier_id)) ? $request->supplier_id : null;
        $invoice_filter = $request->invoice_filter ?? 'all';
        $invoice_no = ($invoice_filter == 'single' && !empty($request->invoice_no)) ? $request->invoice_no : null;

        // Get purchase lines - exclude opening stock
        $purchase_query = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
            ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'purchase')
            ->where('t.status', 'received')
            ->select(
                'purchase_lines.id as purchase_line_id',
                't.transaction_date',
                't.type as transaction_db_type',
                DB::raw("'Purchase' as transaction_type"),
                DB::raw('COALESCE(t.invoice_no, t.ref_no, "") as invoice_no'),
                'c.id as supplier_id',
                'c.name as supplier_name',
                'c.supplier_business_name',
                'cat.id as category_id',
                'cat.name as category_name',
                'p.id as product_id',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_name"),
                'purchase_lines.quantity',
                'purchase_lines.purchase_price_inc_tax as unit_price',
                'purchase_lines.purchase_price as unit_cost',
                DB::raw("(purchase_lines.quantity * purchase_lines.purchase_price_inc_tax) as amount"),
                DB::raw("(purchase_lines.quantity * purchase_lines.purchase_price) as cost_amount")
            );

        // Apply filters
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $purchase_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $purchase_query->where('t.location_id', $location_id);
        }

        if (!empty($category_id)) {
            $purchase_query->where('p.category_id', $category_id);
        }

        if (!empty($supplier_id)) {
            $purchase_query->where('c.id', $supplier_id);
        }

        if (!empty($invoice_no)) {
            $purchase_query->where(function($q) use ($invoice_no) {
                $q->where('t.invoice_no', 'like', '%' . $invoice_no . '%')
                  ->orWhere('t.ref_no', 'like', '%' . $invoice_no . '%');
            });
        }

        if (!empty($start_date) && !empty($end_date)) {
            $purchase_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        $purchase_lines = $purchase_query->get();

        // Get sell lines
        // Exclude those with no supplier or no invoice (will be handled as opening stock)
        $sell_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->join('transaction_sell_lines_purchase_lines as tspl', 'transaction_sell_lines.id', '=', 'tspl.sell_line_id')
            ->join('purchase_lines as pl', 'tspl.purchase_line_id', '=', 'pl.id')
            ->join('transactions as pt', 'pl.transaction_id', '=', 'pt.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('contacts as c', 'pt.contact_id', '=', 'c.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell')
            ->where('t.status', 'final')
            ->where('transaction_sell_lines.children_type', '!=', 'combo')
            ->whereNotNull('tspl.purchase_line_id')
            ->where('tspl.purchase_line_id', '!=', 0)
            ->whereNotNull('pt.contact_id')
            ->where(function($q) {
                $q->whereNotNull('pt.invoice_no')
                  ->orWhereNotNull('pt.ref_no');
            })
            ->where(function($q) {
                $q->where('pt.invoice_no', '!=', '')
                  ->orWhere('pt.ref_no', '!=', '');
            })
            ->select(
                'pl.id as purchase_line_id',
                't.transaction_date',
                'pt.transaction_date as purchase_date',
                DB::raw('COALESCE(pt.invoice_no, pt.ref_no, "") as invoice_no'),
                'c.id as supplier_id',
                'c.name as supplier_name',
                'c.supplier_business_name',
                'cat.id as category_id',
                'cat.name as category_name',
                'p.id as product_id',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_name"),
                DB::raw("(tspl.quantity - COALESCE(tspl.qty_returned, 0)) as quantity"),
                'transaction_sell_lines.unit_price_inc_tax as unit_price',
                'pl.purchase_price as unit_cost',
                DB::raw("((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * transaction_sell_lines.unit_price_inc_tax) as amount"),
                DB::raw("((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * pl.purchase_price) as cost_amount"),
                DB::raw("((tspl.quantity - COALESCE(tspl.qty_returned, 0)) * (transaction_sell_lines.unit_price_inc_tax - pl.purchase_price_inc_tax)) as profit")
            );

        // Apply same filters
        if ($permitted_locations != 'all') {
            $sell_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $sell_query->where('t.location_id', $location_id);
        }

        if (!empty($category_id)) {
            $sell_query->where('p.category_id', $category_id);
        }

        if (!empty($supplier_id)) {
            $sell_query->where('c.id', $supplier_id);
        }

        if (!empty($invoice_no)) {
            $sell_query->where(function($q) use ($invoice_no) {
                $q->where('pt.invoice_no', 'like', '%' . $invoice_no . '%')
                  ->orWhere('pt.ref_no', 'like', '%' . $invoice_no . '%');
            });
        }

        // Filter by sales transaction date, not purchase date
        if (!empty($start_date) && !empty($end_date)) {
            $sell_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        $sell_lines = $sell_query->get();

        // Get sales returns
        $return_query = TransactionSellLine::join('transactions as t', 'transaction_sell_lines.transaction_id', '=', 't.id')
            ->join('transaction_sell_lines_purchase_lines as tspl', 'transaction_sell_lines.id', '=', 'tspl.sell_line_id')
            ->join('purchase_lines as pl', 'tspl.purchase_line_id', '=', 'pl.id')
            ->join('transactions as pt', 'pl.transaction_id', '=', 'pt.id')
            ->leftJoin('variations as v', 'transaction_sell_lines.variation_id', '=', 'v.id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
            ->leftJoin('contacts as c', 'pt.contact_id', '=', 'c.id')
            ->where('t.business_id', $business_id)
            ->where('t.type', 'sell_return')
            ->where('t.status', 'final')
            ->where('transaction_sell_lines.children_type', '!=', 'combo')
            ->whereRaw('tspl.qty_returned > 0')
            ->select(
                'pl.id as purchase_line_id',
                't.transaction_date',
                'pt.transaction_date as purchase_date',
                DB::raw('COALESCE(pt.invoice_no, pt.ref_no, "") as invoice_no'),
                'c.id as supplier_id',
                'c.name as supplier_name',
                'c.supplier_business_name',
                'cat.id as category_id',
                'cat.name as category_name',
                'p.id as product_id',
                DB::raw("CASE 
                    WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                    ELSE COALESCE(p.name, 'N/A')
                END as product_name"),
                'tspl.qty_returned as quantity',
                'transaction_sell_lines.unit_price_inc_tax as unit_price',
                'pl.purchase_price as unit_cost',
                DB::raw("(tspl.qty_returned * transaction_sell_lines.unit_price_inc_tax) as amount"),
                DB::raw("(tspl.qty_returned * pl.purchase_price) as cost_amount"),
                DB::raw("(tspl.qty_returned * (transaction_sell_lines.unit_price_inc_tax - pl.purchase_price_inc_tax)) as profit")
            );

        // Apply same filters
        if ($permitted_locations != 'all') {
            $return_query->whereIn('t.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $return_query->where('t.location_id', $location_id);
        }

        if (!empty($category_id)) {
            $return_query->where('p.category_id', $category_id);
        }

        if (!empty($supplier_id)) {
            $return_query->where('c.id', $supplier_id);
        }

        if (!empty($invoice_no)) {
            $return_query->where(function($q) use ($invoice_no) {
                $q->where('pt.invoice_no', 'like', '%' . $invoice_no . '%')
                  ->orWhere('pt.ref_no', 'like', '%' . $invoice_no . '%');
            });
        }

        // Filter by return transaction date, not purchase date
        if (!empty($start_date) && !empty($end_date)) {
            $return_query->whereDate('t.transaction_date', '>=', $start_date)
                ->whereDate('t.transaction_date', '<=', $end_date);
        }

        $return_lines = $return_query->get();

        // Group data by supplier and product
        // Logic: 
        // - Products purchased from different suppliers are tracked separately
        // - Sales are linked to purchase lines via transaction_sell_lines_purchase_lines
        // - Sales are grouped by supplier+product to show which supplier's stock was sold
        // - Opening stock has no supplier and is grouped separately
        $grouped_data = [];
        
        // Process purchase lines - exclude opening stock
        foreach ($purchase_lines as $line) {
            $supplier_name = $line->supplier_name ?? '';
            if (!empty($line->supplier_business_name)) {
                $supplier_name .= ' (' . $line->supplier_business_name . ')';
            }
            $product_name = $line->product_name ?? 'N/A';
            $key = $supplier_name . '|' . $product_name;
            
            if (!isset($grouped_data[$key])) {
                $grouped_data[$key] = [
                    'supplier_name' => $supplier_name,
                    'product_name' => $product_name,
                    'category_name' => $line->category_name ?? 'UNCATEGORIZED',
                    'transactions' => [],
                    'total_profit' => 0
                ];
            }
            
            $grouped_data[$key]['transactions'][] = [
                'type' => 'Purchase',
                'invoice_no' => $line->invoice_no,
                'date' => $line->transaction_date,
                'qty' => $line->quantity,
                'unit_price' => $line->unit_price,
                'unit_cost' => $line->unit_cost,
                'cost_amount' => $line->cost_amount,
                'amount' => $line->amount,
                'profit' => 0
            ];
        }

        // Process sell lines - consolidate by product
        $sales_by_product = [];
        foreach ($sell_lines as $line) {
            $supplier_name = $line->supplier_name ?? '';
            if (!empty($line->supplier_business_name)) {
                $supplier_name .= ' (' . $line->supplier_business_name . ')';
            }
            $product_name = $line->product_name ?? 'N/A';
            $key = $supplier_name . '|' . $product_name;
            
            if (!isset($sales_by_product[$key])) {
                $sales_by_product[$key] = [
                    'supplier_name' => $supplier_name,
                    'product_name' => $product_name,
                    'category_name' => $line->category_name ?? 'UNCATEGORIZED',
                    'qty' => 0,
                    'unit_price' => $line->unit_price ?? 0,
                    'unit_cost' => $line->unit_cost ?? 0,
                    'cost_amount' => 0,
                    'amount' => 0,
                    'profit' => 0,
                    'invoice_no' => $line->invoice_no ?? '',
                    'date' => $line->transaction_date
                ];
            }
            
            $sales_by_product[$key]['qty'] += $line->quantity ?? 0;
            $sales_by_product[$key]['cost_amount'] += $line->cost_amount ?? 0;
            $sales_by_product[$key]['amount'] += $line->amount ?? 0;
            $sales_by_product[$key]['profit'] += $line->profit ?? 0;
        }
        
        // Add consolidated sales to grouped_data
        foreach ($sales_by_product as $key => $sales) {
            if (!isset($grouped_data[$key])) {
                $grouped_data[$key] = [
                    'supplier_name' => $sales['supplier_name'],
                    'product_name' => $sales['product_name'],
                    'category_name' => $sales['category_name'],
                    'transactions' => [],
                    'total_profit' => 0
                ];
            }
            
            // Recalculate unit price and unit cost based on totals
            if ($sales['qty'] > 0) {
                $sales['unit_price'] = $sales['amount'] / $sales['qty'];
                $sales['unit_cost'] = $sales['cost_amount'] / $sales['qty'];
            }
            
            $grouped_data[$key]['transactions'][] = [
                'type' => 'Sales',
                'invoice_no' => $sales['invoice_no'],
                'date' => $sales['date'],
                'qty' => $sales['qty'],
                'unit_price' => $sales['unit_price'],
                'unit_cost' => $sales['unit_cost'],
                'cost_amount' => $sales['cost_amount'],
                'amount' => $sales['amount'],
                'profit' => $sales['profit']
            ];
            $grouped_data[$key]['total_profit'] += $sales['profit'];
        }

        // Process return lines - consolidate by product
        $returns_by_product = [];
        foreach ($return_lines as $line) {
            $supplier_name = $line->supplier_name ?? '';
            if (!empty($line->supplier_business_name)) {
                $supplier_name .= ' (' . $line->supplier_business_name . ')';
            }
            $product_name = $line->product_name ?? 'N/A';
            $key = $supplier_name . '|' . $product_name;
            
            if (!isset($returns_by_product[$key])) {
                $returns_by_product[$key] = [
                    'supplier_name' => $supplier_name,
                    'product_name' => $product_name,
                    'category_name' => $line->category_name ?? 'UNCATEGORIZED',
                    'qty' => 0,
                    'unit_price' => $line->unit_price ?? 0,
                    'unit_cost' => $line->unit_cost ?? 0,
                    'cost_amount' => 0,
                    'amount' => 0,
                    'profit' => 0,
                    'invoice_no' => $line->invoice_no ?? '',
                    'date' => $line->transaction_date
                ];
            }
            
            $returns_by_product[$key]['qty'] += $line->quantity ?? 0;
            $returns_by_product[$key]['cost_amount'] += $line->cost_amount ?? 0;
            $returns_by_product[$key]['amount'] += $line->amount ?? 0;
            $returns_by_product[$key]['profit'] += $line->profit ?? 0;
        }
        
        // Add consolidated returns to grouped_data
        foreach ($returns_by_product as $key => $return) {
            if (!isset($grouped_data[$key])) {
                $grouped_data[$key] = [
                    'supplier_name' => $return['supplier_name'],
                    'product_name' => $return['product_name'],
                    'category_name' => $return['category_name'],
                    'transactions' => [],
                    'total_profit' => 0
                ];
            }
            
            // Recalculate unit price and unit cost based on totals
            if ($return['qty'] > 0) {
                $return['unit_price'] = $return['amount'] / $return['qty'];
                $return['unit_cost'] = $return['cost_amount'] / $return['qty'];
            }
            
            $grouped_data[$key]['transactions'][] = [
                'type' => 'Sales Return',
                'invoice_no' => $return['invoice_no'],
                'date' => $return['date'],
                'qty' => $return['qty'],
                'unit_price' => $return['unit_price'],
                'unit_cost' => $return['unit_cost'],
                'cost_amount' => $return['cost_amount'],
                'amount' => $return['amount'],
                'profit' => $return['profit']
            ];
            $grouped_data[$key]['total_profit'] += $return['profit'];
        }

        // Calculate stock in hand for each product
        foreach ($grouped_data as $key => &$group) {
            $purchase_qty = 0;
            $purchase_cost = 0;
            $purchase_amount = 0;
            $sales_qty = 0;
            $sales_cost = 0;
            $sales_amount = 0;
            $return_qty = 0;
            $return_cost = 0;
            $return_amount = 0;

            foreach ($group['transactions'] as $trans) {
                if ($trans['type'] == 'Purchase') {
                    $purchase_qty += $trans['qty'];
                    $purchase_cost += $trans['cost_amount'];
                    $purchase_amount += $trans['amount'];
                } elseif ($trans['type'] == 'Sales') {
                    $sales_qty += $trans['qty'];
                    $sales_cost += $trans['cost_amount'];
                    $sales_amount += $trans['amount'];
                } elseif ($trans['type'] == 'Sales Return') {
                    $return_qty += $trans['qty'];
                    $return_cost += $trans['cost_amount'];
                    $return_amount += $trans['amount'];
                }
            }

            $stock_qty = $purchase_qty - $sales_qty + $return_qty;
            $stock_cost = $purchase_cost - $sales_cost + $return_cost;
            
            // Stock amount should be based on cost, not selling price
            // Use stock_cost as the amount (stock is valued at cost)
            $stock_amount = $stock_cost;
            
            if ($stock_qty > 0) {
                $avg_unit_cost = $stock_cost / $stock_qty;
                // Unit price for stock should be the same as unit cost (stock is valued at cost)
                $avg_unit_price = $avg_unit_cost;
            } else {
                $avg_unit_price = 0;
                $avg_unit_cost = 0;
                $stock_amount = 0; // Ensure amount is 0 when qty is 0
            }

            $group['stock_in_hand'] = [
                'qty' => $stock_qty,
                'unit_price' => $avg_unit_price,
                'unit_cost' => $avg_unit_cost,
                'cost_amount' => $stock_cost,
                'amount' => $stock_amount,
                'profit' => 0
            ];
        }

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::purchase_register.print_purchased_product_movement', compact(
            'grouped_data',
            'business',
            'logo',
            'start_date',
            'end_date',
            'util',
            'currency'
        ));
    }

    /**
     * Supplier wise Purchase Return Report
     *
     * @return \Illuminate\Http\Response
     */
    public function supplierWisePurchaseReturn(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $suppliers = Contact::suppliersDropdown($business_id, false);
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            // Parse date range
            $start_date = null;
            $end_date = null;
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $start_date = $request->start_date;
                $end_date = $request->end_date;
            }

            $supplier_filter = $request->supplier_filter ?? 'all';
            $supplier_id = ($supplier_filter == 'single' && !empty($request->supplier_id)) ? $request->supplier_id : null;
            $location_id = $request->location_id ?? null;

            // Get purchase return transactions
            $return_transactions = \App\Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->whereIn('status', ['received', 'final']);

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $return_transactions->whereIn('location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                $return_transactions->where('location_id', $location_id);
            }

            if (!empty($start_date) && !empty($end_date)) {
                $return_transactions->whereDate('transaction_date', '>=', $start_date)
                    ->whereDate('transaction_date', '<=', $end_date);
            }

            $return_transactions = $return_transactions->get();

            $return_data = [];

            foreach ($return_transactions as $return_transaction) {
                // Check if this is a regular return (has return_parent_id) or combined return
                if (!empty($return_transaction->return_parent_id)) {
                    // Regular return - get original purchase lines with quantity_returned > 0
                    $purchase_lines = PurchaseLine::join('transactions as pt', 'purchase_lines.transaction_id', '=', 'pt.id')
                        ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                        ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                        ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                        ->leftJoin('contacts as c', 'pt.contact_id', '=', 'c.id')
                        ->where('pt.id', $return_transaction->return_parent_id)
                        ->where('purchase_lines.quantity_returned', '>', 0)
                        ->select(
                            'purchase_lines.id',
                            'c.id as supplier_id',
                            'c.name as supplier_name',
                            'c.supplier_business_name',
                            'p.id as product_id',
                            DB::raw("CASE 
                                WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                                ELSE COALESCE(p.name, 'N/A')
                            END as product_name"),
                            DB::raw('purchase_lines.quantity_returned as quantity'),
                            'purchase_lines.purchase_price_inc_tax as unit_price',
                            DB::raw('COALESCE(purchase_lines.pp_without_discount, purchase_lines.purchase_price_inc_tax) as price_without_discount'),
                            DB::raw('COALESCE(purchase_lines.discount_percent, 0) as discount_percent'),
                            DB::raw('COALESCE(
                                CASE 
                                    WHEN purchase_lines.discount_percent > 0 AND purchase_lines.pp_without_discount > 0 
                                    THEN (purchase_lines.pp_without_discount * purchase_lines.discount_percent) / 100
                                    WHEN purchase_lines.pp_without_discount > 0 AND purchase_lines.purchase_price_inc_tax < purchase_lines.pp_without_discount
                                    THEN purchase_lines.pp_without_discount - purchase_lines.purchase_price_inc_tax
                                    ELSE 0
                                END, 0
                            ) as discount'),
                            DB::raw('purchase_lines.purchase_price_inc_tax as net_unit_price'),
                            DB::raw('(purchase_lines.quantity_returned * purchase_lines.purchase_price_inc_tax) as amount')
                        );

                    // Apply supplier filter
                    if (!empty($supplier_id)) {
                        $purchase_lines->where('c.id', $supplier_id);
                    }

                    $lines = $purchase_lines->get();
                    foreach ($lines as $line) {
                        // Manually set the return transaction fields
                        $line->transaction_date = $return_transaction->transaction_date;
                        $line->invoice_no = !empty($return_transaction->invoice_no) ? $return_transaction->invoice_no : (!empty($return_transaction->ref_no) ? $return_transaction->ref_no : '');
                        $line->ref_no = $return_transaction->ref_no;
                        $return_data[] = $line;
                    }
                } else {
                    // Combined return - get purchase_lines directly from return transaction
                    $purchase_lines = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                        ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                        ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                        ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                        ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                        ->where('t.id', $return_transaction->id)
                        ->where('purchase_lines.quantity_returned', '>', 0)
                        ->select(
                            'purchase_lines.id',
                            't.transaction_date',
                            't.invoice_no',
                            't.ref_no',
                            'c.id as supplier_id',
                            'c.name as supplier_name',
                            'c.supplier_business_name',
                            'p.id as product_id',
                            DB::raw("CASE 
                                WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                                ELSE COALESCE(p.name, 'N/A')
                            END as product_name"),
                            DB::raw('purchase_lines.quantity_returned as quantity'),
                            'purchase_lines.purchase_price_inc_tax as unit_price',
                            DB::raw('COALESCE(purchase_lines.pp_without_discount, purchase_lines.purchase_price_inc_tax) as price_without_discount'),
                            DB::raw('COALESCE(purchase_lines.discount_percent, 0) as discount_percent'),
                            DB::raw('COALESCE(
                                CASE 
                                    WHEN purchase_lines.discount_percent > 0 AND purchase_lines.pp_without_discount > 0 
                                    THEN (purchase_lines.pp_without_discount * purchase_lines.discount_percent) / 100
                                    WHEN purchase_lines.pp_without_discount > 0 AND purchase_lines.purchase_price_inc_tax < purchase_lines.pp_without_discount
                                    THEN purchase_lines.pp_without_discount - purchase_lines.purchase_price_inc_tax
                                    ELSE 0
                                END, 0
                            ) as discount'),
                            DB::raw('purchase_lines.purchase_price_inc_tax as net_unit_price'),
                            DB::raw('(purchase_lines.quantity_returned * purchase_lines.purchase_price_inc_tax) as amount')
                        );

                    // Apply supplier filter
                    if (!empty($supplier_id)) {
                        $purchase_lines->where('c.id', $supplier_id);
                    }

                    $lines = $purchase_lines->get();
                    foreach ($lines as $line) {
                        $return_data[] = $line;
                    }
                }
            }

            // Convert to collection for DataTables
            $collection = collect($return_data);

            $util = new Util();
            
            return DataTables::of($collection)
                ->editColumn('supplier_name', function ($row) {
                    $supplier_name = $row->supplier_name ?? '';
                    if (!empty($row->supplier_business_name)) {
                        $supplier_name .= ' (' . $row->supplier_business_name . ')';
                    }
                    return $supplier_name;
                })
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row->transaction_date, true);
                })
                ->editColumn('invoice_no', function ($row) {
                    return !empty($row->invoice_no) ? $row->invoice_no : (!empty($row->ref_no) ? $row->ref_no : '');
                })
                ->addColumn('unit_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->unit_price . '</span>';
                })
                ->addColumn('discount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->discount . '</span>';
                })
                ->addColumn('net_unit_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->net_unit_price . '</span>';
                })
                ->addColumn('amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row->amount . '</span>';
                })
                ->rawColumns(['unit_price', 'discount', 'net_unit_price', 'amount'])
                ->make(true);
        }

        return view('businessmanagement::purchase_register.supplier_wise_purchase_return', compact('suppliers', 'locations'));
    }

    /**
     * Print Supplier wise Purchase Return Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printSupplierWisePurchaseReturn(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get invoice layout logo
        $logo = false;
        $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
            ->where('is_default', 1)
            ->first();
        if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
            $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
        }

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = explode(' ~ ', $request->date_range);
            if (count($date_range) != 2) {
                $date_range = explode(' - ', $request->date_range);
            }
            if (count($date_range) == 2) {
                $start_date_raw = trim($date_range[0]);
                $end_date_raw = trim($date_range[1]);
                
                try {
                    $start_date = \Carbon::createFromFormat('m/d/Y', $start_date_raw)->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon::parse($start_date_raw)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $start_date = $start_date_raw;
                    }
                }
                
                try {
                    $end_date = \Carbon::createFromFormat('m/d/Y', $end_date_raw)->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $end_date = \Carbon::parse($end_date_raw)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $end_date = $end_date_raw;
                    }
                }
            }
        }

        $supplier_filter = $request->supplier_filter ?? 'all';
        $supplier_id = ($supplier_filter == 'single' && !empty($request->supplier_id)) ? $request->supplier_id : null;
        $location_id = $request->location_id ?? null;

        // Get purchase return transactions
        $return_transactions = \App\Transaction::where('business_id', $business_id)
            ->where('type', 'purchase_return')
            ->whereIn('status', ['received', 'final']);

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $return_transactions->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $return_transactions->where('location_id', $location_id);
        }

        if (!empty($start_date) && !empty($end_date)) {
            $return_transactions->whereDate('transaction_date', '>=', $start_date)
                ->whereDate('transaction_date', '<=', $end_date);
        }

        $return_transactions = $return_transactions->get();

        $return_data = [];

        foreach ($return_transactions as $return_transaction) {
            // Check if this is a regular return (has return_parent_id) or combined return
            if (!empty($return_transaction->return_parent_id)) {
                // Regular return - get original purchase lines with quantity_returned > 0
                $purchase_lines = PurchaseLine::join('transactions as pt', 'purchase_lines.transaction_id', '=', 'pt.id')
                    ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                    ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                    ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                    ->leftJoin('contacts as c', 'pt.contact_id', '=', 'c.id')
                    ->where('pt.id', $return_transaction->return_parent_id)
                    ->where('purchase_lines.quantity_returned', '>', 0)
                    ->select(
                        'purchase_lines.id',
                        'c.id as supplier_id',
                        'c.name as supplier_name',
                        'c.supplier_business_name',
                        'p.id as product_id',
                        DB::raw("CASE 
                            WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                            ELSE COALESCE(p.name, 'N/A')
                        END as product_name"),
                        DB::raw('purchase_lines.quantity_returned as quantity'),
                        'purchase_lines.purchase_price_inc_tax as unit_price',
                        DB::raw('COALESCE(purchase_lines.pp_without_discount, purchase_lines.purchase_price_inc_tax) as price_without_discount'),
                        DB::raw('COALESCE(purchase_lines.discount_percent, 0) as discount_percent'),
                        DB::raw('COALESCE(
                            CASE 
                                WHEN purchase_lines.discount_percent > 0 AND purchase_lines.pp_without_discount > 0 
                                THEN (purchase_lines.pp_without_discount * purchase_lines.discount_percent) / 100
                                WHEN purchase_lines.pp_without_discount > 0 AND purchase_lines.purchase_price_inc_tax < purchase_lines.pp_without_discount
                                THEN purchase_lines.pp_without_discount - purchase_lines.purchase_price_inc_tax
                                ELSE 0
                            END, 0
                        ) as discount'),
                        DB::raw('purchase_lines.purchase_price_inc_tax as net_unit_price'),
                        DB::raw('(purchase_lines.quantity_returned * purchase_lines.purchase_price_inc_tax) as amount')
                    );

                // Apply supplier filter
                if (!empty($supplier_id)) {
                    $purchase_lines->where('c.id', $supplier_id);
                }

                $lines = $purchase_lines->get();
                foreach ($lines as $line) {
                    // Manually set the return transaction fields
                    $line->transaction_date = $return_transaction->transaction_date;
                    $line->invoice_no = !empty($return_transaction->invoice_no) ? $return_transaction->invoice_no : (!empty($return_transaction->ref_no) ? $return_transaction->ref_no : '');
                    $line->ref_no = $return_transaction->ref_no;
                    $return_data[] = $line;
                }
            } else {
                // Combined return - get purchase_lines directly from return transaction
                $purchase_lines = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                    ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                    ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                    ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                    ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                    ->where('t.id', $return_transaction->id)
                    ->where('purchase_lines.quantity_returned', '>', 0)
                    ->select(
                        'purchase_lines.id',
                        't.transaction_date',
                        't.invoice_no',
                        't.ref_no',
                        'c.id as supplier_id',
                        'c.name as supplier_name',
                        'c.supplier_business_name',
                        'p.id as product_id',
                        DB::raw("CASE 
                            WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                            ELSE COALESCE(p.name, 'N/A')
                        END as product_name"),
                        DB::raw('purchase_lines.quantity_returned as quantity'),
                        'purchase_lines.purchase_price_inc_tax as unit_price',
                        DB::raw('COALESCE(purchase_lines.pp_without_discount, purchase_lines.purchase_price_inc_tax) as price_without_discount'),
                        DB::raw('COALESCE(purchase_lines.discount_percent, 0) as discount_percent'),
                        DB::raw('COALESCE(
                            CASE 
                                WHEN purchase_lines.discount_percent > 0 AND purchase_lines.pp_without_discount > 0 
                                THEN (purchase_lines.pp_without_discount * purchase_lines.discount_percent) / 100
                                WHEN purchase_lines.pp_without_discount > 0 AND purchase_lines.purchase_price_inc_tax < purchase_lines.pp_without_discount
                                THEN purchase_lines.pp_without_discount - purchase_lines.purchase_price_inc_tax
                                ELSE 0
                            END, 0
                        ) as discount'),
                        DB::raw('purchase_lines.purchase_price_inc_tax as net_unit_price'),
                        DB::raw('(purchase_lines.quantity_returned * purchase_lines.purchase_price_inc_tax) as amount')
                    );

                // Apply supplier filter
                if (!empty($supplier_id)) {
                    $purchase_lines->where('c.id', $supplier_id);
                }

                $lines = $purchase_lines->get();
                foreach ($lines as $line) {
                    $return_data[] = $line;
                }
            }
        }

        // Group data by supplier and date
        $grouped_data = [];
        $grand_total_qty = 0;
        $grand_total_amount = 0;

        foreach ($return_data as $line) {
            $supplier_name = $line->supplier_name ?? '';
            if (!empty($line->supplier_business_name)) {
                $supplier_name .= ' (' . $line->supplier_business_name . ')';
            }
            $date_key = date('Y-m-d', strtotime($line->transaction_date));
            $key = $supplier_name . '|' . $date_key;
            
            if (!isset($grouped_data[$key])) {
                $grouped_data[$key] = [
                    'supplier_name' => $supplier_name,
                    'date' => $line->transaction_date,
                    'items' => [],
                    'subtotal_qty' => 0,
                    'subtotal_amount' => 0
                ];
            }
            
            $invoice_no = !empty($line->invoice_no) ? $line->invoice_no : (!empty($line->ref_no) ? $line->ref_no : '');
            
            $grouped_data[$key]['items'][] = [
                'invoice_no' => $invoice_no,
                'product_name' => $line->product_name,
                'qty' => $line->quantity,
                'unit_price' => $line->unit_price,
                'discount' => $line->discount,
                'net_unit_price' => $line->net_unit_price,
                'amount' => $line->amount
            ];
            
            $grouped_data[$key]['subtotal_qty'] += $line->quantity;
            $grouped_data[$key]['subtotal_amount'] += $line->amount;
            $grand_total_qty += $line->quantity;
            $grand_total_amount += $line->amount;
        }

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::purchase_register.print_supplier_wise_purchase_return', compact(
            'grouped_data',
            'business',
            'logo',
            'start_date',
            'end_date',
            'util',
            'currency',
            'grand_total_qty',
            'grand_total_amount'
        ));
    }

    /**
     * Date wise Purchase Return Report
     *
     * @return \Illuminate\Http\Response
     */
    public function dateWisePurchaseReturn(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            // Parse date range
            $start_date = null;
            $end_date = null;
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $start_date = $request->start_date;
                $end_date = $request->end_date;
            }

            $location_id = $request->location_id ?? null;

            // Get purchase return transactions
            $return_transactions = \App\Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->whereIn('status', ['received', 'final']);

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $return_transactions->whereIn('location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                $return_transactions->where('location_id', $location_id);
            }

            if (!empty($start_date) && !empty($end_date)) {
                $return_transactions->whereDate('transaction_date', '>=', $start_date)
                    ->whereDate('transaction_date', '<=', $end_date);
            }

            $return_transactions = $return_transactions->orderBy('transaction_date', 'asc')->get();

            $return_data = [];

            foreach ($return_transactions as $return_transaction) {
                $supplier_name = '';
                $total_before_tax = 0;
                $discount_amount = 0;
                $shipping_charges = 0;
                $final_total = 0;

                // Get supplier info
                if (!empty($return_transaction->contact_id)) {
                    $supplier = \App\Contact::find($return_transaction->contact_id);
                    if ($supplier) {
                        $supplier_name = $supplier->name;
                        if (!empty($supplier->supplier_business_name)) {
                            $supplier_name .= ' (' . $supplier->supplier_business_name . ')';
                        }
                    }
                }

                // Get transaction totals
                $total_before_tax = $return_transaction->total_before_tax ?? 0;
                $discount_amount = $return_transaction->discount_amount ?? 0;
                $shipping_charges = $return_transaction->shipping_charges ?? 0;
                $final_total = $return_transaction->final_total ?? 0;

                $invoice_no = !empty($return_transaction->invoice_no) ? $return_transaction->invoice_no : (!empty($return_transaction->ref_no) ? $return_transaction->ref_no : '');

                $return_data[] = [
                    'transaction_date' => $return_transaction->transaction_date,
                    'invoice_no' => $invoice_no,
                    'supplier_name' => $supplier_name,
                    'narration' => $return_transaction->additional_notes ?? '',
                    'total_amount' => $total_before_tax,
                    'discount' => $discount_amount,
                    'ex_charges' => $shipping_charges,
                    'amount' => $final_total
                ];
            }

            // Convert to collection for DataTables
            $collection = collect($return_data);

            $util = new Util();
            
            return DataTables::of($collection)
                ->editColumn('transaction_date', function ($row) use ($util) {
                    return $util->format_date($row['transaction_date'], true);
                })
                ->addColumn('total_amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['total_amount'] . '</span>';
                })
                ->addColumn('discount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['discount'] . '</span>';
                })
                ->addColumn('ex_charges', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['ex_charges'] . '</span>';
                })
                ->addColumn('amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['amount'] . '</span>';
                })
                ->rawColumns(['total_amount', 'discount', 'ex_charges', 'amount'])
                ->make(true);
        }

        return view('businessmanagement::purchase_register.date_wise_purchase_return', compact('locations'));
    }

    /**
     * Print Date wise Purchase Return Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printDateWisePurchaseReturn(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get invoice layout logo
        $logo = false;
        $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
            ->where('is_default', 1)
            ->first();
        if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
            $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
        }

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = explode(' ~ ', $request->date_range);
            if (count($date_range) != 2) {
                $date_range = explode(' - ', $request->date_range);
            }
            if (count($date_range) == 2) {
                $start_date_raw = trim($date_range[0]);
                $end_date_raw = trim($date_range[1]);
                
                try {
                    $start_date = \Carbon::createFromFormat('m/d/Y', $start_date_raw)->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon::parse($start_date_raw)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $start_date = $start_date_raw;
                    }
                }
                
                try {
                    $end_date = \Carbon::createFromFormat('m/d/Y', $end_date_raw)->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $end_date = \Carbon::parse($end_date_raw)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $end_date = $end_date_raw;
                    }
                }
            }
        }

        $location_id = $request->location_id ?? null;

        // Get purchase return transactions
        $return_transactions = \App\Transaction::where('business_id', $business_id)
            ->where('type', 'purchase_return')
            ->whereIn('status', ['received', 'final']);

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $return_transactions->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $return_transactions->where('location_id', $location_id);
        }

        if (!empty($start_date) && !empty($end_date)) {
            $return_transactions->whereDate('transaction_date', '>=', $start_date)
                ->whereDate('transaction_date', '<=', $end_date);
        }

        $return_transactions = $return_transactions->orderBy('transaction_date', 'asc')->get();

        $return_data = [];

        foreach ($return_transactions as $return_transaction) {
            $supplier_name = '';
            $total_before_tax = 0;
            $tax_amount = 0;
            $discount_amount = 0;
            $shipping_charges = 0;
            $final_total = 0;

            // Get supplier info
            if (!empty($return_transaction->contact_id)) {
                $supplier = \App\Contact::find($return_transaction->contact_id);
                if ($supplier) {
                    $supplier_name = $supplier->name;
                    if (!empty($supplier->supplier_business_name)) {
                        $supplier_name .= ' (' . $supplier->supplier_business_name . ')';
                    }
                }
            }

            // Get transaction totals
            $total_before_tax = $return_transaction->total_before_tax ?? 0;
            $tax_amount = $return_transaction->tax_amount ?? 0;
            $discount_amount = $return_transaction->discount_amount ?? 0;
            $shipping_charges = $return_transaction->shipping_charges ?? 0;
            $final_total = $return_transaction->final_total ?? 0;

            $invoice_no = !empty($return_transaction->invoice_no) ? $return_transaction->invoice_no : (!empty($return_transaction->ref_no) ? $return_transaction->ref_no : '');

            $return_data[] = [
                'transaction_date' => $return_transaction->transaction_date,
                'invoice_no' => $invoice_no,
                'supplier_name' => $supplier_name,
                'narration' => $return_transaction->additional_notes ?? '',
                'total_amount' => $total_before_tax,
                'discount' => $discount_amount,
                'ex_charges' => $shipping_charges,
                'amount' => $final_total
            ];
        }

        // Group data by date
        $grouped_data = [];
        $grand_total_amount = 0;
        $grand_total_discount = 0;
        $grand_total_ex_charges = 0;
        $grand_total_final = 0;

        foreach ($return_data as $item) {
            $date_key = date('Y-m-d', strtotime($item['transaction_date']));
            
            if (!isset($grouped_data[$date_key])) {
                $grouped_data[$date_key] = [
                    'date' => $item['transaction_date'],
                    'items' => [],
                    'subtotal_amount' => 0,
                    'subtotal_discount' => 0,
                    'subtotal_ex_charges' => 0,
                    'subtotal_final' => 0
                ];
            }
            
            $grouped_data[$date_key]['items'][] = $item;
            $grouped_data[$date_key]['subtotal_amount'] += $item['total_amount'];
            $grouped_data[$date_key]['subtotal_discount'] += $item['discount'];
            $grouped_data[$date_key]['subtotal_ex_charges'] += $item['ex_charges'];
            $grouped_data[$date_key]['subtotal_final'] += $item['amount'];
            
            $grand_total_amount += $item['total_amount'];
            $grand_total_discount += $item['discount'];
            $grand_total_ex_charges += $item['ex_charges'];
            $grand_total_final += $item['amount'];
        }

        // Sort by date
        ksort($grouped_data);

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::purchase_register.print_date_wise_purchase_return', compact(
            'grouped_data',
            'business',
            'logo',
            'start_date',
            'end_date',
            'util',
            'currency',
            'grand_total_amount',
            'grand_total_discount',
            'grand_total_ex_charges',
            'grand_total_final'
        ));
    }

    /**
     * Product wise Purchase Return Report
     *
     * @return \Illuminate\Http\Response
     */
    public function productWisePurchaseReturn(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $categories = \App\Category::forDropdown($business_id, false, true);
        $locations = BusinessLocation::forDropdown($business_id, true);
        
        // Get products for dropdown
        $products = \App\Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get();
        
        $products_array = [];
        foreach ($products as $product) {
            $products_array[$product->id] = $product->name . ($product->sku ? ' (' . $product->sku . ')' : '');
        }

        if ($request->ajax()) {
            // Parse date range
            $start_date = null;
            $end_date = null;
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $start_date = $request->start_date;
                $end_date = $request->end_date;
            }

            $product_filter = $request->product_filter ?? 'all';
            $category_id = ($product_filter == 'category' && !empty($request->category_id)) ? $request->category_id : null;
            $product_id = ($product_filter == 'single' && !empty($request->product_id)) ? $request->product_id : null;
            $location_id = $request->location_id ?? null;

            // Get purchase return transactions
            $return_transactions = \App\Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_return')
                ->whereIn('status', ['received', 'final']);

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $return_transactions->whereIn('location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                $return_transactions->where('location_id', $location_id);
            }

            if (!empty($start_date) && !empty($end_date)) {
                $return_transactions->whereDate('transaction_date', '>=', $start_date)
                    ->whereDate('transaction_date', '<=', $end_date);
            }

            $return_transactions = $return_transactions->get();

            $return_data = [];

            foreach ($return_transactions as $return_transaction) {
                // Check if this is a regular return (has return_parent_id) or combined return
                if (!empty($return_transaction->return_parent_id)) {
                    // Regular return - get original purchase lines with quantity_returned > 0
                    $purchase_lines = PurchaseLine::join('transactions as pt', 'purchase_lines.transaction_id', '=', 'pt.id')
                        ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                        ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                        ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                        ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                        ->leftJoin('contacts as c', 'pt.contact_id', '=', 'c.id')
                        ->where('pt.id', $return_transaction->return_parent_id)
                        ->where('purchase_lines.quantity_returned', '>', 0)
                        ->select(
                            'p.id as product_id',
                            DB::raw("CASE 
                                WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                                ELSE COALESCE(p.name, 'N/A')
                            END as product_name"),
                            'cat.id as category_id',
                            'cat.name as category_name',
                            'purchase_lines.quantity_returned as quantity',
                            'purchase_lines.purchase_price_inc_tax as unit_price',
                            'purchase_lines.purchase_price_inc_tax as net_unit_price',
                            DB::raw('(purchase_lines.quantity_returned * purchase_lines.purchase_price_inc_tax) as amount')
                        );

                    // Apply category filter
                    if (!empty($category_id)) {
                        $purchase_lines->where('cat.id', $category_id);
                    }

                    // Apply product filter
                    if (!empty($product_id)) {
                        $purchase_lines->where('p.id', $product_id);
                    }

                    $lines = $purchase_lines->get();
                    foreach ($lines as $line) {
                        $supplier_name = '';
                        if (!empty($return_transaction->contact_id)) {
                            $supplier = \App\Contact::find($return_transaction->contact_id);
                            if ($supplier) {
                                $supplier_name = $supplier->name;
                                if (!empty($supplier->supplier_business_name)) {
                                    $supplier_name .= ' (' . $supplier->supplier_business_name . ')';
                                }
                            }
                        }

                        $invoice_no = !empty($return_transaction->invoice_no) ? $return_transaction->invoice_no : (!empty($return_transaction->ref_no) ? $return_transaction->ref_no : '');

                        $return_data[] = [
                            'product_name' => $line->product_name,
                            'invoice_no' => $invoice_no,
                            'supplier_name' => $supplier_name,
                            'quantity' => $line->quantity,
                            'unit_price' => $line->unit_price,
                            'net_unit_price' => $line->net_unit_price,
                            'amount' => $line->amount
                        ];
                    }
                } else {
                    // Combined return - get purchase_lines directly from return transaction
                    $purchase_lines = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                        ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                        ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                        ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                        ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                        ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                        ->where('t.id', $return_transaction->id)
                        ->where('purchase_lines.quantity_returned', '>', 0)
                        ->select(
                            'p.id as product_id',
                            DB::raw("CASE 
                                WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                                ELSE COALESCE(p.name, 'N/A')
                            END as product_name"),
                            'cat.id as category_id',
                            'cat.name as category_name',
                            't.invoice_no',
                            't.ref_no',
                            'c.name as supplier_name',
                            'c.supplier_business_name',
                            DB::raw('purchase_lines.quantity_returned as quantity'),
                            'purchase_lines.purchase_price_inc_tax as unit_price',
                            'purchase_lines.purchase_price_inc_tax as net_unit_price',
                            DB::raw('(purchase_lines.quantity_returned * purchase_lines.purchase_price_inc_tax) as amount')
                        );

                    // Apply category filter
                    if (!empty($category_id)) {
                        $purchase_lines->where('cat.id', $category_id);
                    }

                    // Apply product filter
                    if (!empty($product_id)) {
                        $purchase_lines->where('p.id', $product_id);
                    }

                    $lines = $purchase_lines->get();
                    foreach ($lines as $line) {
                        $supplier_name = $line->supplier_name ?? '';
                        if (!empty($line->supplier_business_name)) {
                            $supplier_name .= ' (' . $line->supplier_business_name . ')';
                        }

                        $invoice_no = !empty($line->invoice_no) ? $line->invoice_no : (!empty($line->ref_no) ? $line->ref_no : '');

                        $return_data[] = [
                            'product_name' => $line->product_name,
                            'invoice_no' => $invoice_no,
                            'supplier_name' => $supplier_name,
                            'quantity' => $line->quantity,
                            'unit_price' => $line->unit_price,
                            'net_unit_price' => $line->net_unit_price,
                            'amount' => $line->amount
                        ];
                    }
                }
            }

            // Convert to collection for DataTables
            $collection = collect($return_data);

            $util = new Util();
            
            return DataTables::of($collection)
                ->addColumn('unit_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['unit_price'] . '</span>';
                })
                ->addColumn('net_unit_price', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['net_unit_price'] . '</span>';
                })
                ->addColumn('amount', function ($row) {
                    return '<span class="display_currency" data-currency_symbol="true">' . $row['amount'] . '</span>';
                })
                ->rawColumns(['unit_price', 'net_unit_price', 'amount'])
                ->make(true);
        }

        return view('businessmanagement::purchase_register.product_wise_purchase_return', compact('categories', 'locations', 'products_array'));
    }

    /**
     * Print Product wise Purchase Return Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printProductWisePurchaseReturn(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get invoice layout logo
        $logo = false;
        $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
            ->where('is_default', 1)
            ->first();
        if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
            $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
        }

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = explode(' ~ ', $request->date_range);
            if (count($date_range) != 2) {
                $date_range = explode(' - ', $request->date_range);
            }
            if (count($date_range) == 2) {
                $start_date_raw = trim($date_range[0]);
                $end_date_raw = trim($date_range[1]);
                
                try {
                    $start_date = \Carbon::createFromFormat('m/d/Y', $start_date_raw)->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon::parse($start_date_raw)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $start_date = $start_date_raw;
                    }
                }
                
                try {
                    $end_date = \Carbon::createFromFormat('m/d/Y', $end_date_raw)->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $end_date = \Carbon::parse($end_date_raw)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $end_date = $end_date_raw;
                    }
                }
            }
        }

        $product_filter = $request->product_filter ?? 'all';
        $category_id = ($product_filter == 'category' && !empty($request->category_id)) ? $request->category_id : null;
        $product_id = ($product_filter == 'single' && !empty($request->product_id)) ? $request->product_id : null;
        $location_id = $request->location_id ?? null;

        // Get purchase return transactions
        $return_transactions = \App\Transaction::where('business_id', $business_id)
            ->where('type', 'purchase_return')
            ->whereIn('status', ['received', 'final']);

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $return_transactions->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $return_transactions->where('location_id', $location_id);
        }

        if (!empty($start_date) && !empty($end_date)) {
            $return_transactions->whereDate('transaction_date', '>=', $start_date)
                ->whereDate('transaction_date', '<=', $end_date);
        }

        $return_transactions = $return_transactions->get();

        $return_data = [];

        foreach ($return_transactions as $return_transaction) {
            // Check if this is a regular return (has return_parent_id) or combined return
            if (!empty($return_transaction->return_parent_id)) {
                // Regular return - get original purchase lines with quantity_returned > 0
                $purchase_lines = PurchaseLine::join('transactions as pt', 'purchase_lines.transaction_id', '=', 'pt.id')
                    ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                    ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                    ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                    ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                    ->leftJoin('contacts as c', 'pt.contact_id', '=', 'c.id')
                    ->where('pt.id', $return_transaction->return_parent_id)
                    ->where('purchase_lines.quantity_returned', '>', 0)
                    ->select(
                        'p.id as product_id',
                        DB::raw("CASE 
                            WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                            ELSE COALESCE(p.name, 'N/A')
                        END as product_name"),
                        'cat.id as category_id',
                        'cat.name as category_name',
                        'purchase_lines.quantity_returned as quantity',
                        'purchase_lines.purchase_price_inc_tax as unit_price',
                        'purchase_lines.purchase_price_inc_tax as net_unit_price',
                        DB::raw('(purchase_lines.quantity_returned * purchase_lines.purchase_price_inc_tax) as amount')
                    );

                    // Apply category filter
                    if (!empty($category_id)) {
                        $purchase_lines->where('cat.id', $category_id);
                    }

                    // Apply product filter
                    if (!empty($product_id)) {
                        $purchase_lines->where('p.id', $product_id);
                    }

                $lines = $purchase_lines->get();
                foreach ($lines as $line) {
                    $supplier_name = '';
                    if (!empty($return_transaction->contact_id)) {
                        $supplier = \App\Contact::find($return_transaction->contact_id);
                        if ($supplier) {
                            $supplier_name = $supplier->name;
                            if (!empty($supplier->supplier_business_name)) {
                                $supplier_name .= ' (' . $supplier->supplier_business_name . ')';
                            }
                        }
                    }

                    $invoice_no = !empty($return_transaction->invoice_no) ? $return_transaction->invoice_no : (!empty($return_transaction->ref_no) ? $return_transaction->ref_no : '');

                    $return_data[] = [
                        'product_id' => $line->product_id,
                        'product_name' => $line->product_name,
                        'category_id' => $line->category_id,
                        'category_name' => $line->category_name,
                        'transaction_date' => $return_transaction->transaction_date,
                        'invoice_no' => $invoice_no,
                        'supplier_name' => $supplier_name,
                        'quantity' => $line->quantity,
                        'unit_price' => $line->unit_price,
                        'net_unit_price' => $line->net_unit_price,
                        'amount' => $line->amount
                    ];
                }
            } else {
                // Combined return - get purchase_lines directly from return transaction
                $purchase_lines = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                    ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                    ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                    ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                    ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                    ->leftJoin('contacts as c', 't.contact_id', '=', 'c.id')
                    ->where('t.id', $return_transaction->id)
                    ->where('purchase_lines.quantity_returned', '>', 0)
                    ->select(
                        'p.id as product_id',
                        DB::raw("CASE 
                            WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                            ELSE COALESCE(p.name, 'N/A')
                        END as product_name"),
                        'cat.id as category_id',
                        'cat.name as category_name',
                        't.transaction_date',
                        't.invoice_no',
                        't.ref_no',
                        'c.name as supplier_name',
                        'c.supplier_business_name',
                        DB::raw('purchase_lines.quantity_returned as quantity'),
                        'purchase_lines.purchase_price_inc_tax as unit_price',
                        'purchase_lines.purchase_price_inc_tax as net_unit_price',
                        DB::raw('(purchase_lines.quantity_returned * purchase_lines.purchase_price_inc_tax) as amount')
                    );

                // Apply category filter
                if (!empty($category_id)) {
                    $purchase_lines->where('cat.id', $category_id);
                }

                // Apply product filter
                if (!empty($product_id)) {
                    $purchase_lines->where('p.id', $product_id);
                }

                $lines = $purchase_lines->get();
                foreach ($lines as $line) {
                    $supplier_name = $line->supplier_name ?? '';
                    if (!empty($line->supplier_business_name)) {
                        $supplier_name .= ' (' . $line->supplier_business_name . ')';
                    }

                    $invoice_no = !empty($line->invoice_no) ? $line->invoice_no : (!empty($line->ref_no) ? $line->ref_no : '');

                    $return_data[] = [
                        'product_id' => $line->product_id,
                        'product_name' => $line->product_name,
                        'category_id' => $line->category_id,
                        'category_name' => $line->category_name,
                        'transaction_date' => $line->transaction_date,
                        'invoice_no' => $invoice_no,
                        'supplier_name' => $supplier_name,
                        'quantity' => $line->quantity,
                        'unit_price' => $line->unit_price,
                        'net_unit_price' => $line->net_unit_price,
                        'amount' => $line->amount
                    ];
                }
            }
        }

        // Group data by product
        $grouped_data = [];
        $grand_total_qty = 0;
        $grand_total_amount = 0;

        foreach ($return_data as $item) {
            $product_key = $item['product_id'];
            
            if (!isset($grouped_data[$product_key])) {
                $grouped_data[$product_key] = [
                    'product_name' => $item['product_name'],
                    'items' => [],
                    'subtotal_qty' => 0,
                    'subtotal_amount' => 0
                ];
            }
            
            $grouped_data[$product_key]['items'][] = $item;
            $grouped_data[$product_key]['subtotal_qty'] += $item['quantity'];
            $grouped_data[$product_key]['subtotal_amount'] += $item['amount'];
            
            $grand_total_qty += $item['quantity'];
            $grand_total_amount += $item['amount'];
        }

        // Sort by product name
        uasort($grouped_data, function($a, $b) {
            return strcmp($a['product_name'], $b['product_name']);
        });

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::purchase_register.print_product_wise_purchase_return', compact(
            'grouped_data',
            'business',
            'logo',
            'start_date',
            'end_date',
            'util',
            'currency',
            'grand_total_qty',
            'grand_total_amount'
        ));
    }

    /**
     * Date wise Purchase Order Report
     *
     * @return \Illuminate\Http\Response
     */
    public function dateWisePurchaseOrder(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);

        if ($request->ajax()) {
            // Parse date range
            $start_date = null;
            $end_date = null;
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $start_date = $request->start_date;
                $end_date = $request->end_date;
            }

            $location_id = $request->location_id ?? null;

            // Get purchase order transactions
            $purchase_orders = \App\Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_order');

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $purchase_orders->whereIn('location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                $purchase_orders->where('location_id', $location_id);
            }

            if (!empty($start_date) && !empty($end_date)) {
                $purchase_orders->whereDate('transaction_date', '>=', $start_date)
                    ->whereDate('transaction_date', '<=', $end_date);
            }

            $purchase_orders = $purchase_orders->orderBy('transaction_date', 'asc')->get();

            $order_data = [];

            foreach ($purchase_orders as $order) {
                // Get requisition numbers
                $requisition_nos = [];
                if (!empty($order->purchase_requisition_ids) && is_array($order->purchase_requisition_ids)) {
                    $requisitions = \App\Transaction::whereIn('id', $order->purchase_requisition_ids)
                        ->where('type', 'purchase_requisition')
                        ->pluck('ref_no')
                        ->toArray();
                    $requisition_nos = $requisitions;
                }
                $requisition_no = !empty($requisition_nos) ? implode(', ', $requisition_nos) : '';

                // Get purchase lines
                $purchase_lines = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                    ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                    ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                    ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                    ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                    ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
                    ->where('t.id', $order->id)
                    ->select(
                        'purchase_lines.id',
                        'purchase_lines.quantity',
                        'purchase_lines.transaction_id',
                        DB::raw("CASE 
                            WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                            ELSE COALESCE(p.name, 'N/A')
                        END as product_description"),
                        'cat.name as product_group',
                        'u.short_name as unit_name',
                        't.ref_no as order_no',
                        't.transaction_date',
                        't.invoice_no'
                    )
                    ->get();

                foreach ($purchase_lines as $line) {
                    $order_data[] = [
                        'transaction_date' => $order->transaction_date,
                        'order_no' => $order->ref_no,
                        'requisition_no' => $requisition_no,
                        'product_group' => $line->product_group ?? 'UNCATEGORIZED',
                        'product_description' => $line->product_description,
                        'quantity' => $line->quantity,
                        'unit' => $line->unit_name ?? ''
                    ];
                }
            }

            // Convert to collection for DataTables
            $collection = collect($order_data);

            return DataTables::of($collection)
                ->addColumn('quantity', function ($row) {
                    return number_format($row['quantity'], 2);
                })
                ->make(true);
        }

        return view('businessmanagement::purchase_register.date_wise_purchase_order', compact('locations'));
    }

    /**
     * Print Date wise Purchase Order Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printDateWisePurchaseOrder(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get invoice layout logo
        $logo = false;
        $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
            ->where('is_default', 1)
            ->first();
        if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
            $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
        }

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = explode(' ~ ', $request->date_range);
            if (count($date_range) != 2) {
                $date_range = explode(' - ', $request->date_range);
            }
            if (count($date_range) == 2) {
                $start_date_raw = trim($date_range[0]);
                $end_date_raw = trim($date_range[1]);
                
                try {
                    $start_date = \Carbon::createFromFormat('m/d/Y', $start_date_raw)->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon::parse($start_date_raw)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $start_date = $start_date_raw;
                    }
                }
                
                try {
                    $end_date = \Carbon::createFromFormat('m/d/Y', $end_date_raw)->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $end_date = \Carbon::parse($end_date_raw)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $end_date = $end_date_raw;
                    }
                }
            }
        }

        $location_id = $request->location_id ?? null;

        // Get purchase order transactions
        $purchase_orders = \App\Transaction::where('business_id', $business_id)
            ->where('type', 'purchase_order');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $purchase_orders->whereIn('location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $purchase_orders->where('location_id', $location_id);
        }

        if (!empty($start_date) && !empty($end_date)) {
            $purchase_orders->whereDate('transaction_date', '>=', $start_date)
                ->whereDate('transaction_date', '<=', $end_date);
        }

        $purchase_orders = $purchase_orders->orderBy('transaction_date', 'asc')->get();

        $order_data = [];

        foreach ($purchase_orders as $order) {
            // Get requisition numbers
            $requisition_nos = [];
            if (!empty($order->purchase_requisition_ids) && is_array($order->purchase_requisition_ids)) {
                $requisitions = \App\Transaction::whereIn('id', $order->purchase_requisition_ids)
                    ->where('type', 'purchase_requisition')
                    ->pluck('ref_no')
                    ->toArray();
                $requisition_nos = $requisitions;
            }
            $requisition_no = !empty($requisition_nos) ? implode(', ', $requisition_nos) : '';

            // Get purchase lines
            $purchase_lines = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
                ->where('t.id', $order->id)
                ->select(
                    'purchase_lines.id',
                    'purchase_lines.quantity',
                    'purchase_lines.transaction_id',
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_description"),
                    'cat.name as product_group',
                    'u.short_name as unit_name',
                    't.ref_no as order_no',
                    't.transaction_date',
                    't.invoice_no'
                )
                ->get();

            foreach ($purchase_lines as $line) {
                $order_data[] = [
                    'transaction_date' => $order->transaction_date,
                    'order_no' => $order->ref_no,
                    'requisition_no' => $requisition_no,
                    'product_group' => $line->product_group ?? 'UNCATEGORIZED',
                    'product_description' => $line->product_description,
                    'quantity' => $line->quantity,
                    'unit' => $line->unit_name ?? '',
                    'invoice_no' => $order->invoice_no ?? ''
                ];
            }
        }

        // Group data by date
        $grouped_data = [];
        $grand_total_qty = 0;

        foreach ($order_data as $item) {
            $date_key = date('Y-m-d', strtotime($item['transaction_date']));
            
            if (!isset($grouped_data[$date_key])) {
                $grouped_data[$date_key] = [
                    'date' => $item['transaction_date'],
                    'items' => [],
                    'invoice_wise_total_qty' => 0,
                    'date_wise_total_qty' => 0
                ];
            }
            
            $grouped_data[$date_key]['items'][] = $item;
            $grouped_data[$date_key]['date_wise_total_qty'] += $item['quantity'];
            
            // Count invoice wise total (only items with invoice_no)
            if (!empty($item['invoice_no'])) {
                $grouped_data[$date_key]['invoice_wise_total_qty'] += $item['quantity'];
            }
            
            $grand_total_qty += $item['quantity'];
        }

        // Sort by date
        ksort($grouped_data);

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::purchase_register.print_date_wise_purchase_order', compact(
            'grouped_data',
            'business',
            'logo',
            'start_date',
            'end_date',
            'util',
            'currency',
            'grand_total_qty'
        ));
    }

    /**
     * Supplier wise Purchase Order Report
     *
     * @return \Illuminate\Http\Response
     */
    public function supplierWisePurchaseOrder(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $locations = BusinessLocation::forDropdown($business_id, true);
        
        // Get suppliers for dropdown
        $suppliers = \App\Contact::where('business_id', $business_id)
            ->where('type', 'supplier')
            ->select('id', 'name', 'supplier_business_name')
            ->orderBy('name', 'asc')
            ->get();
        
        $suppliers_array = [];
        foreach ($suppliers as $supplier) {
            $name = $supplier->name;
            if (!empty($supplier->supplier_business_name)) {
                $name .= ' (' . $supplier->supplier_business_name . ')';
            }
            $suppliers_array[$supplier->id] = $name;
        }
        
        // Get categories for dropdown
        $categories = \App\Category::forDropdown($business_id, false, true);
        
        // Get products for dropdown
        $products = \App\Product::where('business_id', $business_id)
            ->where('type', '!=', 'modifier')
            ->select('id', 'name', 'sku')
            ->orderBy('name', 'asc')
            ->get();
        
        $products_array = [];
        foreach ($products as $product) {
            $products_array[$product->id] = $product->name . ($product->sku ? ' (' . $product->sku . ')' : '');
        }

        if ($request->ajax()) {
            // Parse date range
            $start_date = null;
            $end_date = null;
            if (!empty($request->start_date) && !empty($request->end_date)) {
                $start_date = $request->start_date;
                $end_date = $request->end_date;
            }

            $supplier_filter = $request->supplier_filter ?? 'all';
            $supplier_id = ($supplier_filter == 'single' && !empty($request->supplier_id)) ? $request->supplier_id : null;
            
            $product_filter = $request->product_filter ?? 'all';
            $product_id = ($product_filter == 'single' && !empty($request->product_id)) ? $request->product_id : null;
            $category_id = ($product_filter == 'category' && !empty($request->category_id)) ? $request->category_id : null;
            
            $location_id = $request->location_id ?? null;

            // Get purchase order transactions using DB query builder to avoid model scope issues
            $purchase_orders_query = DB::table('transactions')
                ->leftJoin('contacts as c', 'transactions.contact_id', '=', 'c.id')
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'purchase_order');

            // Apply location filter
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $purchase_orders_query->whereIn('transactions.location_id', $permitted_locations);
            }

            if (!empty($location_id)) {
                $purchase_orders_query->where('transactions.location_id', $location_id);
            }

            // Apply supplier filter
            if (!empty($supplier_id)) {
                $purchase_orders_query->where('c.id', $supplier_id);
            }

            if (!empty($start_date) && !empty($end_date)) {
                $purchase_orders_query->whereDate('transactions.transaction_date', '>=', $start_date)
                    ->whereDate('transactions.transaction_date', '<=', $end_date);
            }

            $purchase_orders = $purchase_orders_query->select('transactions.id', 'transactions.ref_no', 'transactions.transaction_date', 'transactions.purchase_requisition_ids', 'c.name as supplier_name', 'c.supplier_business_name')
                ->orderBy('transactions.transaction_date', 'asc')
                ->get();

            $order_data = [];

            foreach ($purchase_orders as $order) {
                try {
                    $order_id = $order->id ?? null;
                    if (!$order_id) {
                        continue;
                    }
                    
                    // Get requisition numbers - handle JSON string from DB query builder
                    $requisition_nos = [];
                    if (!empty($order->purchase_requisition_ids)) {
                        $requisition_ids = $order->purchase_requisition_ids;
                        // Handle if it's a JSON string (DB query builder returns strings)
                        if (is_string($requisition_ids)) {
                            $requisition_ids = json_decode($requisition_ids, true);
                        }
                        if (is_array($requisition_ids) && !empty($requisition_ids)) {
                            try {
                                $requisitions = DB::table('transactions')
                                    ->whereIn('id', $requisition_ids)
                                    ->where('type', 'purchase_requisition')
                                    ->pluck('ref_no')
                                    ->toArray();
                                $requisition_nos = $requisitions;
                            } catch (\Exception $e) {
                                \Log::error('Error fetching requisitions: ' . $e->getMessage());
                            }
                        }
                    }
                    $requisition_no = !empty($requisition_nos) ? implode(', ', $requisition_nos) : '';

                    // Get supplier name
                    $supplier_name = $order->supplier_name ?? '';
                    if (!empty($order->supplier_business_name)) {
                        $supplier_name .= ' (' . $order->supplier_business_name . ')';
                    }

                    // Get purchase lines
                    $purchase_lines = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                        ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                        ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                        ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                        ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                        ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
                        ->where('t.id', $order_id)
                        ->select(
                            'purchase_lines.id',
                            'purchase_lines.quantity',
                            'purchase_lines.transaction_id',
                            'p.id as product_id',
                            DB::raw("CASE 
                                WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                                ELSE COALESCE(p.name, 'N/A')
                            END as product_description"),
                            'cat.id as category_id',
                            'cat.name as product_group',
                            'u.short_name as unit_name',
                            't.ref_no as order_no',
                            't.transaction_date',
                            't.invoice_no'
                        );

                    // Apply product filter
                    if (!empty($product_id)) {
                        $purchase_lines->where('p.id', $product_id);
                    }

                    // Apply category filter
                    if (!empty($category_id)) {
                        $purchase_lines->where('cat.id', $category_id);
                    }

                    $lines = $purchase_lines->get();

                    // Format transaction date
                    $transaction_date = $order->transaction_date ?? '';
                    if ($transaction_date && is_string($transaction_date)) {
                        try {
                            $transaction_date = \Carbon\Carbon::parse($transaction_date)->format('Y-m-d');
                        } catch (\Exception $e) {
                            $transaction_date = '';
                        }
                    }

                    foreach ($lines as $line) {
                        $order_data[] = [
                            'transaction_date' => $transaction_date,
                            'order_no' => $order->ref_no ?? '',
                            'supplier_name' => $supplier_name,
                            'requisition_no' => $requisition_no,
                            'product_group' => $line->product_group ?? 'UNCATEGORIZED',
                            'product_description' => $line->product_description ?? 'N/A',
                            'quantity' => $line->quantity ?? 0,
                            'unit' => $line->unit_name ?? ''
                        ];
                    }
                } catch (\Exception $e) {
                    // Log error and continue
                    \Log::error('Error processing purchase order ID ' . ($order->id ?? 'unknown') . ': ' . $e->getMessage() . ' | Trace: ' . $e->getTraceAsString());
                    continue;
                }
            }

            // Convert to collection for DataTables
            $collection = collect($order_data);

            try {
                $datatable = DataTables::of($collection)
                    ->addColumn('quantity', function ($row) {
                        $qty = $row['quantity'] ?? 0;
                        return is_numeric($qty) ? number_format($qty, 2) : '0.00';
                    })
                    ->rawColumns(['quantity']);
                
                return $datatable->make(true);
            } catch (\Exception $e) {
                \Log::error('DataTables error in supplierWisePurchaseOrder: ' . $e->getMessage());
                \Log::error('Stack trace: ' . $e->getTraceAsString());
                \Log::error('Request data: ' . json_encode($request->all()));
                
                return response()->json([
                    'draw' => (int) $request->get('draw', 1),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'error' => 'An error occurred while processing the request. Please check the logs.'
                ], 200);
            }
        }

        return view('businessmanagement::purchase_register.supplier_wise_purchase_order', compact('locations', 'suppliers_array', 'categories', 'products_array'));
    }

    /**
     * Print Supplier wise Purchase Order Report
     *
     * @return \Illuminate\Http\Response
     */
    public function printSupplierWisePurchaseOrder(Request $request)
    {
        if (!auth()->user()->can('businessmanagement.purchase_register')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $business = \App\Business::find($business_id);
        
        // Get invoice layout logo
        $logo = false;
        $invoice_layout = \App\InvoiceLayout::where('business_id', $business_id)
            ->where('is_default', 1)
            ->first();
        if ($invoice_layout && $invoice_layout->show_logo != 0 && !empty($invoice_layout->logo) && file_exists(public_path('uploads/invoice_logos/'.$invoice_layout->logo))) {
            $logo = asset('uploads/invoice_logos/'.$invoice_layout->logo);
        }

        // Parse date range
        $start_date = null;
        $end_date = null;
        if (!empty($request->date_range)) {
            $date_range = explode(' ~ ', $request->date_range);
            if (count($date_range) != 2) {
                $date_range = explode(' - ', $request->date_range);
            }
            if (count($date_range) == 2) {
                $start_date_raw = trim($date_range[0]);
                $end_date_raw = trim($date_range[1]);
                
                try {
                    $start_date = \Carbon::createFromFormat('m/d/Y', $start_date_raw)->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $start_date = \Carbon::parse($start_date_raw)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $start_date = $start_date_raw;
                    }
                }
                
                try {
                    $end_date = \Carbon::createFromFormat('m/d/Y', $end_date_raw)->format('Y-m-d');
                } catch (\Exception $e) {
                    try {
                        $end_date = \Carbon::parse($end_date_raw)->format('Y-m-d');
                    } catch (\Exception $e2) {
                        $end_date = $end_date_raw;
                    }
                }
            }
        }

        $supplier_filter = $request->supplier_filter ?? 'all';
        $supplier_id = ($supplier_filter == 'single' && !empty($request->supplier_id)) ? $request->supplier_id : null;
        
        $product_filter = $request->product_filter ?? 'all';
        $product_id = ($product_filter == 'single' && !empty($request->product_id)) ? $request->product_id : null;
        $category_id = ($product_filter == 'category' && !empty($request->category_id)) ? $request->category_id : null;
        
        $location_id = $request->location_id ?? null;

        // Get purchase order transactions using DB query builder to avoid model scope issues
        $purchase_orders_query = DB::table('transactions')
            ->leftJoin('contacts as c', 'transactions.contact_id', '=', 'c.id')
            ->where('transactions.business_id', $business_id)
            ->where('transactions.type', 'purchase_order');

        // Apply location filter
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $purchase_orders_query->whereIn('transactions.location_id', $permitted_locations);
        }

        if (!empty($location_id)) {
            $purchase_orders_query->where('transactions.location_id', $location_id);
        }

        // Apply supplier filter
        if (!empty($supplier_id)) {
            $purchase_orders_query->where('c.id', $supplier_id);
        }

        if (!empty($start_date) && !empty($end_date)) {
            $purchase_orders_query->whereDate('transactions.transaction_date', '>=', $start_date)
                ->whereDate('transactions.transaction_date', '<=', $end_date);
        }

        $purchase_orders = $purchase_orders_query->select('transactions.*', 'c.name as supplier_name', 'c.supplier_business_name')
            ->orderBy('transactions.transaction_date', 'asc')
            ->get();

        $order_data = [];

        foreach ($purchase_orders as $order) {
            // Get requisition numbers
            $requisition_nos = [];
            if (!empty($order->purchase_requisition_ids) && is_array($order->purchase_requisition_ids)) {
                $requisitions = \App\Transaction::whereIn('id', $order->purchase_requisition_ids)
                    ->where('type', 'purchase_requisition')
                    ->pluck('ref_no')
                    ->toArray();
                $requisition_nos = $requisitions;
            }
            $requisition_no = !empty($requisition_nos) ? implode(', ', $requisition_nos) : '';

            // Get supplier name
            $supplier_name = $order->supplier_name ?? '';
            if (!empty($order->supplier_business_name)) {
                $supplier_name .= ' (' . $order->supplier_business_name . ')';
            }

            // Get purchase lines
            $purchase_lines = PurchaseLine::join('transactions as t', 'purchase_lines.transaction_id', '=', 't.id')
                ->leftJoin('variations as v', 'purchase_lines.variation_id', '=', 'v.id')
                ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
                ->leftJoin('products as p', 'pv.product_id', '=', 'p.id')
                ->leftJoin('categories as cat', 'p.category_id', '=', 'cat.id')
                ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
                ->where('t.id', $order->id)
                ->select(
                    'purchase_lines.id',
                    'purchase_lines.quantity',
                    'purchase_lines.transaction_id',
                    'p.id as product_id',
                    DB::raw("CASE 
                        WHEN p.type = 'variable' THEN CONCAT(COALESCE(p.name, ''), ' - ', COALESCE(pv.name, ''), ' - ', COALESCE(v.name, ''))
                        ELSE COALESCE(p.name, 'N/A')
                    END as product_description"),
                    'cat.id as category_id',
                    'cat.name as product_group',
                    'u.short_name as unit_name',
                    't.ref_no as order_no',
                    't.transaction_date',
                    't.invoice_no'
                );

            // Apply product filter
            if (!empty($product_id)) {
                $purchase_lines->where('p.id', $product_id);
            }

            // Apply category filter
            if (!empty($category_id)) {
                $purchase_lines->where('cat.id', $category_id);
            }

            $lines = $purchase_lines->get();

            foreach ($lines as $line) {
                $order_data[] = [
                    'transaction_date' => $order->transaction_date,
                    'order_no' => $order->ref_no,
                    'supplier_name' => $supplier_name,
                    'requisition_no' => $requisition_no,
                    'product_group' => $line->product_group ?? 'UNCATEGORIZED',
                    'product_description' => $line->product_description,
                    'quantity' => $line->quantity,
                    'unit' => $line->unit_name ?? ''
                ];
            }
        }

        // Group data by supplier
        $grouped_data = [];
        $grand_total_qty = 0;

        foreach ($order_data as $item) {
            $supplier_key = $item['supplier_name'];
            
            if (!isset($grouped_data[$supplier_key])) {
                $grouped_data[$supplier_key] = [
                    'supplier_name' => $item['supplier_name'],
                    'items' => [],
                    'supplier_total_qty' => 0
                ];
            }
            
            $grouped_data[$supplier_key]['items'][] = $item;
            $grouped_data[$supplier_key]['supplier_total_qty'] += $item['quantity'];
            $grand_total_qty += $item['quantity'];
        }

        // Sort by supplier name
        ksort($grouped_data);

        $util = new Util();
        $currency = session('currency');

        return view('businessmanagement::purchase_register.print_supplier_wise_purchase_order', compact(
            'grouped_data',
            'business',
            'logo',
            'start_date',
            'end_date',
            'util',
            'currency',
            'grand_total_qty'
        ));
    }
}

